package Zim::FS::Subversion;

use strict;
use Carp;
use Zim::Utils qw/buffer/; # be careful not to import file() and dir()

our $VERSION = '0.26';
our @ISA = qw/Zim::FS::Dir/;

=head1 NAME

Zim::FS::Subversion - Subversion version control

=head1 DESCRIPTION

This module handles a dir with Subversion version control for Zim.

It inherits from L<Zim::FS|Zim::FS::Dir>.

=head1 METHODS

=head2 Version Control Interface

=over 4

=item C<new(@PATH)>

Simple constructor for an existing Bazaar dir.

=item C<init(@PATH)>

Constructor that will initialize a new Bazaar repository.

=cut

sub new {
	my $class = shift;
	my $self = $class->SUPER::new(@_);
	if ($$self{dir} and $$self{dir}{svn_root}) {
		# inherit svn_root from parent
		$$self{svn_root} = $$self{dir}{svn_root};
	}
	else {
		# ask svn for root to be sure
		$$self{svn_root} = "$self"; # tmp value for svn()
		#my $bzr_root = $self->bzr('root');
		#chomp $bzr_root;
		#die "Not a Bazaar directory: $self\n" unless length $bzr_root;
		#$$self{bzr_root} = Zim::FS->abs_path($bzr_root);
	}
	return $self;
}

sub init {
	die 'TODO';
}

=item C<commit(COMMENT)>

Commit a new version if there are any changes.

=cut

sub commit {
	die 'TODO';
}

=item C<revert(PATH, VERSION)>

Revert a file to a certain version. Discards all local edits.
If version is undefined, assume latest saved version.

=cut

sub revert {
	die 'TODO';
}

=item C<status()>

Returns the `svn status` output.

=cut

sub status {
	die 'TODO';
}

=item C<list_versions(PATH)>

Returns a list with version info. PATH is an optional argument, 
without this argument all versions are listed.

=cut

sub list_versions {
	die 'TODO';
}

=item C<cat_version(PATH, VERSION1)>

Returns the content of a file a a specific version.

=cut

sub cat_version {
	die 'TODO';
}

=item C<diff(PATH, VERSION1, VERSION2)>

Returns the diff between VERSION1 and VERSION2 for PATH or undef.
If either version is undefined a diff versus the current 
(possibly un-commited) version is returned. If PATH is undefined 
a diff over the whole tree is returned.

=cut

sub diff {
	die 'TODO';
}

=item C<annotate(PATH, VERSION)>

Returns the annotated source for PATH or undef. VERSION is an optional 
argument; if VERSION is not defined the current (possibly un-commited) 
version is assumed.

=cut

sub annotate {
	my ($self, $path, $v) = @_;
	$path .= '@'.$v if defined $v;
	$self->svn('annotate', $path);
}

=back

=head2 Directory Interface

Overloaded methods:

=over 4

=cut


=back

=head2 Private Interface

These methods are used to call the svn command.

=over 4

=item C<svn(@ARG)>

Run the C<svn> command in the appropriate directory and return any output
to STDOUT or STDERR. Will die on error with output in C<$@>.

=cut

sub svn {
	my $self = shift;
	my @arg = grep defined($_), @_;
	@arg = map {s/"/\\"/g; /\W/ ? "\"$_\"" : $_} @arg;
		# proper quoting of arguments, quotemeta is not win32 compat
	warn "# svn @arg\n";

	chdir $$self{svn_root} or die "Could not chdir to: $$self{svn_root}\n";
	if (open SVN, "svn @arg 2>&1 |") {
		binmode SVN, ':utf8';
		my $text = join '', <SVN>;
		my $r = close SVN;
		chdir $ENV{PWD};
		if (!$r) {
			# NOK - error code on exit 
			my $error = join "\n", grep /^svn:/, split "\n", $text;
			die "Failed: svn @arg\n\n$error\n";
		};
		return $text;
	}
	else {
		chdir $ENV{PWD};
		die "Failed: svn @arg\n";
	}
}

1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2008 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

=cut

