package Zim::GUI::NotebookDialog;

use strict;
use File::BaseDir 0.03 qw/config_home config_files data_files/;
use Zim;
use Zim::Utils;
use Zim::GUI::Component;
use Zim::GUI::PropertiesDialog;

our $VERSION = '0.24';
our @ISA = qw/Zim::GUI::Component/;

=head1 NAME

Zim::GUI::NotebookDialog - Chooser dialog with notebooks

=head1 DESCRIPTION

FIXME descriptve text

=head1 METHODS

=over 4

=item C<show()>

Construct a new dialog and show it.

=item C<run()>

Construct a new dialog and run it modal. Returns notebook or undef.
Use C<show()> instead of C<run()> when gtk main loop is already running.

=cut

sub show {
	my $self = shift;
	my ($dialog, $data) = $self->_new;
	$$data{mode} = 'non-modal';
	$$data{self} = $self;
	$dialog->signal_connect(response => \&on_response, $data);
}

sub run {
	my $self = shift;
	my ($dialog, $data) = $self->_new;
	$SIG{USR1} ||= sub { $dialog->present };
	$$self{app}{window} ||= $dialog;
	$$data{mode} = 'modal';
	$$data{self} = $self;
	my $notebook;
	while (my $re = $dialog->run) {
		last unless $re eq 'ok' or $re eq 'help';
		$notebook = on_response($dialog, $re, $data);
		last if defined $notebook;
	}
	$dialog->destroy;
	return $notebook;
}

sub _new {
	my $self = shift;
	my $data = {};

	my $dialog = Gtk2::Dialog->new(
		__("Open notebook"), undef, #. dialog title
	       	[qw/destroy-with-parent no-separator/],
		'gtk-help'   => 'help',
		'gtk-cancel' => 'cancel',
		'gtk-open'   => 'ok',
	);
	#$dialog->set_type_hint('utility');
	$dialog->set_default_size(500,350);
	$dialog->set_border_width(10);
	$dialog->set_icon(
		Gtk2::Gdk::Pixbuf->new_from_file($Zim::ICON) );
			# defined in Component.pm
			# do not use parent because we could run stand-alone
	$dialog->set_default_response('ok');

	$dialog->vbox->set_spacing(20);
	my $path = data_files('zim', 'globe_banner_small.png');
	my $image = Gtk2::Image->new_from_file($path);
	my $align = Gtk2::Alignment->new(0,0.5,0,0);
	$align->add($image);
	$dialog->vbox->pack_start($align, 0,0,0);
	
	my $hbox = Gtk2::HBox->new(0, 12);
	$dialog->vbox->add($hbox);
	
	my $list = Gtk2::SimpleList->new(
		__('Notebooks') => 'text', #. list header
	);
	$list->set_rules_hint(1);
	$list->set_reorderable(1);
	$list->get_selection->set_mode('browse');
	$list->get_column(0)->set_sort_column_id(0);
	$list->signal_connect(
		row_activated => sub { $dialog->response('ok') });
	$list->get_model->signal_connect(
		row_inserted => sub { $$data{modified} = 1 });
		# This signal is activated after re-ordering rows by DnD
	$$data{list} = $list;
	
	my $scrollw = Gtk2::ScrolledWindow->new;
	$scrollw->set_policy('automatic', 'automatic');
	$scrollw->set_shadow_type('in');
	$scrollw->add($list);
	$hbox->add($scrollw);
	
	# load list
	my %notebooks;
	for (Zim->get_notebook_list) {
		$notebooks{$$_[0]} = $$_[1];
		push @{$list->{data}}, $$_[0] unless $$_[0] =~ /^_/;
			# filter out _default_
	}
	$$data{notebooks} = \%notebooks;
	$$data{modified} = 0; # set after loading initial list
	
	my $vbox1 = Gtk2::VBox->new(0,0);
	$hbox->pack_start($vbox1, 0,0,0);

	my $vbox = Gtk2::VButtonBox->new();
	$vbox->set_layout('start');
	$vbox1->pack_start($vbox, 0,0,0);
	
	my @buttons = map Gtk2::Button->new_from_stock($_),
		qw/gtk-add gtk-remove/;
	splice @buttons, 1, 0, $self->new_button(
		'gtk-properties', __('Cha_nge')); #. button
	$vbox->add($_) for @buttons;
	
	my $modify = sub {
		my $new = pop @_;
		my ($i, $name, $dir);
		if ($new) {
			$i    = scalar @{$list->{data}};
		}
		else {
			($i)    = $list->get_selected_indices;
			($name) = @{$list->{data}[$i]};
			$dir    = $notebooks{$name};
		}
		($name, $dir) = $self->prompt_notebook($new, $dir);
		return unless defined $name;
		$notebooks{$name} = $dir;
		splice @{$list->{data}}, $i, 1, $name;
		$$data{modified} = 1;
		$list->select($i);
	};
	$buttons[0]->signal_connect(clicked => $modify, 1); # new
	$buttons[1]->signal_connect(clicked => $modify, 0); # edit
	$buttons[2]->signal_connect(clicked => sub {
			my ($i) = $list->get_selected_indices;
			splice @{$list->{data}}, $i, 1;
			$$data{modified} = 1;
		} );
	
	my $default_button = $self->new_button(
		'gtk-home', __('Set _Default') ); #. button
	$vbox1->pack_end($default_button, 0,0,0);
	$default_button->signal_connect(clicked => sub {
			my $dir = $notebooks{_default_};
			$dir = $self->prompt_default($dir) || '';
			$notebooks{_default_} = $dir;
			$$data{modified} = 1;
		} );

	$dialog->show_all;
	return $dialog, $data;
}

sub on_response {
	my ($dialog, $response, $data) = @_;
	my $self = $$data{self};
	my $list = $$data{list};
	my %notebooks = %{$$data{notebooks}};
		
	# Save list
	if ($$data{modified}) {
		my @notebooks =
			( ['_default_' => $notebooks{_default_}] );
		push @notebooks,
			map [$_ => $notebooks{$_}],
			map $$_[0], @{$list->{data}};
		Zim->set_notebook_list(@notebooks);
	}

	if ($response eq 'help') {
		return $self->ShowHelp(':usage:notebooks');
	}
	elsif ($response eq 'ok') {
		# Open selected notebook
		my ($i) = (@{$list->{data}} == 1)
			? (0) : ($list->get_selected_indices);
		return unless defined $i;

		my ($name) = @{$list->{data}[$i]};
		my $path = Zim::FS->abs_path($notebooks{$name}, $ENV{HOME});
		unless (-d $path) {
			$self->error_dialog(
			__("No such directory: {name}", name => $path) #. error
			);
			return undef;
		}

		# Open new notebook if zim instance already running
		if ($$data{mode} eq 'non-modal') {
			$$self{app}->OpenNotebook($path);
			$dialog->destroy;
		}
		else { return $path }
	}
	else { $dialog->destroy if $$data{mode} eq 'non-modal' }
}

=item C<prompt_notebook(NEW, DIR)>

NEW is a boolean, DIR gives the location of the notebook.

=cut

sub prompt_notebook {
	my ($self, $new, $dir, $name) = @_;
	# FIXME: add a "help" button - link to :usage:properties ?

	my ($prompt) = $self->new_prompt(
		( $new ? __('New notebook')     #. dialog title
		       : __('Edit notebook') ), #. dialog title
		[], {}, undef, undef,
		'<i>'.__("Please give at least a directory to store your pages.\nFor a new notebook this should be an empty directory.\nFor example a \"Notes\" directory in your home dir.").'</i>' 
	);
	
	$dir ||= '~/Notes';
	my $file = Zim->get_notebook_config($dir);
	my $config = $file->exists
		? $file->read(undef, 'Notebook')
		: { # defaults - mainly for first usage
			name => 'Notes',
			document_root => '~/Documents',
			home => ':Home',
		} ;
	
	my $properties = Zim::GUI::PropertiesDialog->new(app => $$self{app});
	my ($nb_vbox, $nb_data) =
		$properties->get_notebook_tab($config, $dir, 1, 1);
	$nb_vbox->show_all;
	$prompt->vbox->add($nb_vbox);

	while (my $r = $prompt->run) {
		if ($r eq 'help') {
			die 'TODO';
			next;
		}
		elsif ($r eq 'ok') {
			$name = $$nb_data{name}->get_text;
			$dir = $$nb_data{dir}->get_text;
			$dir = Zim::FS->abs_path($dir, $ENV{HOME});
			$file = Zim->get_notebook_config($dir);
			$properties->save_notebook_tab($nb_data, $config);
			$file->write($config, 'Notebook');
		}
		last;
	}
	$prompt->destroy;

	return undef unless $dir =~ /\S/;
	$dir =~ s/$ENV{HOME}\/*/~\//;
	unless ($name =~ /\S/) {
		$name = $dir;
		$name =~ s/.*[\/\\]//;
	}
	
	return ($name, $dir);
}

=item C<prompt_default()>

Ask for default notebook.

=cut

sub prompt_default {
	my ($self, $p) = @_;
	my $val = $self->run_prompt(
		__('Set default notebook'), #. dialog title
		['root'], {
			root => [__('Notebook'), 'notebook', $p] #. input
		}, undef, undef,
		'<i>'.__("Please select a default notebook here").'</i>' )
		or return undef;
	my ($dir) = @$val;
	return undef unless $dir =~ /\S/;
	return $dir;
}

1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2007 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

=cut

