#!/usr/bin/perl

use Module::Build;

# Define our custom build actions
my $class = Module::Build->subclass(
	class => 'My::Builder',
	code => << 'EOC'

use strict;
use File::Copy ();
use File::Path;

$ENV{PERL5LIB} ||= ''; # get rid of some warnings...

sub check_prereq {
	my $self = shift;

	my $prereq_ok = $self->SUPER::check_prereq(@_);
	
	my $gtk_ok = 0;
	eval 'use Gtk2';
	if ($@) {
		warn " * ERROR: Could not load perl module Gtk2\n";
	}
	elsif (! Gtk2->CHECK_VERSION(2, 4, 0)) {
		my $version = join '.', Gtk2->GET_VERSION_INFO;
		warn
" * ERROR: gtk+ version $version is installed, but we need >=2.4.0\n" .
"   If you think you have the right version of gtk+ installed, try\n" .
"   recompiling the Gtk2 perl package.\n";
	}
	else { 
		#my $version = join '.', Gtk2->GET_VERSION_INFO;
		#warn "# gtk version $version found\n";
		$gtk_ok = 1;
	}

	warn "ERRORS FOUND IN PREREQUISITES.\n"
		if $prereq_ok && ! $gtk_ok;
	
	return $prereq_ok && $gtk_ok;
}

sub process_share_files {
	my $self = shift;
	my $blib = $self->blib;

	# Copy manula contents
	for (@{$self->rscan_dir('share')}) {
		next if /(^|[\/\\])\.|Development|notebook\.zim/ or -d $_;
		$self->copy_if_modified(
			from => $_,
			to   => File::Spec->catfile($blib, $_) );
	}
	
	# Put in some special files
	my $dir = File::Spec->catdir($blib, qw/share zim doc/);
	$self->copy_if_modified(
		from => 'Changes',
		to   => File::Spec->catfile($dir, 'changelog.txt'));
	my $conf = File::Spec->catfile($dir, 'notebook.zim');
	unless (-e $conf) {
		print "Writing $conf\n";
		open CONF, '>', $conf or die $!;
		print CONF
			"[Notebook]\n",
			"home=:Help\n",
			"name=Zim user manual\n",
			"profile=reader\n",
			"read_only=1\n",
			"slow_fs=0\n",
			"\n",
			"# vim: syntax=desktop\n";
		close CONF;
	}
}

sub ACTION_install {
	my $self = shift;
	$self->SUPER::ACTION_install(@_);
	$self->depends_on('postinstall');
}


sub ACTION_test {
	my $self = shift;
	$self->depends_on('test_data');
	$self->SUPER::ACTION_test(@_);
}

sub ACTION_test_data {
	my $self = shift;

	for (qw#t/html t/notebook t/config t/share t/cache#) {
		next unless -e $_;
		$self->delete_filetree($_);
	}

	print "Extracting test notebook\n";

	open IN, 't/notebook.txt' or die "t/notebook.txt: $!";
	binmode IN, ':utf8' if not $] < 5.008;

	my ($file, $fh);
	while (<IN>) {
		if (/^%%\s+(\S+)\s+%%$/) {
			close $fh if $fh;
			$file = $1;
			mkpath $1 if $file =~ m#(.*/)#;
			open $fh, ">$file" or die "$file: $!\n";
			binmode $fh, ':utf8' if not $] < 5.008;
		}
		elsif ($fh) {
			print $fh $_;
		}
		else { die "oops" }
	}
	close $fh if $fh;
}

sub ACTION_distmeta {
	my $self = shift;
	$self->depends_on('messages');
	$self->run_perl_script('Lingua.PL') or return;
	$self->run_perl_script('Config.PL') or return;
	$self->SUPER::ACTION_distmeta(@_);
}

sub ACTION_disttest {
	my $self = shift;
	$self->SUPER::ACTION_disttest(@_);

	# Check how clean we build
	my $start_dir = $self->cwd;
	my $dist_dir = $self->dist_dir;
	chdir $dist_dir or die "Cannot chdir to $dist_dir: $!";
	$self->run_perl_script('Build', [], ['realclean']);
	chdir $start_dir;

	my %manifest;
	open MANIFEST, 'MANIFEST' or die "Cannot open MANIFEST: $!";
	while (<MANIFEST>) {
		chomp;
		s/\s.*//;
		$manifest{$_}++;
	}
	close MANIFEST;
	my @dirt;
	my @files = grep {-f $_} @{$self->rscan_dir($dist_dir)};
	for (@files) {
		s/^$dist_dir[\/\\]//;
		push @dirt, $_ unless $manifest{$_};
	}
	die "ERROR: The following files where not cleaned up properly:\n",
		map "\t$_\n", @dirt
		if @dirt;
	print "Cleanup OK\n";
}

sub ACTION_realclean {
	my $self = shift;
	$self->SUPER::ACTION_realclean(@_);
	my @files = grep /(\~|\.bak)$/, @{$self->rscan_dir('.')};
	return unless @files;
	warn "Also cleaning up:\n", map "\t$_\n", @files;
	unlink $_ or die $! for @files;
}

=head1 ACTIONS

=over 4

=item C<postinstall>

Runs commands to update the XDG databases.

=cut

sub ACTION_postinstall {
	my $self = shift;
	my $sharedir = $self->install_destination('share');
	my $mimedir = File::Spec->catdir($sharedir, 'mime');
	my $icon = File::Spec->catfile($sharedir, 'pixmaps', 'zim.png');
	for (
		['update-desktop-database'],
		['update-mime-database', $mimedir],
	#	[qw{xdg-icon-resource install --context apps --size 64},
	#		$icon],
		[qw{xdg-icon-resource install --context mimetypes --size 64},
			$icon, q{text-x-zim-notebook}],
	) {
		print "Trying: @$_\n";
		print system(@$_) == 0 ? "Ok\n\n" : "Failed\n\n" ;
	}

}

=item C<messages>

Generates a new F<zim.pot> file. Only works from development tree,
not from dist.

=cut

sub ACTION_messages {
	my $self = shift;
	die "This is not a development tree, can not build messages.\n"
		unless -d './dev';
	$self->do_system($self->perl, './dev/xgettext.pl');
	$self->do_system(qw/msguniq -o zim.pot.new zim.pot/);
	File::Copy::move('zim.pot.new', 'zim.pot');
}

=item C<static>

Builds a self-contained excutable for zim. Needs the L<PAR> and
L<ExtUtils::PkgConfig> modules.

=back

=cut

sub ACTION_static {
	my $self = shift;
	warn "## WARNING: Static building is experimental - feedback is welcome\n";
	warn "## You need to have zim installed before running this command\n";
	warn "## You can pass extra options to 'pp' using PP_OPTS\n";
	eval 'use ExtUtils::PkgConfig';
	die $@ if $@;
	sleep 5;
	$self->depends_on('build');
	my @libs = map {s/^-l//i; ('-l', $_)} grep {length $_} split /\s+/,
		ExtUtils::PkgConfig->libs('gtk+-2.0');
	my @add = ('-a', 'share');
	open MANIFEST, '<MANIFEST' or die "Could not read MANIFEST";
	my @modules;
	for (<MANIFEST>) { # grep modules from the manifest
		/^lib\/(\S+?)\.pm/ or next;
		my $mod = $1;
		$mod =~ s/\//::/g;
		push @modules, '-M', $mod;
	}
	close MANIFEST;
	my $script = File::Spec->catfile(qw/script zim/);
	my $blib = File::Spec->catdir($ENV{PWD}, $self->blib);
	my $libdir = File::Spec->catdir($blib, 'lib');
	print ">> cd $blib\n";
	chdir $blib or die "Could not open dir $blib\n";
	$ENV{PWD} = $blib;
	print ">> pp -I $libdir -o zim @libs @add @modules $script\n";
	system qw{pp -o ../zim}, @libs, @add, @modules, $script;
	warn "\n## Your executable is ready\n";
}

#=item C<testflow>
#
#Run test suite and record which method calls which other methods.
#
#=cut

sub ACTION_testflow {
	# TODO: Move "tmon.out" after each test, now it gets overwritten
	# TODO: wrap script around "dprofpp -T" to make summaries or graphs
	my $self = shift;

#	$self->add_to_cleanup('flow_db');
	$self->depends_on('code');

	local $Test::Harness::switches    =
	local $Test::Harness::switches    =
	local $ENV{HARNESS_PERL_SWITCHES} = '-d:DProf';

	$self->depends_on('test');
#	$self->do_system('./dev/flow.pl');
}

EOC
);

# Define the build parameters
my $build = $class->new(
	dist_name => 'Zim',
	dist_version_from => 'lib/Zim.pm',
	dist_abstract => 'A desktop wiki',
	dist_author => 'Jaap Karssenberg <pardus@cpan.org>',
	license => 'perl',
	script_files => ['bin/zim'],
	requires => {
		'perl'           => '5.8.0',
		'Gtk2'           => '1.040',
		# Older doesn't support Gtk2::UIManager I think
		'POSIX'          => 0,
		'IO::File'       => 0,
		'File::Spec'     => 0,
		'File::Copy'     => 0,
		'File::BaseDir'  => 0.03,
		'File::MimeInfo' => '0.12',
		'File::DesktopEntry' => 0.03,
		'Encode'         => 0,
	},
	recommends => {
		'Gtk2::TrayIcon' => 0,
		'Gtk2::Spell'    => 0,
	},
	build_requires => {
		'Module::Build'  => '0.28',
		'Test::More'     => 0,
		'File::Path'     => 0,
	},
	add_to_cleanup => [
		'./cover_db/',	# from Build testcover
		'./tmon.out',	# from dprof
		'./tmp',
		# Used by tests:
		't/notebook/',
		't/html/',
		't/config/',
		't/share/',
		't/cache/',
	],
	create_makefile_pl => 'passthrough',
	dynamic_config => 1,
);

# Add share directory
$build->add_build_element('share');
$build->install_base_relpaths(share => 'share');

# Figure out where to install the share directory and put it in each
# install set. Just setting 'install_path' breaks --install_base.
my $sets = $$build{properties}{install_sets};
for my $dirs (values %$sets) {
	my @dirs = File::Spec->splitdir($$dirs{bin});
	pop @dirs; # loose 'bin'
	$$dirs{share} = File::Spec->catdir(@dirs, 'share');
}

# And finally create the script "./Build"
$build->create_build_script;

