package Zim::GUI::PreferencesDialog;

use strict;
use File::BaseDir qw/xdg_data_files xdg_data_dirs xdg_data_home/;
use Gtk2;

our $VERSION = '0.20';

=head1 NAME

Zim::GUI::PreferencesDialog - dialog for zim

=head1 DESCRIPTION

This module contains the preferences dialog for zim.
It is autoloaded when needed.

=head1 METHODS

=over 4

=item C<new()>

Simple constructor

=cut

sub new {
	my ($class, %self) = @_;
	bless \%self, $class;
}

=item C<show()>

Present the dialog.

=cut

my %Labels = ( # setting => [label, tooltip]
	browser => [
		'Web browser',
		"Application to open urls",
	],
	file_browser => [
		'File browser',
		"Application to open directories",
	],
	email_client => [
		'Email client',
		"Application to compose email",
	],
	user_name => [
		'User name',
		"Your name; this can be used in export templates",
	],
	follow_new_link => [
		'Follow new link',
		"Creating a new link directly opens the page",
	],
	ro_cursor => [
		'Show cursor for read-only',
		"Show the cursor even when you can not edit the page. This is useful for keyboard based browsing.",
	],
	tearoff_menus => [
		'Tearoff menus',
		"Add 'tearoff' strips to the menus",
	],
	use_camelcase => [
		'Auto-link CamelCase',
		"Automaticly link CamelCase words when you type",
	],
	use_linkfiles => [
		'Auto-link files',
		"Automatically link file names when you type",
	],
	use_utf8_ent => [
		'Auto-format entities',
		"Use autoformatting to type special characters",
	],
	backsp_unindent => [
		'Use "Backspace" to un-indent',
		"Use the \"Backspace\" key to un-indent bullet lists (Same as \"Shift-Tab\")",
	],
	use_autoselect => [
       		'Auto-select words',
		"Automaticly select the current word when you toggle the format",
	],
	follow_on_enter => [
		'Use "Enter" to follow links',
		"Use the \"Enter\" key to follow links. If disabled you still can use \"Alt-Enter\"",
	],
	use_ctrl_space => [
		'Use "Ctrl-Space" to switch focus',
		"Use the \"Ctrl-Space\" key combo to switch focus between text and side pane. If disabled you can still use \"Alt-Space\".",
	],
	expand_tree => [
		'Expand side pane',
		"Start the side pane with the whole tree expanded."
	],
);

sub show {
	my $self = shift;
	my $app = $self->{app};
	my $dialog = Gtk2::Dialog->new(
		"Preferences - Zim", $app->{window},
	       	[qw/destroy-with-parent no-separator/],
		'gtk-help'   => 'help',
		'gtk-cancel' => 'cancel',
		'gtk-ok'     => 'ok',
	);
	$dialog->set_resizable(0);
	
	## Set up content
	my %entries;
	my %checkboxes;
	my $settings = $self->{app}{settings};
	my $tooltips = Gtk2::Tooltips->new;

	my $_prompt = sub { # new left aligned label
		my $key = shift;
		my $align = Gtk2::Alignment->new(0,0.5, 0,0);
		my $label = Gtk2::Label->new($Labels{$key}[0].':');
		$align->add($label);
		return $align;
	};
	my $_entry = sub { # new entry
		my $key = shift;
		my $entry = Gtk2::Entry->new();
		$entries{$key} = $entry;
		$entry->set_text($$settings{$key});
		$tooltips->set_tip($entry, $Labels{$key}[1])
			if defined $Labels{$key}[1];
		return $entry;
	};
	my $_checkbox = sub { # new checkbox
		my $key = shift;
		my $box = Gtk2::CheckButton->new_with_label($Labels{$key}[0]);
		$checkboxes{$key} = $box;
		$box->set_active($$settings{$key});
		$tooltips->set_tip($box, $Labels{$key}[1])
			if defined $Labels{$key}[1];
		return $box;
	};

	my $tabs = Gtk2::Notebook->new;
	$dialog->vbox->add($tabs);
	
	## General tab
	my $vbox = Gtk2::VBox->new(0,5);
	$vbox->set_border_width(5);
	$tabs->append_page($vbox, 'General');
	
	my @settings = qw/
		user_name
		browser
		file_browser
		email_client
	/;
	my $table = Gtk2::Table->new(scalar(@settings), 2);
	$table->set_row_spacings(5);
	$table->set_col_spacings(12);
	$vbox->pack_start($table,0,1,0);
	my $i = 0;
	for (@settings) {
		$table->attach_defaults($_prompt->($_), 0,1, $i,$i+1);
		$table->attach_defaults($_entry->($_), 1,2, $i,$i+1);
		$i++;
	}

	## Interface tab
	$vbox = Gtk2::VBox->new(0,5);
	$vbox->set_border_width(5);
	$tabs->append_page($vbox, 'Interface');
	
	@settings = qw/
		follow_on_enter
		follow_new_link
		use_ctrl_space
		ro_cursor
		tearoff_menus
		expand_tree
	/;
	for (@settings) {
		$vbox->pack_start($_checkbox->($_), 0,1,0);
	}

	$table = Gtk2::Table->new(3, 2);
	$table->set_row_spacings(5);
	$table->set_col_spacings(12);
	$vbox->pack_start($table,0,1,0);
	$i = 0;

	# Font button
	my $use_font = Gtk2::CheckButton->new_with_label('Use custom font');
	$table->attach_defaults($use_font, 0,2, $i,$i+1);
	$i++;
	$use_font->set_active($$settings{textfont});
	my $font = $$settings{textfont} ||
		$app->PageView->get_style->font_desc->to_string;
	#warn "font: $font\n";
	my $font_button = Gtk2::FontButton->new_with_font($font);
	$font_button->signal_connect(font_set => sub {
		my $string = $font_button->get_font_name;
		$$settings{textfont} = $string;
		$app->PageView->set_font($string);
	} );
	$table->attach_defaults($font_button, 1,2, $i,$i+1);
	$i++;
	$font_button->set_sensitive($use_font->get_active);
	$use_font->signal_connect(toggled => sub {
		my $active = $use_font->get_active;
		$font_button->set_sensitive($active);
		unless ($active) {
			$$settings{textfont} = undef;
			$app->PageView->set_font(undef);
		}
	} );

	## Editing tab
	$vbox = Gtk2::VBox->new(0,5);
	$vbox->set_border_width(5);
	$tabs->append_page($vbox, 'Editing');
	@settings = qw/
		use_camelcase
		use_linkfiles
		use_utf8_ent
		use_autoselect
		backsp_unindent
	/;
	$table = Gtk2::Table->new(scalar(@settings), 1);
	$table->set_row_spacings(5);
	$table->set_col_spacings(12);
	$vbox->pack_start($table, 0,1,0);
	$i = 0;
	for (@settings) {
		$table->attach_defaults($_checkbox->($_), 0,1, $i,$i+1);
		$i++;
	}

	## Plugins tab
	$vbox = Gtk2::VBox->new(0,5);
	$vbox->set_border_width(5);
	$tabs->append_page($vbox, 'Plugins');

	my $label = Gtk2::Label->new();
	$label->set_markup(
		"<i>You need to restart the application\n".
		"for plugin changes to take effect.</i>");
	$vbox->pack_start($label, 0,1,0);
	my @plugins;
	my @plugged = split /,/, $$settings{plugins};
	for (xdg_data_home(), xdg_data_dirs()) {
		my $dir = File::Spec->catdir($_, 'zim', 'plugins');
		next unless -d $dir;
		for my $f (Zim::File->list_dir($dir)) {
			next unless $f =~ s/\.pl$//;
			next if grep {$$_[0] eq $f} @plugins;
			my $bit = grep {$_ eq $f} @plugged;
			push @plugins, [$f, $bit];
		}
	}
	@plugins = sort {$$a[0] cmp $$b[0]} @plugins;
	
	for (@plugins) {
		my $box = Gtk2::CheckButton->new_with_label($$_[0]);
		$box->set_active($$_[1]);
		$vbox->pack_start($box, 0,1,0);
		push @$_, $box;
	}
	
	## Show it all
	$dialog->show_all;
	while ($_ = $dialog->run) {
		if ($_ eq 'help') {
			$app->ShowHelp(':zim:usage:preferences');
			next;
		}
		last unless $_ eq 'ok';
		
		# set setting
		for (keys %entries) {
			$$settings{$_} = $entries{$_}->get_text;
		}
		for (keys %checkboxes) {
			$$settings{$_} = $checkboxes{$_}->get_active ? 1 : 0;
			#warn "$_ = $$settings{$_}\n";
		}

		$app->{ui}->set_add_tearoffs($$settings{tearoff_menus});
			# FIXME signal for this ?
		$app->PageView->on_show_cursor();
			# FIXME signal for this ?
		
		## FIXME signal for this ##
		$app->PageView->{htext}{follow_on_enter} =
			$$settings{follow_on_enter};
		$app->{repository}{user_name} = $$settings{user_name};
		## ##
		
		# set plugins
		@plugged = map $$_[0], grep $$_[2]->get_active, @plugins;
		$$settings{plugins} = join ',', @plugged;

		last;
	}
	$dialog->destroy;
	$app->save_config;
}

1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2006 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

=cut

