package Zim::Page;

use strict;
use overload
	'""' => sub { $_[0]->{name} },
	fallback => 'TRUE' ;
use Carp;
use Zim::Formats;

our $VERSION = '0.18';

our %_Formats;

=head1 NAME

Zim::Page - Page object for Zim

=head1 DESCRIPTION

This class defines a page object. This is a data container used by
L<Zim::Repository> to represent for example a file.

=head1 OVERLOAD

This class overloads the '""' operator, so the string version of an object
is the page name.

=head1 METHODS

=over 4

=item C<new(PARENT, NAME)>

Simple constructor. PARENT should be a repository object of class Zim.
NAME is the page name for this object.

=cut

sub new {
	my ($class, $parent, $name) = @_;
	$name =~ s/:+$//;
	croak "BUG: Can't create a $class object without a page name"
		unless length $name;
	$name =~ s/^:?/:/;
	my $self = bless {
		name => $name,
		repository => $parent,
		status => '',
		properties => {read_only => 1},
	}, $class ;
	return $self;
}

=item C<properties()>

Returns a hash with properties. See L</PROPERTIES>.

=cut

sub properties { $_[0]->{properties} }

=item C<name()>

Get or set the full name of the page.

=cut

sub name {
	$_[0]->{name} = $_[1] if @_ > 1;
	return $_[0]->{name};
}

=item C<basename()>

Returns the last part of the page name.

=cut

sub basename {
	my $name = $_[0]->{name};
	$name =~ s/^(.*:+)//;
	return $name;
}

=item C<split_name()>

=cut

sub split_name {
	my $name = pop;
	$name = $name->{name} if ref $name;
	#print STDERR "namespace for $name ";
	$name =~ s/^(.*:)//;
	my $ns = $1 || ':';
	$ns =~ s/::+/:/g;
	#print STDERR "is $name\n";
	return ($ns, $name);
}

=item C<namespace()>

Returns the namespace to which this page belongs.

=item C<namespaces()>

Like C<namespace()> but returns the namespace path as a list.

=cut

sub namespace { (&split_name)[0] }

sub namespaces {
	my $name = pop;
	$name = $name->{name} if ref $name;
	$name =~ s/^:+|:+$//g;
	my @ns = split /:+/, $name;
	pop @ns;
	return @ns;
}

=item C<status(STRING)>

Set or get a status string for this page.
Typical status strings are 'new' and 'deleted'.

=cut

sub status {
	$_[0]->{status} = $_[1] if @_ > 1;
	return $_[0]->{status};
}

=item C<exists()>

Returns TRUE if the page already exists.

=cut

sub exists { $_[0]->{status} ne 'new' and $_[0]->{status} ne 'deleted' }

=item C<copy(TARGET, UPDATE_LINKS)>

=item C<move(TARGET, UPDATE_LINKS)>

=item C<delete()>

The methods C<copy()>, C<move()> and C<delete()> are aliases for the methods
C<copy_page()>, C<move_page()> and C<delete_page()> in the public repository
interface; see L<Zim::Repository>.

TARGET is the new name for the page. The UPDATE_LINKS argument is a boolean
that tells whether links to this page should be updated to point to this
new name.

=cut

sub copy { $_[0]->{repository}->root->copy_page(@_) }

sub move {
	$_[0]->{_resolve} = {}; # clear links - FIXME update here ?
	$_[0]->{repository}->root->move_page(@_);
}

sub delete { $_[0]->{repository}->delete_page(@_) }

=item C<parse_link(LINK)>

Returns a link type and a link target, see L<Zim::Formats>.

=cut

sub parse_link {
	my $self = shift;
	return $self->{cloning}->parse_link(@_) if $self->{cloning};
	my $fmt = $self->{format} || 'Zim::Formats';
	return $fmt->parse_link(@_, $self);
}

=item C<resolve_page(NAME, NO_DEFAULT)>

Caching wrapper for C<$repository->resolve_page(NAME, PAGE, NO_DEFAULT)>.

=item C<resolve_name(NAME, NO_DEFAULT)>

Caching wrapper for C<$repository->resolve_name(NAME, PAGE, NO_DEFAULT)>.

=cut

sub resolve_page {
	my $self = shift;
	my $name = $self->resolve_name(@_);
	return $name ? $self->{repository}->root->get_page($name) : undef ;
}

sub resolve_name { # 'cloning' hack needed when exporting
	my ($self, $link, $no_def) = @_;
	return $self->{cloning}->resolve_name($link, $no_def)
		if $self->{cloning};

	if (my $r = $self->{_resolve}{$link}) {
		# Caching
		return ( ($no_def && $$r[1]) ? undef : $$r[0] )
	}

	my $root = $self->{repository}->root;
	my $name = $root->resolve_name($link, $self, 1);
	if ($name) {
		$self->{_resolve}{$link} = [$name, 0];
	}
	else { # default
		$name = $self->namespace . $link;
		$self->{_resolve}{$link} = [$name, 1];
	}

	return $name;
}

=item C<match_word(WORD)>

TODO: stable api for this

=cut

sub match_word { # TODO optimize by caching found links
	my ($self, $word) = @_;
	return $self->{repository}->can('_match_word')
		? $self->{repository}->_match_word($self, $word)
		: undef ;
}

=item C<equals(PAGE)>

Check if PAGE is refering to the same page we are.
This does not guarantee that the actual content is the same.

=cut

sub equals {
	my ($self, $page) = @_;
	return $page eq $self->{name} unless ref $page; 
	return 0 if $page->{repository} ne $self->{repository};
	return $self->{name} eq $page->name;
}

=item C<clone(SOURCE)>

Import content from object SOURCE into this object.

=cut

sub clone {
	my ($self, $page, %opt) = @_;
	local $self->{cloning} = $page;
	if ( $self->{format} and $page->{format} ) {
		my $tree = $page->get_parse_tree;
		$self->_check_media($tree, $opt{media}, $page)
			if $opt{media} and $opt{media} ne 'relative';
		$self->set_parse_tree($tree);
	}
	else { warn "Could not clone page: ". $page->name ."\n" }
}

sub _check_media {
	my ($self, $tree, $media, $cloning) = @_;
	#warn "_check_media: @_\n";
	die "BUG: unknown media option: $media"
		unless $media eq 'absolute' or $media eq 'copy';
	
	my $base = $cloning->{properties}{base};
	for my $l (Zim::Formats->extract_refs('link', $tree)) {
		my ($type, $link) = $self->parse_link($$l[1]{to});
		next unless $type eq 'file';
		$link = Zim::File->abs_path($link, $base);
		$$l[1]{to} = ($media eq 'absolute')
			? $link : $self->_copy_media($link, $cloning);
	}

	for my $i (Zim::Formats->extract_refs('image', $tree)) {
		my $link = $$i[1]{src};
		$link = Zim::File->abs_path($link, $base);
		$$i[1]{src} = ($media eq 'absolute')
			? $link : $self->_copy_media($link, $cloning);
	}
}

sub _copy_media {
	my ($self, $file, $cloning) = @_;

	my $copy;
	my $croot = $cloning->{repository}->root->{dir};
	my $root = $self->{repository}->root->{dir};
	my $rel = Zim::File->rel_path($file, $croot);
	if (defined $rel) { # was relative
		$copy = Zim::File->abs_path($rel, $root);
	}
	else { # was absolute
		my $k = $file;
		$k =~ s#[/\\]+#_#g; # replace dir separater
		$k =~ s#^_+##;
		$copy = Zim::File->abs_path($k, $root.'/_media');
	}

	if (-f $file) {
		warn "#  copy $file => $copy\n";
		Zim::File->copy($file => $copy);
	}

	my $link = Zim::File->rel_path(
		$copy, $self->{properties}{base}, 1);
	return $link || $copy ;
}

=item C<get_next()>

Returns next page.

=item C<get_prev()>

Return previous page.

=cut

sub get_next {
	my $self = shift;
	my ($ns, $name) = ($self->namespace, $self->basename);
	my $prev;
	for ($self->{repository}->list_pages($ns)) {
		s/:$//;
		return $self->{repository}->get_page($ns.$_)
			if $prev eq $name;
		$prev = $_;
	}
	return undef;
}

sub get_prev {
	my $self = shift;
	my ($ns, $name) = ($self->namespace, $self->basename);
	my $prev;
	for ($self->{repository}->list_pages($ns)) {
		s/:$//;
		if ($_ eq $name) {
			return $self->{repository}->get_page($ns.$prev)
				if length $prev;
			last;
		}
		$prev = $_;
	}
	return undef;
}
=back

=head2 Source Interface

=over 4

=item C<set_source(SOURCE)>

SOURCE is an object that supports an C<open(MODE)> method, which 
returns a filehandle (or IO object) for the source, and an C<exists()>
method which checks if there is anything to open.

If SOURCE is C<undef> this unsets the source, making the page read-only.

This method sets the 'read_only' property depending on whether SOURCE
is defined or not.

=cut

sub set_source {
	$_[0]->{source} = $_[1];
	$_[0]->{parse_tree} = undef;
	$_[0]->{properties}{read_only} = ! defined $_[1];
}

=item C<open_source(MODE)>

Returns an IO object or undef when there is none.
This method dies when it fails opening a given source.
In general pages that have status "new" will not yet have a source.

MODE is optional and can either be 'r' or 'w' depending on whether you
would like the source to be opened for reading or for writing.

Do not forget to close the IO object when you are done with it !
Use OO syntax to close it, using C<< $fh->close >> instead of
C<close $fh>.

=cut

sub open_source {
	my ($self, $mode) = @_;
	my $src = $self->{source};
	return unless defined $src;
	return if $mode eq 'r' and ! $src->exists;
	$src->make_dir if $mode eq 'w' and $src->can('make_dir');
	my $fh = $src->open($mode);
	$self->{status} = '' if $mode eq 'w' and defined $fh;
		# remove "new" or "deleted" status
	return $fh;
}

=item C<has_source()>

Returns a boolean.

=cut

sub has_source { return defined $_[0]->{source} }

=back

=head2 Formatted Interface

=over 4

=item C<set_format(FORMAT)>

Sets a source format for this page. This can either be an object of the class
L<Zim::Formats> (or similar), or a name in which case this will be looked up
in the C<Zim::Formats::*> namespace.

Formats are only used for pages that also have a source object.

=cut

sub set_format {
	my ($self, $format) = @_;
	$self->{format} = ref($format) ? $format : _load_format($format);
}

sub _load_format {
	my $name = shift;
	$name = lc $name;
	return $_Formats{$name} if defined $_Formats{$name};

	# TODO: hash lookup using Zim::Formats
	my $class = 'Zim::Formats::'.quotemeta(ucfirst $name);
	eval "use $class";
	die if $@;

	$_Formats{$name} = $class;
	return $class;
}

=item C<get_parse_tree()>

Get the parse tree for this page.

When using source this method will return the tree resulting from running
the given source through the given formatter.

=cut

sub get_parse_tree {
	my $self = shift;
	return $self->{parse_tree} if defined $self->{parse_tree};
	return unless defined $self->{source} and defined $self->{format};
	
	my $io = $self->open_source('r');
	if ($io) {
		my $tree = $self->{format}->load_tree($io, $self);
		$io->close;
		%{$tree->[1]} = (%{$self->{properties}}, %{$tree->[1]});
		return $tree;
	}
	else { return ['Page', $self->{properties}] }
	# FIXME hook _template into this "else"
}

=item C<set_parse_tree(TREE)>

Set the parse tree for this page.

When using source this method will use the formatter to save the parse tree
to the IO object.

=cut

sub set_parse_tree { #warn "set_parse_tree from ", join(' ', caller), "\n" ;
	my ($self, $tree) = @_;
	$self->{status} = ''; # remove "new" or "deleted"
	$self->{_links} = [ $self->list_links($tree) ];
	if (defined $self->{source}) {
		my $io = $self->open_source('w')
			|| die "Could not save parse tree, did not get an IO object.\n";
		#flock($io, LOCK_EX); # lock # Move function to IO::File::Zim
		$self->{format}->save_tree($io, $tree, $self);
		#flock($io, LOCK_UN); # unlock
		$io->close;
		
		$self->{repository}->_cache_page($self)
			if $self->{repository}->can('_cache_page');
		# FIXME this hook does not belong here
	}
	else {
		$self->{parse_tree} = $tree;
	}
}

=item C<list_links()>

Returns a list with names of pages that this page links to.

=cut

sub list_links { #warn "list_links from ", join(' ', caller), "\n" ;
	my $self = shift;
	my $node = shift;
	unless ($node) {
		return @{$self->{_links}} if $self->{_links};
		$node = $self->get_parse_tree;
	}

	my %links;
	for (2 .. $#$node) {
		my $n = $$node[$_];
		next unless ref $n;
		if ($$n[0] eq 'link') {
			my ($type, $link) = ('', $$n[1]{to});
			($type, $link) = $self->parse_link($link);
			next unless $type eq 'page';
			$link = $self->resolve_name($link);
			$links{$link} = 1;
		}
		else {
			%links = (%links, map {$_ => 1} list_links($self, $n)); # recurse
		}
	}

	return keys %links;
}

=item C<list_backlinks()>

Returns a list with names of pages that link to this page.

=cut

sub list_backlinks { # 'cloning' hack needed when exporting
	my $obj = $_[0]->{cloning} ? $_[0]->{cloning} : $_[0];
	my $rep = $obj->{repository}->root;
	return $rep->can('list_backlinks')
		? $rep->list_backlinks($obj) : () ;
}

=item C<update_links(FROM => TO, ...)>

Update links to other pages. This is used for example when
a page is moved to update all links to that page.

=item C<update_links_self(OLD)>

Called if the current page was moved from OLD to current
name. Updates links to reflect this move.

=cut

sub update_links {
	my ($self, %links) = @_;
	my $tree = $self->get_parse_tree || return;
	my $selection = Zim::Selection->new(undef, {}, keys %links);

	warn "Updating links in ", $self->name, "\n";
	my $done = 0;
	for my $ref (Zim::Formats->extract_refs('link', $tree)) {
		my $old = $$ref[1]{to};
		my ($t, $l) = $self->parse_link($old);
		#warn "found link, old = $old => $t, $l\n";
		next unless $t eq 'page';
		my $match = $selection->resolve_name($l, $self, 1) || next;
		#warn "\told match = $match\n";
		my $new_match = $self->resolve_name($l, 1);
		#warn "\tnew match = $new_match\n";
		next if $new_match and length($new_match) > length($match);
			# the only difference is the movement of pages
			# so "new_match" was already there before the move
			# ergo, if "new_match" is there, this was not a link
			# to "match" in the first place
		my $new = $self->relative_name($links{$match});
		warn "\tUpdating $match => $new\n";
		$$ref[1]{to} = $new;
		$$ref[2] = $new if @$ref == 3 and $$ref[2] eq $old;
		$done++;
	}
	$self->set_parse_tree($tree) if $done;
	warn "Updated $done links in $self\n";
	return $done;
}

sub update_links_self {
	my ($self, $from) = @_;
	my $tree = $self->get_parse_tree || return;

	warn "Updating links in ", $self->name, " (was $from)\n";
	my $done = 0;
	for my $ref (Zim::Formats->extract_refs('link', $tree)) {
		my $old = $$ref[1]{to};
		my ($t, $l) = $self->parse_link($old);
		#warn "found link, old = $old => $t, $l\n";
		next unless $t eq 'page';
		my $page = $self->{repository}->root->resolve_name($l, $from);
		#warn "resolved $l => $page\n";
		my $new = $self->relative_name($page);
		warn "\tUpdating $old => $new\n";
		$$ref[1]{to} = $new;
		$$ref[2] = $new if @$ref == 3 and $$ref[2] eq $old;
		$done++;
	}
	$self->set_parse_tree($tree) if $done;
	warn "Updated $done links in $self\n";
	return $done;
}

=back

=head2 Other Functions

=over 4

=item C<relative_name(NAME)>

Turns an absolute page name into one that is relative
to this page. Reverses part of the logic of C<resolve_name()>.
Returns either relative or absolute name.

=cut

sub relative_name {
	my ($self, $name) = @_;
	my @name = grep length($_), split /:+/, $name;
	my @self = grep length($_), split /:+/, $self->{name};
	return join ':', @name unless $name[0] eq $self[0];
	my $anchor;
	while (@self and @name and $self[0] eq $name[0]) {
		$anchor = shift @self;
		shift @name;
	}
	return $name[0] if @self and @name == 1; # direct leaf of same path
	return join ':', $anchor, @name; # indirect leaf
}


=item C<relative_path(FILE, PAGE)>

Checks to see if the path for FILE can be made relative
to the base dir for PAGE.
Returns either the relative or the absolute path.

=cut

sub relative_path {
	my ($self, $file) = @_;
	return $file if $file =~ m#^\w[\w\+\-\.]+:/#; # url

	my $root = $self->{repository}->root->{dir};
	my $rel = Zim::File->rel_path($file, $root, 0);
	return $file unless defined $rel; # file not below root

	my $ref = $self->{properties}{base};
	$rel = Zim::File->rel_path($file, $ref, 1);
	return defined($rel) ? $rel : $file ;
}


1;

__END__

=back

=head1 PROPERTIES

The page object contains a hash with properties. These can be any kind of data
from the backend that needs to be shared with the interface. Typically it are
config options that can be specified per-page.

For the "formatted" page interface the properties hash is used for the Document
meta attributes in the parse-tree.

Common properties are:

=over 4

=item base (url)

Base directory for files that belong to this page in some way or another.
This is for example used by the interface to resolve the location of image
files that are included in the page.

This value can be undefined when the repository does not allow direct access
to the source files.

TODO: At the moment this is the directory which contains the page file, this
is open for change in future versions.

Currently only the C<file://> url is really supported.

=item read_only (boolean)

Tells the interface that this page should not be edited.
Defaults to TRUE.

=item special (boolean)

Rare cases a non-existent read_only page is used to display some information
or because there is simply nothing else to display. This property makes
various components ignore these pages. For example they don't show up in
the history.

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

L<Zim>,
L<Zim::Repository>

=cut
