package Zim;

use strict;
use Carp;
use UNIVERSAL qw/isa/;
use File::Spec;
use File::BaseDir qw/xdg_config_files xdg_data_home xdg_data_dirs xdg_data_files/;
use Zim::Selection;
use Zim::File;
use Zim::Page;

our $VERSION = '0.18';
our $LONG_VERSION = << "EOT";
zim $VERSION - A desktop wiki and outliner

Copyright (c) 2006 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

Please report bugs to pardus\@cpan.org
EOT

=head1 NAME

Zim - Application object for the zim desktop wiki

=head1 SYNOPSIS

	use Zim;
	
	my $repo = Zim->new(dir => $dir);
	my $page = $repo->get_page('Home');

=head1 DESCRIPTION

This class defines the public interface to the document repository
as used by L<zim>(1). It acts as a dispatcher that manages one or
more child objects to handle page requests. It also adds some tests
and thus tries to isolate the handlers from errors in the GUI.

The interface which child objects should support is defined in
L<Zim::Repository>.

=head1 METHODS

=head2 Public Methods

The following methods can be used by the GUI.

=over 4

=item new(PARENT, NAMESPACE, FILE)

=item new(PARENT, NAMESPACE, DIR, TYPE, ...)

Constructor. PARENT can be a parent repository object or undef.
NAMESPACE is the prefix for all pages managed by this repository.

TODO: implement this: FILE should be the path to a repository config file.

DIR is the root namespace of the repository and might contain
a onfig file.

TYPE is optional and is used to initialize a top level child
repository when no other config is given. This is used for example
to intialize a repository. All remaining arguments are passed on
to this top level child repository.

=cut

sub new {
	my ($class, %param) = @_;

	$param{namespace} ||= ':';
	$param{namespace} =~ s/:?$/:/;
	die "BUG: repository initialized without directory"
		unless $param{dir};
	$param{dir} = Zim::File->abs_path($param{dir});
	die "Not a directory: $param{dir}\n"
		if -e $param{dir} and ! -d $param{dir};
	
	my $self = bless {%param}, $class;
	$self->{config} ||= {};
	
	my ($conf) = grep $_->exists(),
	             map  Zim::File->new($param{dir}, $_),
		     qw/_zim.config .zim.config/;
	$conf ||= Zim::File->new($param{dir}, '.zim.config');
	$self->{conf_file} = $conf->path;
	for ($conf->read) {
		s/[\n\r]+$//;
		next unless /\S/;
		my ($opt, $val) = split '=', $_, 2;
		if ($opt =~ /^:/) {
			# opt is a namespace with sub-repository
			# allow e.g. ":namespace=Class,key=val,key=val"
			my ($class, $arg) = split ',', $val, 2;
			my %arg = map split('=',$_,2),
			          map split(',',$_,2), $arg; 
			$self->add_child($opt, $class, %arg);
		}
		else { $self->{config}{$opt} = $val }
	}
	
	# default config
	my $type = $self->{type} || 'Files';
	$self->add_child(':', $type, %param)
		unless defined $self->{':'};

	return $self;
}

=item C<config()>

Returns a hash with config options.

=cut

sub config { $_[0]->{config} }

=item C<save_config()>

Saves the config to disk.

=cut

sub save_config {
	my $self = shift;
	my $node = $self->{dir};
	my $conf = Zim::File->new($self->{conf_file});
	warn "Saving $self->{conf_file}\n";
	my $text = join '', map "$_=$$self{config}{$_}\n", keys %{$$self{config}};
	for ($conf->read) { $text .= $_ if /^:/ }
	$conf->write($text);
}

=item C<base()>

Returns the base directory for files.

The "base" directories for individual pages should either be the same as
this directory or a subdirectory of it.

=cut

sub base { return $_[0]->{dir} }

## 3 methods for checking arguments ##

sub _check_page {
	if (ref $_[0]) {
		croak "Object \"$_[0]\" is not a Zim::Page"
	       		unless isa $_[0], 'Zim::Page';
	}
	elsif ($_[1]) { croak "\"$_[0]\" is not a page object" }
	else { goto \&_check_page_name }
}

# Valid page names can contain letters, numbers and ".", "-", "(" and ")"
# The first character of a name can only be a letter or a number
# Absolute names start with a ":"
# Namespaces end in a ":"

sub _check_page_name {
	croak "\"$_[0]\" is not a valid page name"
		unless $_[0] =~ /^(?::+[\w\%]+[\w\.\-\(\)\%]*)+$/;
}

sub _check_namespace {
	croak "\"$_[0]\" is not a valid namespace"
		unless $_[0] =~ /^(?::+|(?::+[\w\%][\w\.\-\(\)\%]*)+:+)$/;
}

=item C<list_pages(NAMESPACE)>

Lists pages in NAMESPACE. Sub-namespaces have a trailing ':'
in the listing.

If a page is not present in this list that does not mean that it
doesn't exists. This list is used for hierarchical views of the
page structure; not all repositories need to support a hierarchical order.

=cut

sub list_pages {
	my ($self, $namespace) = @_;
	$namespace =~ s/:?$/:/;
	_check_namespace($namespace);
	my @pages =
		grep { m/^:/ and s/^\Q$namespace\E:*([^:]+:*)$/$1:/ } keys %$self;
	$namespace =~ $self->{regex}
		or die "BUG: '$namespace' !~ qr/$self->{regex}/";
	return if $self->{lc($1)}->{no_show_in_sidepane}; # temp HACK
	@pages = sort @pages, $self->{lc($1)}->list_pages($namespace);
	for (0 .. $#pages-1) { # remove doubles etc.
		$pages[$_] = undef if $pages[$_+1] eq $pages[$_];
		$pages[$_] = undef if $pages[$_+1] eq $pages[$_].':';
	}
	return sort {lc($a) cmp lc($b)} grep defined($_), @pages;
}

=item get_page(NAME)

Returns an object of a class inheriting from L<Zim::Page>.
When you ask for a page that doesn't exists yet, you should get a new object.
In case it is not possible to create the page in question C<undef> is returned.

=cut

sub get_page {
	my ($self, $name) = @_;
	_check_page_name($name);
	$name =~ $self->{regex} or die "BUG: '$name' !~ qr/$self->{regex}/";
	#warn "'$name' =~ /$self->{regex}/ => \$1 = '$1'\n";
	#warn "Dispatching get_page('$name') to '".lc($1)."'\n";
	return $self->{lc($1)}->get_page($name);
}

=item C<resolve_page(LINK, PAGE, NO_DEFAULT)>

Like C<get_page()> but now expecting NAME to be user input,
so NAME is sanitized first.

=cut

sub resolve_page {
	my $self = shift;
	my $name = $self->resolve_name(@_);
	return $name ? $self->get_page($name) : undef ;
}

=item C<resolve_name(LINK, PAGE, NO_DEFAULT)>

Cleans up NAME but does not actually fetch a page object.

=cut

sub resolve_name {
	my ($self, $name, $ref, $no_def) = @_;
	if ($ref) {
		# TODO check for multiple children in path
		# TODO use child path case for path
		$name = Zim::Repository->clean_name($name, 1) or return;
		$ref =~ $self->{regex}
			or die "BUG: '$ref' !~ qr/$self->{regex}/";
		return $self->{lc($1)}->resolve_name($name, $ref, $no_def);
	}
	else {
		$name = Zim::Repository->clean_name($name, 0) or return;
		$name =~ $self->{regex}
			or die "BUG: '$name' !~ qr/$self->{regex}/";
		my $child = $1;
		$name =~ s/^$child/$child/i; # set correct caps
		return $self->{lc($child)}->resolve_name($name, undef, $no_def);
	}
}

=item C<resolve_namespace(NAME)>

Returns a namespace string. Used to sanitize user input.

=cut

sub resolve_namespace {
	my ($self, $name) = @_;
	$name = Zim::Repository->clean_name($name);
	$name .= ':'; 	# is a namespace
	$name =~ $self->{regex} or die "BUG: '$name' !~ qr/$self->{regex}/";
	my $child = $1;
	$name =~ s/^$child/$child/i; # set correct caps
	return $name;
}

=item copy_page(SOURCE, TARGET, UPDATE_LINKS)

Copy page SOURCE to TARGET. Both arguments can be either page names
or objects. Returns the (new) TARGET object. The UPDATE_LINKS argument
is a boolean that tells whether links to this page should be updated
to point to this new name.

=item move_page(SOURCE, TARGET, UPDATE_LINKS)

Move page SOURCE to TARGET. Both arguments can be either page names
or objects. Returns the (new) TARGET object.The UPDATE_LINKS argument
is a boolean that tells whether links to this page should be updated
to point to this new name.

=cut

sub copy_page { _clone('copy_page', @_) }

sub move_page { _clone('move_page', @_) }

sub _clone { # Copy and move are almost the same
	my ($method, $self, $source, $target, $update) = @_;
	_check_page($source);
	_check_page($target);
	
	$source = $self->get_page($source) || die "Could not open page '$source'\n"
		unless ref $source;
	$target = $self->get_page($target) || die "Could not create page '$target'\n"
	       	unless ref $target;
	
	die "BUG: You tried to move a page marked as read-only\n"
		if $method eq 'move_page' and $source->{properties}{read_only};
	die "Page '$target' exists\n"
		if $target->exists and ! $target->equals($source);
		# if pages are equal the case can still differ
	
	if ($source->{repository} eq $target->{repository}
		and $target->{repository} ne $self
	) {
		$source->{repository}->$method($source, $target, $update);
	}
	else {
		$target->clone($source);
		if ($update) {
			my ($from, $to) = ($source->name, $target->name);
			$self->get_page($_)->update_links($from => $to)
				for $source->list_backlinks ;
		}
		$source->{repository}->delete_page($source)
			if $method eq 'move_page';
	}

	return $target;
}

=item delete_page(PAGE)

Delete PAGE. PAGE can be either a page name or a page object.

=cut

sub delete_page {
	my ($self, $page) = @_;
	_check_page($page);
	$page = $self->get_page($page) unless ref $page;
	die "BUG: You tried to delete a page marked as read-only\n"
		if $page->{properties}{read_only};
	if ($page->{repository} eq $self) {
		$page->status('deleted');
	}
	else {
		$page->{repository}->delete_page($page);
	}
	return $page;
}

=item C<resolve_link(PAGE, NAME)>

Returns a page object for NAME where NAME can be resolved
relative to PAGE.

=cut

sub resolve_link {
	my ($self, $page, $name) = @_;
	_check_page($page, 1);
	return $self->resolve_page($name) if $page->{repository} eq $self;
	return $page->{repository}->resolve_page($page, $name);
}

=item C<list_backlinks(PAGE)>

Returns a list of links to this page.

=cut

sub list_backlinks {
	my ($self, $page) = @_;
	_check_page($page);
	my @backlinks;
	for (grep /^:/, keys %$self) {
		push @backlinks, $self->{$_}->list_backlinks($page)
			if $self->{$_}->can('list_backlinks');
	}
	return @backlinks;
}

=item C<root()>

Returns paret or self.

=cut

sub root { return $_[0] }

=item C<search(QUERY, CALLBACK)>

TODO stable api for this using Zim::Selection

Results are given as arguments to CALLBACK in the form C<[PAGE, SCORE]>.

=cut

# We could probably use some kind of "query" object that coordinates searches
# on subsets of pages  and combines backlink lists with other keywords

sub search {
	my ($self, $query, $callback) = @_;
	for (grep /^:/, keys %$self) {
		$self->{$_}->_search($query, $callback)
			if $self->{$_}->can('_search');
	}
}

=back

=head2 Utility Methods

These methods can be used by the interface but do not need to be reproduced
in child handlers.

=over 4

=item C<add_child(NAMESPACE, CHILD, ...)>

This will connect a new child object to this repository under a certain
namespace. CHILD is a class name implementing the repository interface.
The class will be looked for in C<Zim::Repository::>.
All remaining arguments will be passed on to the constructor of the object.

=cut

sub add_child {
	my ($self, $namespace, $child, %args) = @_;
	$namespace =~ s/:?$/:/;
	_check_namespace($namespace);
	warn "# Adding child repository of type '$child' at '$namespace'\n";
	
	my $class = "Zim::Repository::" .
		join('::', map quotemeta($_), split '::', $child);
	#warn "use $child ();\n";
	eval "use $class ();";
	die "While loading child repository '$child' for namespace '$namespace' :\n".$@ if $@;
	$args{namespace} = $namespace;
	my $obj = $class->new(parent => $self, %args);

	$namespace =~ s/:*$//;
	$namespace = ':' if $namespace eq '';
	$self->{lc($namespace)} = $obj;

	my $ns = $self->{namespace};
	$ns =~ s/:*$//;
	my $regex = join '|', map quotemeta($_),
		sort {-(length($a) <=> length($b))} grep /^:./, keys %$self;
	$regex = length($regex) ? "(?:$regex)(?![^:])|:" : ':';
	$self->{regex} = qr#^(?i)\Q$ns\E($regex)#;
	#warn "regex: $self->{regex}\n";
}

=item C<interwiki_lookup(KEY, PAGE)>

Returns an url for an interwiki link with interwiki name KEY and page PAGE.
Lookup for names is case in-sensitive.

=cut

our %_urls;

sub interwiki_lookup {
	my ($class, $key, $page) = @_;
	$key = lc $key;
	my $url;
	if (exists $_urls{$key}) { $url = $_urls{$key} }
	else { # file lookup
		# TODO repository specific urls here
		
		# Lookup named zim repositories
		my $file = $class->lookup_by_name($key);
		if (defined $file) {
			$url = Zim::File->path2uri($file);
			$url =~ s#^file:#zim:#i;
			$url .= '?{NAME}';
		}

		# lookup interwiki config
		unless (length $url) {
			for (xdg_data_files(qw/zim urls.list/)) {
				$url = Zim::File->new($_)->read_hash($key);
				last if defined $url;
			}
		}
	}
	return unless length $url;
	$_urls{lc($key)} = $url;

	unless ($url =~ s/{NAME}/$page/) {
		$page =~ s/([^A-Za-z0-9\-_.!~*'()])/sprintf("%%%02X", ord($1))/ge;
		$url =~ s/{URL}/$page/ or $url .= $page;
	}
	
	return $url;
}

=item C<split_page(PAGE, LEVEL)>

Splits a page into multiple pages in the like-named namespace.
Splits the page at headings up to level LEVEL (1..5 default 1).

TODO

=item C<merge_page(NAME)>

TODO - work in progress

=cut

sub _flush_cache { # TODO TODO good API for this
	my $self = shift;
	for (grep /^:/, keys %$self) {
		$self->{$_}->_flush_cache
			if $self->{$_}->can('_flush_cache');
	}
}

=back

=head2 Class Methods

These methods can be called as C<< Zim->method(...) >>.

=over 4

=item C<lookup_by_name(NAME)>

Returns a path for a repository with NAME or undef if the repository is unknown.
The lookup is case in-sensitive.

=cut

sub lookup_by_name {
	my ($class, $name) = @_;
	$name = lc $name;
	for (xdg_config_files('zim', 'repositories.list')) {
		my ($hash) = Zim::File->new($_)->read_config;
		my ($key) = grep {lc($_) eq $name} keys %$hash;
		return Zim::File->abs_path($$hash{$key}, $ENV{HOME})
			if defined $key;
	}
	return undef;
}

1;

__END__

=back

=head1 BUGS

Please mail the author if you find any bugs.

=head1 AUTHOR

Jaap Karssenberg || Pardus [Larus] E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

L<zim(1)>,
L<Zim::Repository>

=cut
