package Zim::Repository;

use utf8;
use strict;
use Carp;
use UNIVERSAL qw/isa/;
use File::Spec;
use File::BaseDir qw/xdg_config_files xdg_data_home xdg_data_dirs xdg_data_files/;
use Zim::Selection;
use Zim::File;

our $VERSION = '0.17';

=head1 NAME

Zim::Repository - A wiki repository

=head1 SYNOPSIS

	use Zim::Repository;
	
	my $repo = Zim::Repository->new(undef, ':', $dir);
	my $page = $repo->get_page('Home');

=head1 DESCRIPTION

This class defines the public interface to the document repository
as used by L<zim>(1). It acts as a dispatcher that manages one or
more child objects to handle page requests. It also adds some tests
and thus tries to isolate the handlers from errors in the GUI.

The interface which child objects should support is defined in
L<Zim::Repository::Base>.

=head1 METHODS

=head2 Public Methods

The following methods can be used by the GUI.

=over 4

=item new(PARENT, NAMESPACE, DIR, ...)

Constructor. PARENT can be a parent repository object or undef.
NAMESPACE is the prefix for all pages managed by this repository.

DIR is the root namespace of the repository and might contain
a onfig file.

=cut

sub new {
	my ($class, $parent, $namespace, $node, $format) = @_;
		# passing along $format here is a hack ! FIXME
	$namespace =~ s/:?$/:/;
	$node = Zim::File->abs_path($node);
	#warn "repository root: $node\n";
	die "Not a directory: $node\n" if -e $node and ! -d $node;
	
	my $self = bless {
		parent => $parent,
		namespace => $namespace,
		dir => $node,
		config => {},
	}, $class;
	
	return $self unless length $node;
	
	# load config file
	#if ($type eq 'dir') {
		my $conf = Zim::File->new($node, '.zim.config');
		for ($conf->read) {
			chomp;
			next unless /\S/;
			my ($opt, $val) = split '=', $_, 2;
			if ($opt =~ /^:/) { # opt is namespace
				my ($class, $arg) = split '=', $val, 2;
				my @arg = defined($arg) ? (split ',', $arg) : ();
				$self->add_child($opt, $class, @arg);
			}
			else { $self->{config}{$opt} = $val }
		}
	#}
	
	# default config
	$self->add_child(':', 'Files', $node, $format) unless defined $self->{':'};

	return $self;
}

=item C<config()>

Returns a hash with config options.

=cut

sub config { $_[0]->{config} }

=item C<save_config()>

Saves the config to disk.

=cut

sub save_config {
	my $self = shift;
	my $node = $self->{dir};
	my $conf = Zim::File->new($node, '.zim.config');
	my $text = join '', map "$_=$$self{config}{$_}\n", keys %{$$self{config}};
	for ($conf->read) { $text .= $_ if /^:/ }
	$conf->write($text);
}

=item C<base()>

Returns the base directory for files.

The "base" directories for individual pages should either be the same as
this directory or a subdirectory of it.

=cut

sub base { return $_[0]->{dir} }

## 3 methods for checking arguments ##

sub _check_page {
	if (ref $_[0]) {
		croak "Object \"$_[0]\" is not a Zim::Page"
	       		unless isa $_[0], 'Zim::Page';
	}
	elsif ($_[1]) { croak "\"$_[0]\" is not a page object" }
	else { goto \&_check_page_name }
}

# Valid page names can contain letters, numbers and ".", "-", "(" and ")"
# The first character of a name can only be a letter or a number
# Absolute names start with a ":"
# Namespaces end in a ":"

sub _check_page_name {
	croak "\"$_[0]\" is not a valid page name"
		unless $_[0] =~ /^(?::+\w[\w\.\-\(\)]*)+$/;
}

sub _check_namespace {
	croak "\"$_[0]\" is not a valid namespace"
		unless $_[0] =~ /^(?::+|(?::+\w[\w\.\-\(\)]*)+:+)$/;
}

=item C<has_pages(NAMESPACE)>

Returns boolean whether NAMESPACE exists.

=item C<list_pages(NAMESPACE)>

Lists pages in NAMESPACE. Sub-namespaces have a trailing ':'
in the listing.

If a page is not present in this list that does not mean that it
doesn't exists. This list is used for hierarchical views of the
page structure; not all repositories need to support a hierarchical order.

=cut

sub has_pages {
	my ($self, $namespace) = @_;
	$namespace =~ s/:?$/:/;
	_check_namespace($namespace);

	return 1 if grep /^$namespace/i, keys %$self;

	$namespace =~ $self->{regex} or die "BUG: '$namespace' !~ qr/$self->{regex}/";
	$self->{lc($1)}->has_pages($namespace);
}

sub list_pages {
	my ($self, $namespace) = @_;
	$namespace =~ s/:?$/:/;
	_check_namespace($namespace);
	my @mount_points =
		grep { m/^:/ and s/$namespace:*([^:]+:*)$/$1:/ } keys %$self;
	$namespace =~ $self->{regex} or die "BUG: '$namespace' !~ qr/$self->{regex}/";
	return if $self->{lc($1)}->{no_show_in_sidepane}; # temp HACK
	return sort {lc($a) cmp lc($b)}
		@mount_points, $self->{lc($1)}->list_pages($namespace);
}

=item get_page(NAME)

Returns an object of a class inheriting from L<Zim::Page>.
When you ask for a page that doesn't exists yet, you should get a new object.
In case it is not possible to create the page in question C<undef> is returned.

=cut

sub get_page {
	my ($self, $name) = @_;
	_check_page_name($name);
	$name =~ $self->{regex} or die "BUG: '$name' !~ qr/$self->{regex}/";
	#warn "'$name' =~ /$self->{regex}/ => \$1 = '$1'\n";
	#warn "Dispatching get_page('$name') to '".lc($1)."'\n";
	return $self->{lc($1)}->get_page($name);
}

=item C<resolve_page(NAME)>

Like C<get_page()> but now expecting NAME to be user input,
so NAME is sanitized first.

=cut

sub resolve_page {
	my ($self, $name) = @_;
	#print STDERR "resolved $name to ";
	$name =~ s/^:?/:/;			# make absolute
	$name =~ s/:+$//;			# not a namespace
	$name =~ s/[^:\w\.\-\(\)]/_/g;		# replace forbidden chars
	$name =~ s/(:+)[\_\.\-\(\)]+/$1/g;	# remove non-letter at begin
	$name =~ s/_+(:|$)/$1/g;		# remove trailing underscore
	#print STDERR "$name\n";
	$name =~ $self->{regex} or die "BUG: '$name' !~ qr/$self->{regex}/";
	#warn "Dispatching resolve_page('$name') to '".lc($1)."'\n";
	#warn "Regex was: $self->{regex}\n";
	return $self->{lc($1)}->resolve_page($name);
}

=item C<resolve_namespace(NAME)>

Returns a namespace string. Used to sanitize user input.

=cut

sub resolve_namespace {
	my ($self, $name) = @_;
	$name =~ s/^:?/:/;			# make absolute
	$name =~ s/:?$/:/;			# is a namespace
	$name =~ s/[^:\w\.\-\(\)]/_/g;		# replace forbidden chars
	$name =~ s/(:+)[\_\.\-\(\)]+/$1/g;	# remove non-letter at begin
	$name =~ s/_+(:|$)/$1/g;		# remove trailing underscore
	return $name;
}

=item copy_page(SOURCE, TARGET)

Copy page SOURCE to TARGET. Both arguments can be either page names
or objects. Returns the (new) TARGET object.

=item move_page(SOURCE, TARGET)

Move page SOURCE to TARGET. Both arguments can be either page names
or objects. Returns the (new) TARGET object.

=cut

sub copy_page { _clone('copy_page', @_) }

sub move_page { _clone('move_page', @_) }

sub _clone { # Copy and move are almost the same
	my ($method, $self, $source, $target) = @_;
	_check_page($source);
	_check_page($target);
	
	$source = $self->get_page($source) || die "Could not open page '$source'\n"
		unless ref $source;
	$target = $self->get_page($target) || die "Could not create page '$target'\n"
	       	unless ref $target;
	
	die "Page '$target' exists\n"
		if $target->exists and ! $target->equals($source);
		# if pages are equal the case can still differ
	
	if ($source->{repository} eq $target->{repository}) {
		$source->{repository}->$method($source, $target);
	}
	else {
		$target->clone($source);
		$source->{repository}->delete_page($source)
			if $method eq 'move_page';
	}

	return $target;
}

=item delete_page(PAGE)

Delete PAGE. PAGE can be either a page name or a page object.

=cut

sub delete_page {
	my ($self, $page) = @_;
	_check_page($page);
	$page = $self->get_page($page) unless ref $page;
	$page->{repository}->delete_page($page);
	return $page;
}

=item C<search(QUERY, CALLBACK)>

TODO stable api for this using Zim::Selection

Results are given as arguments to CALLBACK in the form C<[PAGE, SCORE]>.

=cut

# We could probably use some kind of "query" object that coordinates searches
# on subsets of pages  and combines backlink lists with other keywords

sub search {
	my ($self, $query, $callback) = @_;
	for (grep /^:/, keys %$self) {
		$self->{$_}->_search($query, $callback)
			if $self->{$_}->can('_search');
	}
}

=item C<export(PAGE, %OPTS)>

TODO stable api for this

=cut

# TODO use verbose option

sub export {
	my ($self, $page, %opts) = @_;
	$page = shift @$page; # TODO
	for (qw/format template root/) {
		die "for exporting you need to provide at least a format, a template and an output root dir.\n" unless length $opts{$_};
	}

	$page = $page->name if ref $page;
	my $namespace = ($page =~ /:$/) ? $page : '';
	#print STDERR "export root: $opts{root}\n";
	my $exporter = ref($opts{root}) ? $opts{root} :
		Zim::Repository::Files->new(undef, '', $opts{root}, $opts{format});
	#print STDERR "export template: $opts{template}\n";
	my $template = $opts{template};
	unless (-f $template) {
		if ($template =~ m#/#) { die "No such file: $template\n" }
		else {
			my $key = $template;
			$template = Zim::Repository->lookup_template($opts{format}, $key);
			die "No such template: $key" unless defined $template;
		}
	}
	$exporter->{_template} = $template; # FIXME better interface for this - do not confuse with the template for new pages
	
	$opts{recurs} = 1 unless defined $opts{recurs};
	if (length $namespace) {
		my $index = $self->_export_namespace($exporter, $page, 0, \%opts);
		#use Data::Dumper; print Dumper $index;
		$self->_write_index($exporter, $index);
	}
	else {
		$self->_export_page($exporter, $page, \%opts);
	}

}

sub _export_namespace {
	my ($self, $exporter, $namespace, $lvl, $opts) = @_;
	_check_namespace($namespace);
	
	my @index = $self->list_pages($namespace);
	my @pages = grep $$_[1]->exists,
	            map [$_ => $self->get_page($namespace.$_)], @index;
	
	for (0 .. $#pages) {
		my $page = $pages[$_][1];
		$exporter->{_next} = $pages[($_+1)%@pages][1];
		$exporter->{_prev} = $pages[$_-1][1];
		$self->_export_page($exporter, $page, $opts);
	}
	my %pages = map @$_, @pages;
	@pages = (); # free
	
	my %index;
	if ($$opts{recurs}) {
		$index{$_} = $self->_export_namespace(
			$exporter, $namespace.$_, $lvl+1, $opts )
			for grep /:$/, @index;
	}

	my $index = [$lvl, $namespace];
	for my $i (@index) {
		my $p = $i;
		$p =~ s/:*$//;
		push @$index, $p         if $pages{$i};
		push @$index, $index{$i} if $index{$i};
	}
	return $index;
}

sub _export_page {
	my ($self, $exporter, $page, $opts) = @_;
	
	$page =~ s/:+$// unless ref $page; # TODO doesn't belong here
	
	my $source;
	if (ref $page) { ($source, $page) = ($page, $page->name) }
	else           { $source = $self->get_page($page)        }
	die "No such page: $page\n" unless $source;
	$$opts{callback}->($source) || die "Export Cancelled at page $page\n"
		if $$opts{callback};

	
	return 0 unless $source->exists;
	print STDERR "Exporting: $page\n";
	
	my $target = $exporter->get_page($page);
	$target->clone($source);
	
	return 1;
}

sub _write_index {
	my ($self, $exporter, $index) = @_;
	
	return if $self->get_page(':index')->exists;
	print STDERR "Writing index\n";
	my $page = $exporter->get_page(':index');
	
	$exporter->{_next} = $page;
	$exporter->{_prev} = $page;
	$page->set_parse_tree( ['Document', {},
		['head1', {}, "Document index"],
		['Para', {}, _conv_index($index) ],
	] );
	
}

sub _conv_index {
	my $node = shift;
	my $lvl = shift @$node;
	my $nsp = shift @$node;
	my @list;
	my $prev = ':'; # no page can match this
	for (@$node) {
		if (ref $_) {
			my $n = $$_[1];
			unless ($n =~ /(^|:)$prev:*$/) { # show namespace in tree
				$n =~ s/:+$//;
				$n =~ s/^.*://;
				push @list, ("\t"x$lvl)."* ",['bold', {}, $n], "\n";
			}
			push @list, _conv_index($_); # recurs
		}
		else { push @list, ("\t"x$lvl)."* ", ['link', {to => $nsp.$_}, $_], "\n" }
		$prev = $_;
	}
	return @list;
}

=back

=head2 Utility Methods

These methods can be used by the interface but do not need to be reproduced
in child handlers.

=over 4

=item C<get_selection(\%OPT, @PAGES..)>

Returns a L<Zim::Selection> object for PAGES.
Both PAGES and OPT are optional arguments.

=cut

sub get_selection {
	my $self = shift;
	my $opt = ref($_[0]) ? shift : {} ;
	my @pages = @_ ? (@_) : (':') ;
	return Zim::Selection->new($self, $opt, @pages);
}

=item C<add_child(NAMESPACE, CHILD, ...)>

This will connect a new child object to this repository under a certain
namespace. CHILD is a class name implementing the repository interface.
The class will be looked for in C<Zim::Repository::>.
All remaining arguments will be passed on to the constructor of the object.

=cut

sub add_child {
	my ($self, $namespace, $child, @args) = @_;
	$namespace =~ s/:?$/:/;
	_check_namespace($namespace);
	#warn "Adding child  type '$child' for '$namespace' (@args)\n";
	
	$child = "Zim::Repository::$child" unless $child =~ /::/;
	eval "use $child ();";
	die if $@;
	$child = $child->new($self, $namespace, @args);

	$namespace =~ s/:*$//;
	$namespace = ':' if $namespace eq '';
	$self->{lc($namespace)} = $child;

	my $ns = $self->{namespace};
	$ns =~ s/:*$//;
	my $regex = join '|', reverse
		sort {length($a) <=> length($b)} grep /^:./, keys %$self;
	$regex = length($regex) ? "(?:$regex)(?![^:])|:" : ':';
	$self->{regex} = qr#^(?i)$ns($regex)#;
	#warn "regex: $self->{regex}\n";
}

=item C<interwiki_lookup(KEY, PAGE)>

Returns an url for an interwiki link with interwiki name KEY and page PAGE.
Lookup for names is case in-sensitive.

=cut

our %_urls;

sub interwiki_lookup { # FIXME linked from Repository base for export stuff
	my ($self, $key, $page) = @_;
	$key = lc $key;
	my $url;
	if (exists $_urls{$key}) { $url = $_urls{$key} }
	else { # file lookup
		# TODO repository specific urls here
		
		# Lookup named zim repositories
		my $file = Zim::Repository->lookup_by_name($key);
		if (defined $file) {
			$url = Zim::File->path2uri($file);
			$url =~ s#^file:#zim:#i;
			$url .= '?{NAME}';
		}

		# lookup interwiki config
		unless (length $url) {
			for (xdg_data_files(qw/zim urls.list/)) {
				$url = Zim::File->new($_)->read_hash($key);
				last if defined $url;
			}
		}
	}
	return unless length $url;
	$_urls{lc($key)} = $url;

	unless ($url =~ s/{NAME}/$page/) {
		$page =~ s/([^A-Za-z0-9\-_.!~*'()])/sprintf("%%%02X", ord($1))/ge;
		$url =~ s/{URL}/$page/ or $url .= $page;
	}
	
	return $url;
}

=item C<relative_path(FILE, PAGE)>

Checks to see if the path for FILE can be made relative to the path for PAGE.
If PAGE is undef tries to make relative to the repository.
Returns either the relative or the absolute path.

=cut

sub relative_path {
	# 3 way directory comparison
	# Returns path relative to 'ref' when file is below 'root'
	# returns absolute filename otherwise
	my ($self, $file, $page) = @_;
	my $ref = $page ? $page->properties->{base} : $self->{dir};
	$ref = Zim::File->abs_path($ref);
	return $file unless defined $ref;
	$file = Zim::File->abs_path($file,$ref);
	return $file if $file =~ m#^\w\w+:/#;
	my $root = $self->{dir};
	$root = Zim::File->abs_path($root);
	#warn "check whether '$file' is below '$ref'\n";

	my @root = split m#/+#, $root;
	my @ref = split m#/+#, $ref;
	my @file = split m#/+#, $file;
	for (@root) { # remove root part - return if no match
		my $dir = shift @file;
		#warn "match $dir $_\n";
		return $file if $dir ne $_;
		$dir = shift @ref;
		return $file if $dir ne $_; # just to be sure
	}
	while (@ref) { # remove common path
		last if $ref[0] ne $file[0];
		shift @ref;
		shift @file;
	}

	if (@ref) { unshift @file, '..' for 0 .. $#ref }
	else      { unshift @file, '.'                 }
	
	return join '/', @file; # un-localize
}

=item C<split_page(PAGE, LEVEL)>

Splits a page into multiple pages in the like-named namespace.
Splits the page at headings up to level LEVEL (1..5 default 1).

This method can only be used for page objects that support the
"formatted" interface like L<Zim::Page::Text>.

=cut

# TODO - maybe have a module with pare-tree logic would make this method a bit smaller ?

sub split_page {
	my ($self, $page, $max) = @_;
	_check_page($page);
	$page = $self->get_page($page) unless ref $page;
	$max ||= 1;

	die "Page does not exist\n" unless $page;
	die "Page ".$page->name." does not support the 'formatted' interface\n"
		unless $page->has_interface('formatted');

	# Build nested structure of headings
	my $root = [0, $page->basename, []]; # [level, title, [children], content]
	my @stack = ($root);
	my $tree = $page->get_parse_tree;
	for my $part (@$tree[2 .. $#$tree]) {
		if (! ref $part or $$part[0] !~ /^head/) { push @{$stack[-1]}, $part }
		else { # TODO check head not empty
			$$part[0] =~ /^head(\d+)/;
			my $level = $1;
			if ($level > $max) {
				push @{$stack[-1]}, $part;
				next;
			}
			
			my $title = _to_string($part);
			if ($stack[-1] eq $root) { # match head to page name - TODO better compare logic
				my $name = $title;
				$name =~ s/\W/_/;
				if (lc($name) eq lc($$root[1])) {
					push @{$stack[-1]}, $part;
					next;
				}
			}
			else { # find parent
				until ($stack[-1][0] < $level) { pop @stack }
			}
			
			my $item = [$level, $title, [], $part];
			push @{$stack[-1][2]}, $item; # attach to parent
			push @stack, $item; # become the new head of the tree
		}
	}
	
	# Write files
	_save_head_page($self, $page->name, $_) for @{$$root[2]};
	# Write root files last .. if anything goes wrong it should still be there
	$tree = [ @$tree[0, 1], @$root[3 .. $#$root] ];
	$page->set_parse_tree($tree);
}

sub _to_string { # convert parse tree node to plain text (no formatting)
	my $ref = shift;
	my $string = '';
	my @todo = @$ref[2 .. $#$ref];
	while (@todo) {
		my $n = shift @todo;
		if (ref $n) { unshift @todo, $$n[2 .. $#$n] }
		else        { $string .= $n                 }
	}
	return $string;
}

sub _save_head_page {
	# TODO promote heading => lvl 1
	my ($self, $path, $ref) = @_;
	$path = $self->resolve_page($path.':'.$$ref[1]);
	my $page = $self->get_page($path);
	my $tree = $page->exists ? ['Document', {}] : $page->get_parse_tree;
	push @$tree, @$ref[3 .. $#$ref];
	$page->set_parse_tree($tree);
	_save_head_page($self, $path, $_) for @{$$ref[2]};
}

=item C<merge_page(NAME)>

TODO - work in progress

=cut

sub merge_page {
	my ($self, $page) = @_;
	_check_page($page);
	$page = $self->get_page($page) unless ref $page;

	die "Could not open page\n" unless $page;
	die "Page ".$page->name." does not support the 'formatted' interface\n"
		unless $page->has_interface('formatted');

	my @pages = _list_pages($self, $page->name, 1);
	warn join "\n", "Merging:", map "\t@$_", @pages;

	# demote levels => lvl
}

sub _list_pages {
	my ($self, $name, $lvl) = @_;
	$name =~ s/:$//;
	my @list;
	for ($self->list_pages($name)) {
		my $p = "$name:$_";
		my $r = ($p =~ s/:$//);
		push @list, [$lvl, $p];
		push @list, _list_pages($self, $p, $lvl+1) if $r; # recurs
	}
	return @list;
}

sub _flush_cache {
	my $self = shift;
	for (grep /^:/, keys %$self) {
		$self->{$_}->_flush_cache
			if $self->{$_}->can('_flush_cache');
	}
}

=back

=head2 Class Methods

These methods can be called as C<< Zim::Repository->method(...) >>.

=over 4

=item C<lookup_by_name(NAME)>

Returns a path for a repository with NAME or undef if the repository is unknown.
The lookup is case in-sensitive.

=cut

sub lookup_by_name {
	my ($class, $name) = @_;
	$name = lc $name;
	for (xdg_config_files('zim', 'repositories.list')) {
		my ($hash) = Zim::File->new($_)->read_config;
		my ($key) = grep {lc($_) eq $name} keys %$hash;
		return Zim::File->abs_path($$hash{$key}, $ENV{HOME})
			if defined $key;
	}
	return undef;
}

=item C<lookup_template(FORMATE, NAME)> 

Returns a file name or undef.

TODO: This method should go to either Zim::Template or Zim::Formats
when these are created.

=cut

sub lookup_template {
	my ($class, $format, $name) = @_;
	my $map = $class->list_templates($format);
	return $$map{$name} if exists $$map{$name};
	my ($key) = grep {lc $_ eq lc $name} keys %$map;
	return $$map{$key} if defined $key;
	return undef;
}

=item C<list_templates(FORMAT)>

Returns a hash with name => filename pairs of available templates.
Used for exporting.

TODO: This method should go to either Zim::Template or Zim::Formats
when these are created.

=cut

sub list_templates {
	my ($class, $format) = @_;
	die "FIXME: only html format supported" unless $format eq 'html';
	# Either have a directory with templates for each format
	# or have a format => extension map somewhere

	my %templates;
	for (xdg_data_home(), xdg_data_dirs()) {
		my $dir = File::Spec->catdir($_, 'zim', 'templates');
		next unless -d $dir;
		for my $f (Zim::File->list_dir($dir)) {
			next unless $f =~ s/\.html$//;
			$templates{$f} = "$dir/$f.html"
				unless defined $templates{$f};
		}
	}

	return \%templates;
}

1;

__END__

=back

=head1 BUGS

Please mail the author if you find any bugs.

=head1 AUTHOR

Jaap Karssenberg || Pardus [Larus] E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

L<Zim>, L<Zim::Repository::Files>, L<Zim::Repository::DBI>

=cut
