package Zim::GUI;

use strict;
use vars qw/$AUTOLOAD %Config/;
use Carp;
use POSIX qw(strftime);
use File::BaseDir qw/
	xdg_config_home xdg_config_files
	xdg_data_home xdg_data_dirs xdg_data_files
	xdg_cache_home /;
use File::MimeInfo::Magic;
use Gtk2;
use Gtk2::Gdk::Keysyms;
use Gtk2::SimpleList;
use Zim::File;
use Zim::History;
use Zim::GUI::Component;
use Zim::GUI::PathBar;
use Zim::GUI::PageView;

eval "use File::DesktopEntry; use File::MimeInfo::Applications";
my $has_mimeapplications = $@ ? 0 : 1;
warn "Could not use 'File::MimeInfo::Applications', disabling application bindings\n"
	unless $has_mimeapplications;

our $VERSION = '0.17';
our $LONG_VERSION = << "EOT";
zim $VERSION - A desktop wiki and outliner

Copyright (c) 2006 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

Please report bugs to pardus\@cpan.org
EOT

our @ISA = qw/Zim::GUI::Component/;

our ($ICON) = xdg_data_files('pixmaps', 'zim.png');
warn "WARNING: Could not find 'zim.png', is \$XDG_DATA_DIRS set properly ?\n"
	unless length $ICON;

=head1 NAME

Zim::GUI - The application object for zim

=head1 SYNOPSIS

	use Zim::GUI;
	use Zim::Repository;
	
	my $zim = Zim::GUI->new(\%SETTINGS);
	
	my $repository = Zim::Repository->new($DIR);
	$zim->set_repository($repository);
	
	$zim->gui_init;
	$zim->gui_show;
	
	Gtk2->main;
	
	exit;

=head1 DESCRIPTION

This is developer documentation, for the user manual try
executing C<zim --doc>. For commandline options see L<zim>(1).

This module provides the application object for the Gtk2 application B<zim>.
The application has been split into several components for which the 
modules can be found in the Zim::GUI:: namespace. This object brings
together these components and manages the settings and data objects.

=head1 EXPORT

None by default.

=head1 METHODS

Undefined methods are AUTOLOADED either to an component or to the main
L<Gtk2::Window> object.

=over 4

=cut

our $LINK_ICON = Gtk2->CHECK_VERSION(2, 6, 0) ? 'gtk-connect' : 'gtk-convert';
	# gtk-connect stock item was introduced in 2.6.0

our $CONFIG = # Default config values
q{
pane_pos	120
pane_vis	0
statusbar_vis	1
toolbar_vis	1
pathbar_type	recent
width		500
height		350
x
y
default_root
default_home	Home
browser
file_browser
email_client
date_string	%A %d/%m/%Y
hist_max	20
undo_max	50
follow_new_link	1
backsp_unindent	1
use_xdg_cache	0
show_spell	1
spell_language
cal_namespace	:Date:
show_cal	0
plugins
use_camelcase	1
use_utf8_ent	1
use_linkfiles	1
use_autolink	0
textfont
tearoff_menus	0
use_autoselect	1
follow_on_enter	1
use_ctrl_space	1
};

our %DEFAULTS = (
	file_browser => q/rox '%d'/,
	browser	     => q/firefox '%u'/,
	email_client => q/thunderbird '%u'/,
);

my $ui_menus =
# name		stock id 	label
q{
FileMenu	.		_File
EditMenu	.		_Edit
ViewMenu	.		_View
SearchMenu	.		_Search
FormatMenu	.		For_mat
ToolsMenu	.		_Tools
GoMenu		.		_Go
HelpMenu	.		_Help
PathBarMenu	.		P_athbar type
};

my $ui_actions = 
# name,		stock id,	label,		accelerator,	tooltip
q{
NewPage		gtk-new		_New Page	.		New page
popup_NewPage	gtk-new		_New Page	.		New page
OpenRep		gtk-open	_Open...	<ctrl>O		Open repository
Save		gtk-save	_Save		<ctrl>S		Save page
SaveCopy	gtk-save-as	S_ave A Copy...	<shift><ctrl>S	Save a copy
Export		.		E_xport...	.		Export
EmailPage	.		_Send to...	<ctrl>M		Mail page
RenamePage	.		_Rename page...	F2		Rename page
popup_RenamePage	.		_Rename page...	.		Rename page
DeletePage	.		_Delete page	.		Delete page
popup_DeletePage	.		_Delete page	.		Delete page
Props		gtk-properties	Proper_ties	.		Properties dialog
Close		gtk-close	_Close		<ctrl>W		Close window
Quit		gtk-quit	_Quit		<ctrl>Q		Quit
Search		gtk-find	_Search...	<shift><ctrl>F	Search
SearchBL	.		Search _Backlinks...	.	Search Back links
Prefs		gtk-preferences	Pr_eferences	.		Preferences dialog
Reload		gtk-refresh	_Reload		<ctrl>R		Reload page
OpenDir		gtk-open	Open _folder	.		Open folder
RBIndex		.		Re-build index	.		Rebuild index
GoBack		gtk-go-back	_Back		<alt>Left	Go page back
GoForward	gtk-go-forward	_Forward	<alt>Right	Go page forward
GoParent	gtk-go-up	_Parent		<alt>Up		Go to parent page
GoChild		gtk-go-down	_Child		<alt>Down	Go to child page
GoPrev		.		_Previous	<alt>Page_Up	Go to previous page
GoNext		.		_Next		<alt>Page_Down	Go to next page
GoToday		.		To_day		<alt>D		Today
GoHome		gtk-home	_Home		<alt>Home	Go home
JumpTo		gtk-jump-to	_Jump to...	<ctrl>J		Jump to page
ShowHelp	gtk-help	_Contents	F1		Help contents
About		gtk-about	_About		.		About
};

my $ui_toggle_actions =
# name,		stock id,	label,		accelerator,	tooltip 
q{
TToolBar	.		Toolbar		.		Show toolbar
TStatusBar	.		_Statusbar	.		Show statusbar
TPane		gtk-index	Side _Pane	F9		Show side pane
TCalendar	gtk-paste	Calen_dar	<ctrl><shift>D	Show calendar
};

my $ui_radio_actions =
# name,		stock id,	label,		accelerator,	tooltip
q{
PBRecent	.		_Recent pages	.		.
PBHistory	.		_History	.		.
PBNamespace	.		_Namespace	.		.
PBHidden	.		H_idden		.		.
};

my $ui_layout = q{<ui>
	<menubar name='MenuBar'>
		<menu action='FileMenu'>
			<menuitem action='NewPage'/>
			<menuitem action='OpenRep'/>
			<separator/>
			<menuitem action='Save'/>
			<menuitem action='SaveCopy'/>
			<menuitem action='Export'/>
			<separator/>
			<placeholder name='PrintActions'/>
			<menuitem action='EmailPage'/>
			<separator/>
			<placeholder name='FileMods'/>
			<separator/>
			<menuitem action='Props'/>
			<separator/>
			<menuitem action='Close'/>
			<menuitem action='Quit'/>
		</menu>
		<menu action='EditMenu'>
			<placeholder name='EditPage'/>
			<separator/>
			<placeholder name='InsertItems'/>
			<separator/>
			<menuitem action='Prefs'/>
		</menu>
		<menu action='ViewMenu'>
			<menuitem action='TToolBar'/>
			<menuitem action='TStatusBar'/>
			<menuitem action='TPane'/>
			<menu action='PathBarMenu'>
				<menuitem action='PBRecent'/>
				<menuitem action='PBHistory'/>
				<menuitem action='PBNamespace'/>
				<menuitem action='PBHidden'/>
			</menu>
			<separator/>
			<menuitem action='TCalendar'/>
			<placeholder name='PluginItems'/>
			<separator/>
			<menuitem action='Reload'/>
		</menu>
		<menu action='FormatMenu'></menu>
		<menu action='SearchMenu'>
			<placeholder name='FindItems'/>
			<separator/>
			<menuitem action='Search'/>
			<menuitem action='SearchBL'/>
		</menu>
		<menu action='ToolsMenu'>
			<placeholder name='PageTools'/>
			<menuitem action='OpenDir'/>
			<separator/>
			<menuitem action='RBIndex'/>
		</menu>
		<placeholder name='PluginMenus'/>
		<menu action='GoMenu'>
			<menuitem action='GoBack'/>
			<menuitem action='GoForward'/>
			<menuitem action='GoParent'/>
			<separator/>
			<menuitem action='GoNext'/>
			<menuitem action='GoPrev'/>
			<separator/>
			<menuitem action='GoToday'/>
			<placeholder name='PluginItems'/>
			<separator/>
			<menuitem action='GoHome'/>
			<menuitem action='JumpTo'/>
		</menu>
		<menu action='HelpMenu'>
			<menuitem action='ShowHelp'/>
			<menuitem action='About'/>
		</menu>
	</menubar>
	<toolbar name='ToolBar'>
		<placeholder name='File'/>
		<separator/>
		<placeholder name='Edit'/>
		<separator/>
		<placeholder name='View'>
			<toolitem action='TPane'/>
		</placeholder>
		<separator/>
		<placeholder name='Go'>
			<toolitem action='GoHome'/>
			<toolitem action='GoBack'/>
			<toolitem action='GoForward'/>
		</placeholder>
		<separator/>
		<placeholder name='Search'/>
		<separator/>
		<placeholder name='Format'/>
		<separator/>
		<toolitem action='TCalendar'/>
		<placeholder name='Tools'/>
	</toolbar>
	<popup name='PagePopup'>
		<menuitem action='popup_NewPage'/>
		<menuitem action='popup_RenamePage'/>
		<menuitem action='popup_DeletePage'/>
	</popup>
	<accelerator action='GoChild'/>
</ui>};

my $ui_layout_rw = q{<ui>
<menubar name='MenuBar'>
		<menu action='FileMenu'>
			<placeholder name='FileMods'>
				<menuitem action='RenamePage'/>
				<menuitem action='DeletePage'/>
			</placeholder>
		</menu>
</menubar>
</ui>};

our $CURRENT;

=item C<new(settings => \%SETTINGS)>

Simple constructor.

=cut

sub init {
	my $self = shift;

	# Sanitize config
	my %set = %{$$self{settings}}; # trust input from bin/zim
	$self->load_config;
	for (split /\n/, $CONFIG) {
		my ($key, $val) = split /\t/, $_, 2;
		$set{$key} = defined($$self{settings}{$key})
			? $$self{settings}{$key} : $val;
	}
	%{$$self{settings}} = %set;
	
	# Initialize other attributes
	$self->{_message_timeout} = -1;
	$self->{_save_timeout} = -1;
	$self->{icon_file} = $ICON;
	$self->{app} = $self; # fool base class
	
	return $self;
}

=item C<gui_init()>

This method initializes all GUI objects that make up the application.

=cut

sub gui_init {
	my $self = shift;
	
	## Setup the window
	my $window = Gtk2::Window->new('toplevel');
	$window->set_default_size(@{$self->{settings}}{'width', 'height'});
	$window->signal_connect_swapped(delete_event => \&on_delete_event, $self);
	$window->signal_connect(destroy => sub { Gtk2->main_quit });
	$window->set_icon(
		Gtk2::Gdk::Pixbuf->new_from_file($self->{icon_file}) );
	$window->set_title('Zim');
	$self->{window} = $window;

	$SIG{'USR1'} = sub { # toggle window on `kill -USR1`
		my $iconified = grep {$_ eq 'iconified'}
			$window->window->get_state;
		$iconified ? $window->present : $window->iconify ;
	}; # defined _before_ plugins are loaded - so TrayIcon can change this
	
	#$window->signal_connect(hide => sub { # FIXME does not work - patched in TrayIcon
	#		@{$self->{settings}}{'x', 'y'} = $window->get_position;
	#		warn "hiding window: @{$self->{settings}}{'x', 'y'}\n";
	#	} );
	$window->signal_connect(show => sub {
			my ($x, $y) = @{$self->{settings}}{'x','y'};
			#warn "showing window: $x, $y\n";
			$window->move($x,$y) if defined($x) and defined ($y);
		} );
			

	my $vbox = Gtk2::VBox->new(0, 0);
	$window->add($vbox);
	$self->{vbox} = $vbox;

	## Setup actions and ui for menubar and toolbar
	my $ui = Gtk2::UIManager->new;
	$window->add_accel_group($ui->get_accel_group);
	$ui->set_add_tearoffs($self->{settings}{tearoff_menus});
	$self->{ui} = $ui;
	
	$self->add_actions($ui_menus, 'MENU');
	$self->add_actions($ui_actions);
	$self->add_actions($ui_toggle_actions, 'TOGGLE');
	$self->add_actions($ui_radio_actions, 'RADIO', 'TPathBar');
	
	$self->actions_set_sensitive(
		Save      => $self->{settings}{read_only} ? 0 : 1,
		GoBack    => 0,
		GoForward => 0,
		GoParent  => 0,
		EmailPage => 0,
	);
	
	$self->add_ui($ui_layout);
	$self->add_ui($ui_layout_rw) unless $self->{settings}{read_only};
	
	$self->{menubar} = $ui->get_widget("/MenuBar");
	$vbox->pack_start($self->{menubar}, 0,1,0);
	$self->{toolbar} = $ui->get_widget("/ToolBar");
	$vbox->pack_start($self->{toolbar}, 0,1,0);
	$self->{toolbar}->hide;
	$self->{toolbar}->set_no_show_all(1);

	## General window layout
	my $hpaned = Gtk2::HPaned->new();
	$hpaned->set_position($self->{settings}{pane_pos});
	$vbox->pack_start($hpaned, 1,1,0);
	$self->{hpaned} = $hpaned;

	my $l_vbox = Gtk2::VBox->new(0, 0);
	$hpaned->add1($l_vbox);
	$l_vbox->set_no_show_all(1);
	$self->{l_vbox} = $l_vbox;

	my $r_vbox = Gtk2::VBox->new(0, 0);
	$hpaned->add2($r_vbox);
	$self->{r_vbox} = $r_vbox;

	my $shbox = Gtk2::HBox->new(0, 2);
	$shbox->set_no_show_all(1);
	$vbox->pack_start($shbox, 0,1,0);
	$self->{statusbar} = $shbox;
	
	## Status bar
	my $s1 = Gtk2::Statusbar->new;
	$s1->set_has_resize_grip(0);
	$shbox->pack_start($s1, 1,1,0);
	$self->{status1} = $s1;

	my $frame = Gtk2::Frame->new();
	$frame->set_shadow_type('in');
	$s1->pack_end($frame, 0,1,0);
	my $ebox = Gtk2::EventBox->new();
	$frame->add($ebox);
	my $l1 = Gtk2::Label->new();
	$l1->set_size_request(120,10);
	$l1->set_alignment(0, 0.5);
	$l1->set_mnemonic_widget( ($self->get_action('SearchBL')->get_proxies)[0] );
	$ebox->add($l1);
	$self->{statusl} = $l1;
	
	my $s2  = Gtk2::Statusbar->new;
	$s2->set_size_request(80,10);
	$shbox->pack_end($s2, 0,1,0);
	$self->{status2} = $s2;
	
	$ebox->signal_connect_swapped(button_press_event => \&on_click_backlinks, $self);

	## Build side pane
	$self->{autoload}{TreeView} = sub {
		my %args = @_;
		eval "require Zim::GUI::TreeView";
		die $@ if $@;
		my $tree_view = Zim::GUI::TreeView->new(%args);
		$tree_view->load_index();
		$tree_view->signal_connect(row_activated =>
			sub { $self->TPane(0) if $self->{_pane_visible} == -1 } );
		$l_vbox->add($tree_view->widget);
		return $tree_view;
	};

	## Build content of the main area
	my $path_bar = Zim::GUI::PathBar->new(app => $self);
	$r_vbox->pack_start($path_bar->widget, 0,1,5);
	$self->{objects}{PathBar} = $path_bar;

	my $page_view = Zim::GUI::PageView->new(app => $self);
	$page_view->set_font($self->{settings}{textfont})
		if defined $self->{settings}{textfont};
	$r_vbox->pack_end($page_view->widget, 1,1,0);
	$self->{objects}{PageView} = $page_view;

	# FIXME shouldn't the repository manage this directory ?
	if ($self->{settings}{use_xdg_cache}) {
		my $root = File::Spec->rel2abs($self->{repository}{dir});
		my ($vol, $dirs) = File::Spec->splitpath($root, 'NO_FILE');
		my @dirs = File::Spec->splitdir($dirs);
		$self->{settings}{hist_file} ||= File::Spec->catfile( 
			xdg_cache_home, 'zim', join ':', $vol, @dirs );
	}
	else {
		my $root = File::Spec->rel2abs($self->{repository}{dir});
		$self->{settings}{hist_file} ||= File::Spec->catfile($root, '.zim.history'); 
	}
	
	eval {
		my $history = Zim::History->new(
			$self->{settings}{hist_file}, $page_view,
			$self->{settings}{hist_max} );
		$self->{objects}{History} = $history;
	};
	$self->exit_error($@) if $@;

	## Some wiring
	my $accels = $ui->get_accel_group;
	# Ctrl-Space / Alt-Space
	my @combo = ([ord(' '), ['mod1-mask']]);
	push @combo, [ord(' '), ['control-mask']] if $self->{settings}{use_ctrl_space};
	$accels->connect( 
		@$_, ['visible'],
		sub {
			if ($self->{settings}{pane_vis}) {
				my $tree_view = $self->TreeView;
				if ($tree_view->has_focus) { $page_view->grab_focus }
				else                       { $tree_view->grab_focus }
			}
			else {
				my $vis = $self->{_pane_visible} ? 0 : -1;
				$self->TPane($vis);
			}
		} ) for @combo;
	$accels->connect( Gtk2::Accelerator->parse('F5'), ['visible'],
		sub { $self->Reload } );

	
	$self->signal_connect('page_loaded', sub {
		my $self = shift;
		my $state = $self->History->get_state;
		$self->actions_set_sensitive(
			GoBack     => ($$state{back} ? 1 : 0),
			GoForward  => ($$state{forw} ? 1 : 0),
			GoParent   => ($self->{page}->namespace =~ /[^:]/),
		)
	} );
	$self->signal_connect('page_loaded', sub {
		my $self = shift;
		my $has_src = grep {$_ eq 'source'} $self->{page}->interfaces;
		$self->actions_set_sensitive(EmailPage => $has_src);
	} );

	## List autoloaded components
	$self->{autoload}{$_} = 'Zim::GUI::'.$_
		for qw/ SearchDialog FindReplaceDialog
		        Calendar ExportDialog
		       	PreferencesDialog PropertiesDialog/ ;

	## Load plugins
	$self->plug($_) for grep length($_), split /,/, $self->{settings}{plugins};

	## Try saving the program on system signals
	for my $sig (qw/TERM HUP PIPE/) {
		$SIG{$sig} = sub {
			$self->Quit();
			$self->exit_error(undef,"Signal $sig caught\n");
		};
	}

	# Try saving on desktop exit
#	$window->get_display->signal_connect( closed => sub {
#			print "Display closed\n";
#		} );
}

=item C<gui_show()>

=cut

sub gui_show {
	my $self = shift;
	
	## Show all widgets .. well most of them
	$self->{ui}->ensure_update; # make sure the menu is complete
	$self->{window}->show_all;
	$self->{statusbar}->set_no_show_all(0);
	$self->{l_vbox}->set_no_show_all(0);
	$self->{toolbar}->set_no_show_all(0);
#	my ($x, $y) = @{$self->{settings}}{'x','y'};
#	$window->move($x,$y) if defined($x) and defined ($y);
	
	## Find a page to load
	unless ($self->{page}) {
		my $page = $self->{objects}{History}->get_current
			|| $self->{home};
		$page = $self->{repository}->resolve_page($page) unless ref $page;
		$self->load_page($page);
		# FIXME what if we are read_only and page does not exist ?
		# 	load_apge returns undef
		# 	first try home, but might also not exist
		# 	need page that always exists ... like about or version page
	}
	
	## Toggle some widgets
	$self->{message_lock} = 1;
	my $set = $self->{settings};
	my $pbtype = lc $$set{pathbar_type};
	$pbtype = 'recent' unless grep {$_ eq $pbtype}
		qw/recent history namespace hidden/;
	$$set{pathbar_type} = $pbtype;
	$self->actions_set_active(
		TToolBar   => $$set{toolbar_vis},
		TStatusBar => $$set{statusbar_vis},
		TPane      => $$set{pane_vis},
		TCalendar  => $$set{show_cal},
	);
	$self->TPathBar($pbtype);
	$self->{message_lock} = 0;

	$self->{objects}{PageView}->grab_focus;

	## set save interval
	$self->{_save_timeout} = 
		Glib::Timeout->add(5000, sub {$self->save_page; 1})
		unless $self->{settings}{read_only};
	
	$self->signal_emit('gui_show');
}


sub DESTROY {
	my $self = shift;
	for (qw/_save_timeout _message_timeout/) {
		my $timeout = $self->{$_};
		Glib::Source->remove($timeout) if $timeout >= 0;
	}
}

sub AUTOLOAD {
	my $self = shift;
	my $class = ref $self;
	$AUTOLOAD =~ s/^.*:://;
	
	return if $AUTOLOAD eq 'DESTROY';
	croak "No such method: $class::$AUTOLOAD" unless ref $self;
	#warn join ' ', "Zim::AUTOLOAD called for $AUTOLOAD by: ", caller, "\n";
	
	if (exists $self->{objects}{$AUTOLOAD}) {
		return $self->{objects}{$AUTOLOAD};
	}
	elsif ($AUTOLOAD =~ /^on_(\w+)$/) { # could be an action handler
		no strict 'refs';
		goto &{"Zim::GUI::Component::$AUTOLOAD"}; # call parent
	}
	elsif (exists $self->{autoload}{$AUTOLOAD}) {
		# Delayed initialization of components
		#warn "Autoloading: $AUTOLOAD\n";
		my $obj = delete $self->{autoload}{$AUTOLOAD};
		eval {
			if (ref($obj) eq 'CODE') { $obj = $obj->(app => $self) }
			else { # expect $load to be a class
				eval "require $obj";
				die $@ if $@;
				$obj = $obj->new(app => $self);
			}
		};
		$self->exit_error($@) if $@;
		$self->{objects}{$AUTOLOAD} = $obj;
		return $obj
	}
	
	# Call method on default widget
	croak "No such method: $class::$AUTOLOAD"
		unless $self->{window}->can($AUTOLOAD);
	return $self->{window}->$AUTOLOAD(@_);
}

=item C<set_repository(REPOSITORY)>

Set the repository object.

=cut

sub set_repository {
	my ($self, $repository) = @_;
	
	my $dir = $repository->{dir}; # FIXME what if there is no dir
	$self->{name} ||= $repository->config->{name};
	unless ($self->{name}) { # needed for trayicon tooltip
		my @dirs = grep length($_),
			File::Spec->splitdir($dir);
		$self->{name} = pop @dirs;
	}
	$repository->config->{name} = $self->{name};

	if (my $icon = $repository->config->{icon}) {
		my $file = File::Spec->rel2abs($icon, $dir);
		($file) = xdg_data_files('pixmaps', $icon) unless -f $file;
		$self->{icon_file} = $file if defined $file and -f $file;
	}

	$self->{home} = $repository->config->{home} || $self->{settings}{default_home};
	$repository->config->{home} = $self->{home};

	$self->{repository} = $repository;
	$repository->{date_string} = $self->{settings}{date_string};
		# Tmp hack till more template support is written
}

sub on_delete_event {
	# Called when the window is deleted
	# return true to avoid destruction
	my $self = shift;
	my $force = uc(shift) eq 'FORCE';

	eval {
		my ($settings, $window) = @{$self}{'settings', 'window'};
		@{$settings}{'width', 'height'} = $window->get_size;
		@{$settings}{'x', 'y'} = $window->get_position if $window->visible;
			# position is 0,0 when window is hidden
	};
	
	if ($self->{settings}{hide_on_delete} and !$force) {
		$self->{window}->hide;
		return 1;
	}
	
	$self->{save_lock} = 0; # just to be sure
	$self->save_page() || return 1;
	$self->save_config;

	unless ($self->{settings}{read_only}) {
		my ($vol, $dirs, undef) =
			File::Spec->splitpath($self->{settings}{hist_file});
		my $dir = File::Spec->catpath($vol, $dirs);
		$self->{objects}{History}->write if -d $dir;
			# Only safe history if directory exists ..
	}

	return 0;
}

=item C<current()>

=cut

sub current { return $CURRENT }

=item C<plug(PLUGIN)>

=item C<unplug(PLUGIN)>

=cut

sub plug { # TODO add plugin to config
	my ($self, $name) = @_;
	return $self->error_dialog("Plugin already loaded: $name")
		if $self->{objects}{$name};
		
	my ($script) = xdg_data_files('zim', 'plugins', $name.'.pl');
	unless (length $script) {
		$self->unplug($name);
		return $self->error_dialog("No such plugin: $name");
	}

	#warn "Loading plugin $name from $script\n";
	local $CURRENT = $self;
	my $return = do $script;
	if ($@ or ! defined $return) {
		$self->unplug($name);
		return $self->error_dialog("Failed to load $name plugin", $@ || $!);
	}

	return 1;
}

sub unplug {
	my ($self, $name) = @_;
	$self->{settings}{plugins} =~ s/\Q$name\E,*//;
	$self->del_object($name);
}

=item C<< add_object(NAME => OBJECT) >>

Add a child object to the application object.
To access this object later you can use NAME as a method on the application
object. By convention object names should be written in CamelCase to avoid
conflict with normal methods. Be aware that actions also use CamelCase though.

OBJECT can either be an object reference, a class name or a code reference.
In case of a class reference the object is autoloaded with
C<< CLASS->new(app => $app) >>, where C<$app> is the main application obejct.
In case of a code reference this code is expected to return an object reference
when we try to autoload the object.

=item C<del_object(NAME)>

Remove a child object from the application object.
Does not necessarily destroy the child object.

=cut

sub add_object {
	my ($self, $name, $obj) = @_;
	croak "Object '$name' already exists"
		if exists $self->{objects}{$name}
		or exists $self->{autoload}{$name} ;
	croak "Object '$name' conflicts with action name"
		if $self->can($name);
	if (! ref $obj or ref($obj) eq 'CODE') {
		$self->{autoload}{$name} = $obj;
	}
	else {
		$self->{objects}{$name} = $obj;
	}
	return 1;
}

sub del_object {
	delete ${$_[0]{autoload}}{$_[1]};
	return delete ${$_[0]{objects}}{$_[1]};
}

sub on_click_backlinks {
	my ($self, $event) = @_;
	return if $event->type ne 'button-press';
	
	if ($event->button == 1) {
		$self->SearchDialog->{dialog}
			? $self->SearchDialog->hide
			: $self->SearchBL  ;
	}
	return unless $event->button == 3;
	
	my @blinks = $self->{page}->_list_backlinks;
	return unless @blinks;
	
	my $menu = Gtk2::Menu->new();
	for my $l (@blinks) {
		my $item = Gtk2::ImageMenuItem->new_with_label($l);
		$item->signal_connect(activate =>
			sub { $self->link_clicked($l) }  );
		$menu->add($item);
	}

	$menu->show_all;
	
	my ($button, $time) = $event
		? ($event->button, $event->time)
		: (0, 0) ;
	$menu->popup(undef, undef, undef, undef, $button, $time);
}

=item C<widget()>

Returns the root window widget.
Use this widget for things like show_all() and hide_all().

=cut

sub widget { return $_[0]->{window} }

=item C<signal_connect(NAME, CODE, DATA)>

=cut

sub signal_connect {
	my ($self, $signal, $code, @data) = @_;
	$self->{_signals}{$signal} ||= [];
	push @{$self->{_signals}{$signal}}, [$code, @data];
}

=item C<signal_emit(NAME, ARGS)>

=cut

sub signal_emit {
	my ($self, $signal, @args) = @_;
	return unless exists $self->{_signals}{$signal};
	for my $handler (grep defined($_), @{$self->{_signals}{$signal}}) {
		my ($code, @data) = @$handler;
		eval { $code->($self, @args, @data) };
		warn if $@;
	}
}

=item C<load_config>

Read config file.

=cut

sub load_config {
	my $self = shift;
	my $profile = $self->{settings}{read_only} ? 'ro' : 'rw' ;
	my ($file) = xdg_config_files('zim', 'conf.'.$profile);
	return unless defined $file;
	
	my $conf = Zim::File->new($file)->read_config();
	for (keys %$conf) {
		next unless length $$conf{$_};
		$self->{settings}{$_} = $$conf{$_};
	}

	my $accelmap = File::Spec->catfile(
		xdg_config_home, 'zim', 'accelmap.'.$profile );
	Gtk2::AccelMap->load($accelmap) if -e $accelmap;
}

=item C<save_config>

Save config file.

=cut

sub save_config {
	my $self = shift;
	$self->{settings}{pane_pos} = $self->{hpaned}->get_position;
	
	my $profile = $self->{settings}{read_only} ? 'ro' : 'rw' ;

	my %conf = %{$self->{settings}};
	Zim::File->new(xdg_config_home, 'zim', 'conf.'.$profile)->write_config(\%conf);

	my $accelmap = File::Spec->catfile(xdg_config_home, 'zim', 'accelmap.'.$profile);
	Gtk2::AccelMap->save($accelmap);
}

=item C<link_clicked(LINK)>

Loads a page in zim or opens an external url in a browser.

LINK is considered to be either an page name, an url or a file name.
Page names are resolved as relative links first.
Dispatches to C<open_file()> or C<open_url()> when
LINK is a file or an url.

=cut

sub link_clicked {
	my ($self, $link, $fallback) = @_;
	#warn "link clicked: >>$link<<\n";
	return warn "Warning: You tried to folow an empty link.\n"
		unless length $link;

	my ($type, $l) = $self->{page}->parse_link($link);
	return $self->error_dialog("Link undefined: $link")
	       unless defined $type;
	
	if ($type eq 'page') {
		my $page = $self->{page}->resolve_link($l);
		$page = $self->{repository}->resolve_page($l)
			if $fallback and ! $page ;
		return $self->error_dialog("Page does not exist:\n$link")
			unless $page;
		$self->load_page($page);
	}
	elsif ($type eq 'file') {
		my $f = Zim::File->localize($l);
		$self->open_file($f);
	}
	elsif ($type eq 'man') { $self->ShowHelp(':man:'.$l) }
	else                   { $self->open_url($l)          }
}

=item C<open_file(FILE)>

Opens FILE with the apropriate program.
Calls C<open_directory()> when FILE is a directory.

=cut

sub open_file {
	my ($self, $file) = @_;
	return $self->open_directory($file) if $file =~ m#/$# or -d $file;
	my $mt = mimetype($file);
	
	return $self->open_directory($file, 'OPEN_FILE')
		unless $mt and $has_mimeapplications;
	
	my ($app) = mime_applications($mt);

	return $self->open_directory($file, 'OPEN_FILE') unless $app;

	unless (fork) { # child process
		warn 'Executing '.$app->get_value('Name')." on $file\n";
		eval { $app->exec($file) };
		exit 1;
	}
}

=item C<open_directory(DIR)>

Opens DIR in the file browser. Open the parent directory when
DIR turns out to be a file.

=cut

sub open_directory {
	my ($self, $dir, $open_file) = @_; # open_file is a private arg
	
	my $browser = $self->{settings}{file_browser};
	unless ($browser) { # prompt for a browser
		my $val = $self->run_prompt(
			'Choose file browser',
			['cmd'], {cmd => ['Command', 'string', $DEFAULTS{file_browser}]},
			undef, undef,
			'Please enter a file browser' );
		($browser) = @$val;
		$browser = undef unless $browser =~ /\S/;
		return $self->error_dialog('You have no browser configured')
			unless defined $browser;
		$self->{settings}{file_browser} = $browser;
	}
	
	if ( ! -d $dir and (! $open_file or $browser =~ /\%d/) ) {
		# strip filename if $dir is a file
		# we default to %f is $open_file
		my ($vol, $dirs, undef) = File::Spec->splitpath($dir);
		$dir = File::Spec->catpath($vol, $dirs);
	}
	
	$browser =~ s/\%[sfd]/$dir/ or $browser .= " '$dir'";
	$self->_exec($browser);
}

=item C<open_url(URL)>

Opens URL in the web browser.

=cut

sub open_url {
	my ($self, $url) = @_;
	
	$url =~ /^(\w[\w\+\-\.]+):/
		or return $self->error_dialog("Not an url: $url");
	my $proto = $1;
	
	if ($proto eq 'zim') { # special case
		$url =~ s#\?(.*)$##;
		my $page = $1;
		$url =~ s#^zim:#file:#i;
		my $dir = Zim::File->parse_uri($url);
		return $self->exec_new_window($dir, $page);
	}
	
	my ($app, $title, $string, $error) = ($proto eq 'mailto')
		? (
			'email_client',
			'Choose Email Client',
			'Please enter an email client',
			'You have no email client configured'
		) : (
			'browser',
			'Choose Browser',
			'Please enter a browser to open external links',
			'You have no browser configured'
		) ;

	my $browser = $self->{settings}{$app};
	$browser ||= $ENV{BROWSER} unless $proto eq 'mailto';
	unless ($browser) { # prompt for a browser
		my $val = $self->run_prompt(
			$title,
			['cmd'], {cmd => ['Command', 'string', $DEFAULTS{$app}]},
			undef, undef,
			$string );
		($browser) = @$val;
		$browser = undef unless $browser =~ /\S/;
		return $self->error_dialog($error)
			unless defined $browser;
		$self->{settings}{$app} = $browser;
	}
	
	$browser =~ s/\%[us]/$url/ or $browser .= " '$url'";
	$self->_exec($browser);
}

sub _exec {
	my ($self, @args) = @_;
	warn "Executing: @args\n";
	unless (fork) { # child process
		exec @args;
		exit 1; # just to be sure
	}
}

=item C<load_page(PAGE)>

Loads a new page in the PageView, updates history etc. when
necessary. PAGE should be either an absolute page name, a history record
or a page object.

use C<link_clicked()> for relative page names, urls etc.

=cut

sub load_page {
	my ($self, $page) = @_;
	
	if ($self->{page}) {
		$self->{save_lock} = 0; # just to be sure
		$self->save_page || return;
	}

	goto &_load_page;
}

sub _load_page { # load _without_ saving first
	my ($self, $page) = @_;
	
	my $prevpage = $self->{page};
	
	my ($rec, $name);
	($name, $page, $rec) = 
		(ref($page) eq 'HASH') ? ($page->{name}, undef, $page) :
		ref($page)             ? ($page->name,   $page, undef) : ($page, undef, undef) ;
	my $from_hist = defined $rec;
	
	unless ($page) { # get object
		return warn "Warning: You tried to load an empty name.\n" unless length $name;
		
		eval { $page = $self->{repository}->get_page($name) };
		return $self->error_dialog("Could not load page $name\n\n$@", $@) if $@;
	}
	return $self->error_dialog("Page does not exist:\n$page")
		if ! defined $page
		or $self->{settings}{read_only} and ! $page->exists ;
	
	unless ($rec) { # get history record
		$rec = $self->{objects}{History}->get_record($name);
	}
	
	$self->{objects}{History}->set_current($page, $from_hist);
	eval { $self->{objects}{PageView}->load_page($page) };
	if ($@) {
		$self->exit_error("Could not load page $name\n\n$@", $@);
		# TODO handle more gratiously .. needs read_only per page
	}
	
	$self->{page} = $page;
	$self->{objects}{PageView}->set_state(%$rec) if defined $rec;
	$self->{objects}{History}->delete_recent($prevpage->name)
		unless !defined $prevpage or $prevpage->exists;
	$self->signal_emit('page_loaded', $name);
	
	$self->{window}->set_title("$name - Zim");
	$self->update_status();
	my $links = scalar $page->_list_backlinks;
	$self->{statusl}->set_text_with_mnemonic("  $links _Back links");
}

=item C<load_date(DAY, MONTH, YEAR)>

Load the zim page coresponding to this date.

=cut

sub load_date {
	my ($self, $day, $month, $year) = @_;
	#warn "User selected: $day, $month, $year\n";
	$year -= 1900;
	my $name = strftime('%Y_%m_%d', 0, 0, 0, $day, $month, $year);
	$name = $self->{settings}{cal_namespace}.$name;
	my $page = $self->{repository}->resolve_page($name);
	$self->load_page($page);
}

=item C<go_recent(NAME)>

Go to the recent page NAME. This is almost identical to C<load_page(NAME)>
but does not change the recent pages stack.

=cut

sub go_recent {
	my ($self, $name) = @_;
	my $rec = $self->{objects}{History}->get_record($name);
	$self->load_page($rec || $name);
}

=back

=head2 Actions

The following methods map directly to the ui actions in the menu-
and toolbars.

=over 4

=item C<NewPage(PAGE)>

Open a dialog which allows you to enter a name for a new page.
PAGE is used to fill in the namespace in the dialog.

=cut

sub NewPage {
	my ($self, $ns) = @_;

	if (defined $ns) { $ns =~ s/[^:]+$// }
	else { $ns = $self->{page}->namespace }
	$ns = '' if $ns eq ':';

	my $values = $self->run_prompt(
		'New page',
		['page'], {page => ['Page name', 'page', $ns]},
		'gtk-new', undef,
		"Note that linking to a non-existing page\nalso automatically creates a new page.\n"
	);
	return unless $values;
	my ($page) = @$values;
	return unless $page =~ /\S/;

	$self->link_clicked($page);
}

=item C<OpenRep(KEY)>

Open another repository. KEY can either be a file name or a repository name.
Without KEY the "open repository" dialog is prompted to the user.

=cut

sub OpenRep {
	my ($self, $key) = @_;
	unless (defined $key) {
		$self->prompt_repository_dialog(
			sub { $self->exec_new_window('--name', @_) } );
	}
	elsif ($key =~ '/' or -e $key) {
		$self->exec_new_window($key);
	}
	else {
		my $file = Zim::Repository->lookup_by_name($key);
		return $self->error_dialog("No such repository: $key")
			unless defined $file;
		$self->exec_new_window($file);
	}
}

=item C<Save()>

Force saving the current page.

=cut

sub Save {
	my $self = shift;
	$self->save_page('FORCE');
	$self->save_config;
	$self->History->write;
}

=item C<save_page(FORCE)>

Check if the current page needs to be saved.
If FORCE is true the page is saved whether it was modified or not.

=cut

# TODO split out to save_page_if_modified() and save_page()
# save_page_if_modified(FORCE) overrules save_lock
# save_page() just saves it, no questions asked
# (maybe call the first one check_save_page() ?)

# TODO part of this logic belongs in the PageView component

sub save_page {
	my ($self, $force) = @_;
	my $modified = $self->{objects}{PageView}->modified;
	return 1 if $self->{app}{settings}{read_only}; # not modified, so save is ok
	unless ($force) {
		return 1 if ! $modified;
		return 0 if $self->{save_lock};
	}
	
	$self->{save_lock} = 1; # Used to block autosave while "could not save" dialog
	
	my $page = $self->{page};
	my $tree = $self->{objects}{PageView}->get_parse_tree;
	
	eval { # FIXME maybe this logic belongs in PageView
		my $existed = $page->exists;
		if (@$tree > 2) { # content
			$page->set_parse_tree($tree);
			$self->signal_emit('page_created', $page->name) unless $existed;
		}
		else {
			$page->delete;
			$self->signal_emit('page_deleted', $page->name);
		}
	};
	unless ($@) {
		$self->{objects}{PageView}->modified(0);
		$self->signal_emit('page_saved', $page->name);
	}
	else {
		my $error = $@;
		my $answer = $self->prompt_question(
			'Could not save - Zim', 'error',
			"<b>Could not save page</b>\n\n$error",
			['discard', undef, '_Discard changes'],
			['copy', 'gtk-save-as', '_Save a copy...' ],
			2 );
		if ($answer eq 'discard') {
			$self->_load_page($self->{page}); # load page wihtout saving
		}
		elsif ($answer eq 'copy') {
			return $self->SaveCopy; # recurs inderectly
		}
		else { return 0 }
	}
		
	$self->update_status();
	$self->{save_lock} = 0;
	return 1;
}

=item C<SaveCopy()>

=cut

sub SaveCopy {
	my $self = shift;
	return unless $self->{page};

	# prompt page
	my $new = $self->{page}->name . '_(Copy)' ;
	my $val = $self->run_prompt(
		'Save Copy - Zim',
		['page'], {page => ['Page', 'page', $new]},
		'gtk-save-as', '_Save Copy',
		'Please enter a name to save' );
	return unless defined $val;
	($new) = @$val;
	return unless length $new;

	my $page = $self->{page}->resolve_link($new);
	
	# check if page exists
	if ($page->exists()) {
		my $answer = $self->prompt_question(
			'Page exists - Zim', 'warning',
			"Page '$new'\nalready exists.",
			['cancel', 'gtk-cancel', undef],
			['overwrite', undef, '_Overwrite'] );
		return unless $answer eq 'overwrite';
	}
	
	# change page and save buffer
	$self->{page} = $page;
	$self->Save;
	$self->load_page($page->name); # trick into reloading
}

=item C<Export()>

=cut

sub Export { pop->ExportDialog->show() }

=item C<EmailPage()>

Open the current page in the email client.

=cut

sub EmailPage {
	# TODO do _not_ use the get_source interface here,
	# let PageView convert to plain text !
	# Using this interface here may screw up mtime detection
	# FIXME call save() first here
	my $self = shift;
	my $subject = $self->{page}->name;
	my $io = $self->{page}->get_source('r');
	my $src = join '', <$io>;
	$src =~ s/(\W)/sprintf '%%%02x', ord $1/eg; # url encoding
	$self->open_url("mailto:?subject=$subject&body=".$src);
}

=item C<RenamePage($FROM, $TO, $UPDATE)>

Wrapper for C<< Zim::Repository->move_page() >>.

Move page from FROM to TO. If TO is undefined a dialog is shown to ask for a
page name. UPDATE is a boolean telling to update all links to this page.

Without arguments prompts the user for input.

=cut

sub RenamePage {
	my ($self, $from, $to, $update) = @_;

	$from = $self->{page}->name unless defined $from;
	
	my $move_current = $self->{page}->equals($from);
	if ($move_current) { # FIXME clearer interface for this
		$self->{save_lock} = 0 if $move_current;
		$self->save_page or return;
	}
	else { $self->save_page }
	
	unless (defined $to) {
		($to, $update) = $self->prompt_rename_page_dialog($from);
		return unless defined $to;
		$to = $self->{repository}->resolve_page($to);
		# TODO check for things like ':'
	}

	# Get backlinks
	my $rfrom = $move_current ? $self->{page} : $self->{repository}->get_page($from); # FIXME ugly lookup
	my @backlinks = $rfrom->_list_backlinks();
	
	# Move page
	#warn "Move '$from' to '$to'\n";
	my $page;
	eval { $page = $self->{repository}->move_page($from, $to) };
	return $self->error_dialog("Could not rename $from to $to\n\n$@") if $@;

	if ($update) { # Update backlinks
		# FIXME - for case-sensitive move ("test" => "Test") do not update links - let the repository tell us what to do
		my ($dialog, $bar, $label) = $self->_new_progress_bar(
			"Updating links", 'Updating..' );
		my $continue = 1;
		$dialog->signal_connect(response => sub {$continue = 0});
		my $i = 1;
		my $callback = sub {
			my $page = shift;
			$bar->pulse;
			$label->set_text('Updating links in ' . $page);
			while (Gtk2->events_pending) { Gtk2->main_iteration_do(0) }
			return $continue;
		};
		eval {
			for (@backlinks) {
				$callback->($_) or last;
				$self->{repository}->get_page($_)->_update_links(
					'page', $from => $to );
			}
		};
		$self->error_dialog($@) if $@;
		$dialog->destroy;
	}
	
	($from, $to) = map {ref($_) ? $_->name : $_} $from, $page;
	warn "moved $from => $to\n";
	$self->signal_emit('page_renamed', $from => $to);
	$self->load_page($to) if $move_current;
	
	$self->update_status;
	return 1;
}

=item C<DeletePage(PAGE)>

Wrapper for C<< Zim::Repository->delete_page >>.
Asks the user for confirmation.

If PAGE is undefined the current page is deleted.

=cut

sub DeletePage {
	# FIXME option to delete a complete sub-tree
	# FIXME integrate with the save_page() logic, now we have 
	# redundant code here
	my ($self, $page) = @_;
	$page = $self->{page}->name unless defined $page;
	
	my $name = ref($page) ? $page->name : $page;
	my $delete_current = ($name eq $self->{page}->name);
	$self->save_page; # Mind you, overriding save lock could lock us here
	
	my $answer = $self->prompt_question(
		'Delete page - Zim', 'warning',
		"Are you sure you want to\ndelete '$name' ?",
		['no',  'gtk-cancel', undef],
		['yes', 'gtk-delete', undef]  );
	return unless $answer eq 'yes';
	
	$page = $self->{page} if $delete_current; # make sure we have the right object
	eval { $self->{repository}->delete_page($page) };
	return $self->error_dialog("Could not delete $name\n\n$@") if $@;

	# TODO trigger this with signals
	$self->{objects}{History}->delete_recent($name);
	$self->signal_emit('page_deleted', $name);

	if ($delete_current) {
		eval { $self->{objects}{PageView}->load_page($page) };
		if ($@) {
			$self->exit_error("Could not load page $name\n\n$@");
			# TODO handle more gratiously
			# .. switching read_only would help here
		}
	}

	$self->update_status;
}

=item C<Props()>

Show the properties dialog.

=cut

sub Props { $_[0]->PropertiesDialog->show }

=item C<Close()>

Close the application.

=item C<Quit()>

Quit the application.

=cut

sub Close {
	my $self = shift;
	$self->on_delete_event(@_) && return;
	Gtk2->main_quit;
}

sub Quit { $_[0]->Close('FORCE') }

=item C<Prefs()>

Show the preferences dialog.

=cut

sub Prefs { shift->PreferencesDialog->show }

=item C<TToolBar(BOOL)>

Toggel toolbar visibility.
If BOOL is undefined it will just toggle the current state.

=cut

sub TToolBar {
	my ($self, $show) = @_;
	$show = $self->{toolbar}->visible ? 0 : 1 unless defined $show;
	$show ? $self->{toolbar}->show_all : $self->{toolbar}->hide_all ;
	$self->{settings}{toolbar_vis} = $show;
	$self->actions_show_active(TToolBar => $show);
}

=item C<TStatusBar(BOOL)>

Toggle statusbar visibility.
If BOOL is undefined it will just toggle the current state.

=cut

sub TStatusBar {
	my ($self, $show) = @_;
	$show = $self->{statusbar}->visible ? 0 : 1 unless defined $show;
	$show ? $self->{statusbar}->show_all : $self->{statusbar}->hide_all ;
	$self->{settings}{statusbar_vis} = $show;
	$self->actions_show_active(TStatusBar => $show);
}

=item C<TPane(BOOL)>

Toggle visibility of the side pane.
If BOOL is undefined it will just toggle the current state.
If BOOL is "-1" the pane will be shown, but hidden again as soon
as a page is selected.

=cut

sub TPane {
	my ($self, $show) = @_;
	$show = $self->{l_vbox}->visible ? 0 : 1 unless defined $show;

	$self->{_pane_visible} = $show;
	my $widget = $self->{l_vbox};
	my $hpaned = $self->{hpaned};
	if ($show) {
		$hpaned->set_position($self->{settings}{pane_pos});
		my $tree_view = $self->TreeView; # possibly autoloaded here
		$widget->show_all;
		$tree_view->grab_focus;
	}
	else {
		$self->{settings}{pane_pos} = $hpaned->get_position();
		$widget->hide_all;
		$self->{objects}{PageView}->grab_focus;
	}

	$self->{settings}{pane_vis} = $show unless $show == -1;
	$self->actions_show_active(TPane => $show);
}

=item C<TPathBar(TYPE)>

Set the pathbar type to TYPE.

=cut


sub on_TPathBar {
	# get the type from the action name
	my $self = pop;
	return if $self->{_block_actions};
	my $type = lc pop->get_name;
	$type =~ s/^pb//;
	$self->TPathBar($type);
}

sub TPathBar {
	my ($self, $type) = @_;
	return warn "No pathbar type given" unless defined $type;
	
	my $path_bar = $self->{objects}{PathBar};
	if (grep {$type eq $_} qw/recent history namespace/) {
		$path_bar->widget->show_all;
		$path_bar->set_type($type);
	}
	elsif ($type eq 'hidden') {
		$path_bar->widget->hide_all;
		$path_bar->clear_items;
	}
	else { warn "unknown pathbar_type: $type\n" }

	$self->{settings}{pathbar_type} = $type;
	$self->message("Pathbar type: $type");
	$self->actions_show_active('PB'.ucfirst($type) => 1);
}

=item C<TCalendar(BOOL)>

Toggle calendar visibility.
If BOOL is undefined it will just toggle the current state.

=cut

sub TCalendar {
	my ($self, $show) = @_;
	my $cal = $self->Calendar;
	$show = $cal->visible ? 0 : 1 unless defined $show;
	$show ? $cal->show : $cal->hide ;
	$self->{settings}{show_cal} = $show;
}

=item C<Reload()>

Save and reload the current page.

=cut

sub Reload {
	my $self = shift;
	$self->load_page(
		$self->{objects}{History}->get_current );
}


=item C<Search($QUERY)>

Open QUERY in the search dialog.

=cut

sub Search { $_[0]->SearchDialog->search($_[1]) }

=item C<SearchBL()>

Open backlinks for current page in search dialog.

=cut

sub SearchBL {
	my $self = shift;
	$self->SearchDialog->search($self->{page}->name);
}

=item C<OpenDir()>

Open the dir for the current page.

=cut

sub OpenDir {
	my $self = shift;
	my $dir = $self->{page}->properties->{base};
	return unless length $dir;
	#warn "Opening $dir\n";
	$self->open_directory($dir);
}

=item C<RBIndex()>

Rebuild the index cache.

=cut

sub RBIndex {
	my $self = shift;
	$self->{repository}->_flush_cache;
	my $tree = $self->TreeView;
	$tree->{_loaded} = 0;
	$tree->load_index;
	$self->Reload;
}


=item C<GoBack($INT)>

Go back one or more steps in the history stack.

=item C<GoForward($INT)>

Go forward one or more steps in the history stack.

=cut

sub GoBack {
	my $self = shift;
	my $i = shift || 1;
	my $rec = $self->{objects}{History}->back($i) || return;
	$self->load_page($rec);
}

sub GoForward {
	my $self = shift;
	my $i = shift || 1;
	my $rec = $self->{objects}{History}->forw($i) || return;
	$self->load_page($rec);
}

=item C<GoParent()>

Go to page up in namespace.

=item C<GoChild()>

Go to page down in namespace.

=cut

sub GoParent {
	my $self = shift;
	my $namespace = $self->{page}->namespace;
	return if $namespace eq ':';
	$namespace =~ s/:+$//;
	$self->load_page($namespace);
}

sub GoChild {
	my $self = shift;
	my $namespace = $self->History->get_namespace;
	my $name = $self->{page}->name;
	return unless $namespace =~ /^(:*$name:+[^:]+)/;
	$self->load_page($1);
}

=item C<GoNext()>

Go to the next page in the index.

=item C<GoPrev()>

Go to the previous page in the index.

=cut

sub GoNext {
	my $self = shift;
	my $page = $self->{page}->get_next;
	$self->load_page($page) if $page;
}

sub GoPrev {
	my $self = shift;
	my $page = $self->{page}->get_prev;
	$self->load_page($page) if $page;
}

=item C<GoToday()>

Go to the page for todays date.

=cut

sub GoToday {
	my $self = shift;
	my ($day, $month, $year) = ( localtime )[3, 4, 5];
	$year += 1900;
	$self->load_date($day, $month, $year);
}

=item C<GoHome()>

Go to the home page.

=cut

sub GoHome {
	my $self = shift;
	my $home = $self->{repository}->resolve_page($self->{home});
	$self->load_page($home);
}

=item C<JumpTo(PAGE)>

Go to PAGE. Shows a dialog when no page is given.

=cut

sub JumpTo {
	my ($self, $page) = @_;

	unless (defined $page) {
		my $ns = $self->{page}->namespace;
		$ns = '' if $ns eq ':';
		my $values = $self->run_prompt(
			'Jump to',
			['page'], {page => ['Jump to Page', 'page', $ns]},
			'gtk-jump-to', undef, undef );
		return unless $values;
		($page) = @$values;
		return unless $page =~ /\S/;
	}

	if (ref $page) { $self->load_page($page)       }
	else           { $self->link_clicked($page, 1) }
}

=item C<ShowHelp(PAGE)>

Show a window showing the documentation. PAGE is optional.

=cut

sub ShowHelp {
	my ($self, $page) = @_;
	my @args = ('--doc', $page ? $page : ());
	$self->exec_new_window(@args);
}


=item C<About()>

This dialog tells you about the version of zim you are using.

=cut

sub About {
	# TODO use Gtk2::AboutDialog for gtk > 2.6
	my $self = shift;
	
	my $dialog = Gtk2::Dialog->new(
		'About Zim', $self->{window},
	       	[qw/modal destroy-with-parent no-separator/],
		'gtk-close' => 'close',
	);
	$dialog->set_resizable(0);
	$dialog->set_border_width(5);
	$dialog->set_icon($self->{window}->get_icon);
	my $button = $self->new_button('gtk-help', '_More');
	$dialog->add_action_widget($button, 'help');
	$dialog->set_default_response('close');
	
	$dialog->vbox->add(
		Gtk2::Image->new_from_file($ICON) );

	my $text = $LONG_VERSION;
	$text =~ s/^(.*)$/\n<b>$1<\/b>/m;
	my $label = Gtk2::Label->new();
	$label->set_markup($text);
	$label->set_justify('center');
	$dialog->vbox->add($label);
	
	$dialog->show_all;
	my $response = $dialog->run;
	$dialog->destroy;

	$self->ShowHelp('zim:about') if $response eq 'help';
}


=back

=head2 Other functions

Functions below are used by other methods.
They are not considered part of the api.

=over 4

=item C<message(STRING)>

Flash a message in the statusbar for a short time.

=cut

sub message {
	my ($self, $str) = @_;
	return if $self->{message_lock};
	my $timeout = $self->{_message_timeout};
	$self->push_status($str, 'message');
	Glib::Source->remove($timeout) if $timeout >= 0;
	$timeout = Glib::Timeout->add(2500, sub {
		$self->pop_status('message');
		$self->{_message_timeout} = -1;
		return 0; # removes timeout
	} );
	$self->{_message_timeout} = $timeout;
}

=item C<update_status()>

Sets the statusbar to display the current page name and some other
information.

=cut

sub update_status {
	my $self = shift;
	my $stat = ' '.$self->{page}->name;
	$stat .= '*' if $self->PageView->modified;
	if ($_ = $self->{page}->status()) { $stat .= '  -  '.uc($_) }
	$stat .= ' [readonly]' if $self->{settings}{read_only};
	$self->push_status($stat, 'page');
}

=item C<push_status(STRING, CONTEXT)>

Put STRING in the status bar.

=cut

sub push_status {
	my ($self, $str, $id) = @_;
	my $statusbar = $self->{status1};
	$id = $statusbar->get_context_id($id);
	$statusbar->pop($id);
	$statusbar->push($id, $str);
}

=item pop_status(CONTEXT)

Removes a string from the status bar.

=cut

sub pop_status {
	my ($self, $id) = @_;
	my $statusbar = $self->{status1};
	$id = $statusbar->get_context_id($id);
	$statusbar->pop($id);
}

=item C<new_button(STOCK, TEXT)>

Creates a button with a stock image but different text.

=cut

sub new_button {
	my ($self, $stock, $text) = @_;
	my $hbox = Gtk2::HBox->new;
	$hbox->pack_start(
		Gtk2::Image->new_from_stock($stock, 'button'), 0,0,0);
	$hbox->pack_start(
		Gtk2::Label->new_with_mnemonic($text), 1,1,0);
	my $button = Gtk2::Button->new();
	$button->add(Gtk2::Alignment->new(0.5,0.5, 0,0));
	$button->child->add($hbox);
	return $button;
}

=item C<new_prompt(TITLE, FIELDS, DATA, BUTTON_STOCK, BUTTON_TEXT, TEXT)>

Generates a dialog asking for one or more fields of input.
Returns the dialog widget and a list with Gtk2::Entry objects.

TITLE is the dialog title.

FIELDS is an array ref giving the order of the input fields.

DATA is a hash ref containing definitions of the input fields.
The key is the name used in FIELDS, the value an array ref with a label text,
a data type and a value.
At the moment only the "string", "page", "file" and "dir" data types are treated special, all other will be ignored silently.

TEXT, BUTTON_STOCK and BUTTON_TEXT are optional.

=cut

sub new_prompt {
	my ($self, $title, $fields, $data, $stock, $string, $text) = @_;
	
	## Setup dialog
	my $dialog = Gtk2::Dialog->new(
		$title, $self->{window},
		[qw/modal destroy-with-parent no-separator/],
		'gtk-cancel'  => 'cancel',
	);
	$dialog->set_resizable(0);
	#$dialog->vbox->set_border_width(12); # FIXME
	$dialog->set_icon($self->{window}->get_icon);
	
	$stock ||= 'gtk-ok';
	my $button = $string
		? $self->new_button($stock, $string)
		: Gtk2::Button->new_from_stock($stock);
	$dialog->add_action_widget($button, 'ok');
	# $dialog->set_default_response('ok'); FIXME

	if (defined $text) {
		my $label = Gtk2::Label->new($text);
		my $align = Gtk2::Alignment->new(0,0.5, 0,0);
		$align->add($label);
		$dialog->vbox->add($align);
	}
	
	## Generate table with input fields
	my $table = Gtk2::Table->new(scalar(@$fields), 2);
	$table->set_border_width(5);
	$table->set_row_spacings(5);
	$table->set_col_spacings(12);
	$dialog->vbox->add($table);
	
	my @entries;
	for my $i (0 .. $#$fields) {
		my @f = @{ $$data{$$fields[$i]} };
		
		my $label = Gtk2::Label->new($f[0].':');
		my $align = Gtk2::Alignment->new(0,0.5, 0,0);
		$align->add($label);
		
		my $entry = Gtk2::Entry->new();
		$entry->set_text($f[2]) if defined $f[2];
		$entry->signal_connect(
			activate => sub { $dialog->response('ok') } );
		push @entries, $entry;
		
		$table->attach_defaults($align, 0,1, $i,$i+1);

		if ($f[1] eq 'file' or $f[1] eq 'dir') {
			my $hbox = Gtk2::HBox->new(0,3);
			$hbox->add($entry);

			my $is_dir = ($f[1] eq 'dir');
			my $button = Gtk2::Button->new('_Browse...');
			$button->signal_connect( clicked => sub {
				my $val = $self->filechooser_dialog(
					$entry->get_text(), $is_dir );
				$entry->set_text($val);
			} );
			$hbox->pack_start($button, 0,1,0);

			$table->attach_defaults($hbox, 1,2, $i,$i+1);
		}
		elsif ($f[1] eq 'page') {
			$self->set_page_completion($entry);
			$table->attach_defaults($entry, 1,2, $i,$i+1);
		}
		else {
			$table->attach_defaults($entry, 1,2, $i,$i+1);
		}
	}
	
	$dialog->show_all;
	return $dialog, \@entries;
}

=item C<set_page_completion(ENTRY)>

Attach page completions code to a L<Gtk2::Entry> object.

=cut

sub set_page_completion {
	my ($self, $entry) = @_;
	my $completion = Gtk2::EntryCompletion->new;
	my $model = Gtk2::ListStore->new('Glib::String');
	$completion->set_model($model);
	$completion->set_text_column(0);
	$completion->set_inline_completion(1)  if Gtk2->CHECK_VERSION(2, 6, 0);
	$entry->set_completion($completion);
	$entry->signal_connect(changed => \&_update_completion, $self);
}

sub _update_completion {
	my ($entry, $self) = @_;
	my $ns = $entry->get_text;
	$ns =~ s/[^:]+$//;
	return if defined $entry->{_ns} and $entry->{_ns} eq $ns;
	$entry->{_ns} = $ns;
	
	my $_ns = length($ns)
		? $self->{repository}->resolve_namespace($ns)
		: $self->{page}->namespace() ;
		#warn "Complete namespace: $_ns\n";
	
	my $model = $entry->get_completion->get_model;
	$model->clear;
	for ($self->{repository}->list_pages($_ns)) {
		s/_/ /g;
		my $iter = $model->append();
		$model->set($iter, 0 => $ns.$_);
		#warn "Appended: $ns$_\n";
	}
}

=item C<run_prompt(..)>

Wrapper around C<new_prompt()> that runs the dialog and
returns a list with input values. Returns undef on 'cancel'.

=cut

sub run_prompt {
	my $self = shift;
	my ($dialog, $entries) = $self->new_prompt(@_);

	my $values = ($dialog->run eq 'ok')
		? [map $_->get_text, @$entries]
		: undef ;
	$dialog->destroy;

	return $values;
}



# TODO the various dialogs could be out-sourced using Autoloader
# also a lot of code could be reduced by haveing a generic
# _prompt_dialog(title => ... text => ... fields => { })

=item C<exit_error(ERROR)>

Like C<error_dialog> but exits afterwards.

=cut

sub exit_error {
	my $self = shift;
	my ($text1, $text2) = @_;
	if (defined $text1) { $self->error_dialog($text1, $text2) }
	else {
		$text2 ||= "Unknown error";
		warn "zim: $text2\n";
	}
	unlink $self->{pidfile} if defined $self->{pidfile};
	exit 1;
}

=item C<error_dialog(ERROR)>

This method is used to display errors.

=cut

sub error_dialog {
	my ($self, $text1, $text2) = @_;
	$text2 ||= $@ || $text1;
	warn "zim: $text2\n";
	my $window = $self->{window};
	$window = undef unless defined $window and $window->visible;
		# window might not yet be realized
	my $dialog = Gtk2::MessageDialog->new(
		# no markup, $@ can contain "<" symbols
		$window, 'modal', 'error', 'ok', $text1 );
		# parent, flags, type, buttons, message
	$dialog->run;
	$dialog->destroy;
	return undef;
}

=item C<prompt_rename_page_dialog>

=cut

sub prompt_rename_page_dialog {
	my ($self, $page) = @_;
	
	my ($dialog, $entries) = $self->new_prompt(
		'Rename page',
		['page'], {page => ['Rename to', 'page', $page]},
		'gtk-save', '_Rename'  );
	
	my ($entry) = @$entries;
	$entry->select_region(length($1), -1) if $page =~ /^(.+:)/;

	my $rpage = ($page eq $self->{page}->name) ? $self->{page} : $self->{repository}->get_page($page); # FIXME ugly lookup
	my $nlinks = scalar $rpage->_list_backlinks;
	#my $check = Gtk2::CheckButton->new("_Update $nlinks pages linking here");
	#if ($nlinks > 0) { $check->set_active(1)    }
	#else             { $check->set_sensitive(0) }
	#$check->show;
	#$dialog->vbox->add($check);
	
	if ($dialog->run eq 'ok') { $page = $entry->get_text }
	else                      { $page = undef            }
	my $update = 0; #$check->get_active;
	$dialog->destroy;

	$page = undef unless $page =~ /\S/;
	return defined($page) ? ($page, $update) : ();
}

=item C<prompt_question(TITLE, TYPE, TEXT, BUTTONS ..., TIME)>

Runs a dialog displaying TEXT

BUTTONS is a list of array references, each containing a name, a stock item
name and/or text. The id of the button that was pressed is returned.

TYPE can either be 'error', 'warning', 'question', 'info' or C<undef>.

TIME is an optional argument, it gives a timeout in seconds. This is used
for popups that can popup while the user is typing to prevent accidental
triggering of a accelerator.

=cut

sub prompt_question {
	my ($self, $title, $type, $text, @buttons) = @_;
	my $time = pop @buttons unless ref $buttons[-1];
	
	my $dialog = Gtk2::Dialog->new(
		$title, $self->{window},
	       	[qw/modal destroy-with-parent no-separator/],
	);
	$dialog->set_resizable(0);
	$dialog->set_icon($self->{window}->get_icon);

	my @button_widgets;
	for (0 .. $#buttons) {
		my ($id, $stock, $string) = @{$buttons[$_]};
		my $button = (defined($stock) && ! defined($string))
			? Gtk2::Button->new_from_stock($stock)
			: $self->new_button($stock, $string)   ;
		$button->set_sensitive(0);
		$dialog->add_action_widget($button, $_);
		push @button_widgets, $button;
	}
	
	my $hbox = Gtk2::HBox->new(0,12);
	$hbox->set_border_width(12);
	$dialog->vbox->pack_start($hbox, 0,0,0);

	if (defined $type) {
		my $image = Gtk2::Image->new_from_stock(
			"gtk-dialog-$type", 'dialog' );
		$image->set_alignment(0.0, 0.5); # valign=top
		$hbox->pack_start($image, 0,0,0);
	}
	if (defined $text) {
		my $label = Gtk2::Label->new($text);
		$label->set_use_markup(1);
		$label->set_selectable(1);
		$label->set_alignment(0.0, 0.0); # align left top corner
		$hbox->add($label);
	}

	$dialog->show_all;
	if ($time) {
		Glib::Timeout->add( $time*1000,
			sub { $_->set_sensitive(1) for @button_widgets; 0 } );
	}
	else { $_->set_sensitive(1) for @button_widgets }
	my $id = $dialog->run;
	$dialog->destroy;
	
	return $buttons[$id][0];
}

=item  C<exec_new_window(..)>

Executes a new process for $0, this gives a detached window.
Any arguments are passed on to the new process.

=cut

sub exec_new_window {
	my ($self, @args) = @_;
	
	if (ref($self->{window}) !~ /Dialog/) {
		@{$self->{settings}}{'width', 'height'} = $self->{window}->get_size;
	}

	$self->_exec($^X, $0, @args);
}

=item C<filechooser_dialog(FILE)>

Ask the user for a filename. FILE is the suggested filename.

=cut

sub filechooser_dialog {
	my ($self, $file, $dir, $title) = @_;
	
	my $dialog;
	$title ||= $dir ? 'Select Folder' : 'Select File' ;
	# if (Gtk2->CHECK_VERSION(2, 4, 0) and $Gtk2::VERSION >= 1.040) {
	$dialog = Gtk2::FileChooserDialog->new(
		$title, $self->{window}, 'open',
		'gtk-cancel' => 'cancel',
		'gtk-ok'     => 'ok'
	);
	# }
	#else { # old & ugly interface
	#	$dialog = Gtk2::FileSelection->new($title);
	#}
	$dialog->set_icon($self->{window}->get_icon) if $self->{window};
	$dialog->set_action('select-folder') if $dir;
	if (defined $file) {
		$file =~ s#^file:(//localhost)?/+#/#;
		$dialog->set_filename(File::Spec->rel2abs($file));
	}
	elsif (defined $self->{page}) {
		my $dir = $self->{page}->properties->{base};
		if (defined $dir) {
			$dir =~ s#^file:(//localhost)?/+#/#;
			$dialog->set_current_folder(File::Spec->rel2abs($dir));
		}
	}
	$dialog->signal_connect('response', sub {
		$file = $_[1] eq 'ok' ? $dialog->get_filename : undef;
		$dialog->destroy;
	} );
	$dialog->run;

	return $file;
}

=item C<prompt_repository_dialog(CALLBACK)>

This dialog does not run "modal" so it needs a callback because the
calling code can not wait for it to return.

=cut

sub prompt_repository_dialog { # TODO improve visual layout
	my $self = shift;
	my $callback = shift;

	#warn "window: $self->{window}\n";
	my $dialog = Gtk2::Dialog->new(
		"Open repository - Zim", $self->{window},
	       	[qw/destroy-with-parent no-separator/],
		'gtk-help'   => 'help',
		'gtk-cancel' => 'cancel',
		'gtk-open'   => 'ok',
	);
	#$dialog->set_type_hint('utility');
	$dialog->set_default_size(100,250);
	$dialog->set_border_width(5);
	$dialog->set_icon(
		Gtk2::Gdk::Pixbuf->new_from_file($ICON) );
	$dialog->set_default_response('ok');
	$self->{window} ||= $dialog; # parent might not yet be initialized
	
	$dialog->vbox->set_spacing(5);
	$dialog->vbox->pack_start(
		Gtk2::Label->new('Please choose a repository'), 0,1,0 );
	
	my $hbox = Gtk2::HBox->new(0, 12);
	$dialog->vbox->add($hbox);
	
	my $list = Gtk2::SimpleList->new('Repository' => 'text');
	$list->set_headers_visible(0);
	$list->get_selection->set_mode('browse');
	$list->signal_connect(row_activated => sub { $dialog->response('ok') });
	#$dialog->vbox->pack_start($list, 1,1,0);
	$hbox->add($list);
	
	# load list
	my ($source, $order) = ({}, []);
	my ($file) = xdg_config_files('zim', 'repositories.list');
	if (defined $file) {
		$file = Zim::File->new($file);
		($source, $order) = $file->read_config;
		@{$list->{data}} = @$order;
	}
	
	#my $hbox = Gtk2::HBox->new(1,10);
	#my $add_button = Gtk2::Button->new_from_stock('gtk-add');
	#my $rem_button = Gtk2::Button->new_from_stock('gtk-remove');
	#$hbox->add($add_button);
	#$hbox->add($rem_button);
	#$dialog->vbox->pack_start($hbox, 0,1,0);
	my $vbox = Gtk2::VButtonBox->new();
	$vbox->set_layout('start');
	$hbox->pack_start($vbox, 0,0,0);
	
	my @buttons = map Gtk2::Button->new_from_stock($_),
		qw/gtk-add gtk-remove/;
	splice @buttons, 1, 0, $self->new_button('gtk-properties', 'Change');
	$vbox->add($_) for @buttons;
	
	my $modified = 0;
	my $modify = sub {
		my ($i, $name, $dir);
		if (pop @_) { #boolean user data
			($i) = $list->get_selected_indices;
			($name) = @{$list->{data}[$i]};
			$dir = $$source{$name};
		}
		else { $i = scalar @{$list->{data}} }
		($name, $dir) = $self->prompt_new_repository_dialog($name, $dir);
		return unless defined $name;
		$$source{$name} = $dir;
		splice @{$list->{data}}, $i, 1, $name;
		$modified = 1;
		$list->select($i);
	};
	$buttons[0]->signal_connect(clicked => $modify, 0);
	$buttons[1]->signal_connect(clicked => $modify, 1);
	$buttons[2]->signal_connect(clicked => sub {
			my ($i) = $list->get_selected_indices;
			splice @{$list->{data}}, $i, 1;
			$modified = 1;
		} );
	
	$dialog->signal_connect(response => sub {
		my ($dialog, $response) = @_;
		
		# Save list
		if ($modified) {
			@$order = map $$_[0], @{$list->{data}};
			$file = Zim::File->new(
				xdg_config_home, 'zim', 'repositories.list' );
			$file->write_config($source, $order);
		}

		if ($response eq 'help') {
			return $self->ShowHelp('zim:usage:repositories');
		}
		elsif ($response ne 'ok') {
			$dialog->destroy;
			Gtk2->main_quit if $self->{window} eq $dialog;
			return;
		}

		my ($i) = (@{$list->{data}} == 1)
			? (0) : ($list->get_selected_indices);
		return $self->error_dialog("Please select a repository first.\n")
			unless defined $i;
	
		my ($name) = @{$list->{data}[$i]};
		my $src = Zim::File->abs_path($$source{$name}, $ENV{HOME});
		$callback->($name, $src);
		$dialog->destroy;
	} );
	
	$dialog->show_all;
}

=item C<prompt_new_repository_dialog(WINDOW)>

=cut

sub prompt_new_repository_dialog { # TODO add "browse" button for dir
	my $self = shift;
	my $val = $self->run_prompt(
		'New repository - Zim',
		['name', 'dir'], {
			name => ['Name', 'string', $_[0]],
			dir  => ['Directory', 'dir', $_[1]],
		}, undef, undef,
		'Please give a directory to store your pages' )
		or return undef;
	
	my ($name, $dir) = @$val;
	return undef unless $dir =~ /\S/;
	$name = $dir unless $name =~ /\S/;
	
	return ($name, $dir);
}

sub _new_progress_bar {
	my ($self, $title, $label) = @_;
	my $dialog = Gtk2::Dialog->new(
		$title, $self->{window},
	   	[qw/destroy-with-parent no-separator/],
		'gtk-cancel' => 'cancel',
	);
	$dialog->set_resizable(0);
	$dialog->vbox->set_spacing(5);
	$dialog->vbox->set_border_width(10);
	$label = Gtk2::Label->new($label);
	$dialog->vbox->add($label);
	my $bar = Gtk2::ProgressBar->new;
	$dialog->vbox->add($bar);
	$dialog->show_all;
	return ($dialog, $bar, $label);
}

1;

__END__

=back

=head1 BUGS

Please mail the author if you find any bugs.

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

L<zim>(1),
L<Zim::GUI::Component>,
L<Zim::Repository>

=cut

