package Zim::Selection;

use strict;

our $VERSION = '0.16';

=head1 NAME

Zim::Selection - selection of pages

=head1 SYNOPSIS

FIXME simple code example

=head1 DESCRIPTION

This class is used to manage selections of pages.
The pages are still asociates with a repository, this object can
be used to call operations on a sub-set of a repository.

=head1 METHODS

=over 4

=item C<new($REPOSITORY, \%OPT, @PAGES)>

Constructor. REPOSITORY should be an object of class L<Zim::Repository>
and is the object used to look up the pages in the selection.

The OPT hash can contain several options. Curently only "recurse" and
"resolve" are recognized. If "recurse" is set for all pages the corresponding
namespace is also added. If "resolve" is set the page list is treated as
user input.

The list of PAGES can contain both pages and namespaces. Namespaces are always
added recursively.

=cut

sub new {
	my ($class, $rep, $opt, @pages) = @_;
	my $self = bless {rep => $rep}, $class;
	@pages = grep /\S/, map {
		/:$/ ? $self->{rep}->resolve_namespace($_)
		     : $self->{rep}->resolve_page($_)->name ;
	} @pages if $$opt{resolve} ;
	@pages = map {
		/:$/ ? ($_) : ($_, $_.':')
	} @pages if $$opt{recurse} ;
	$self->{list} = \@pages;
	warn "Selection: @pages\n";
	return $self;
}

=item C<foreach(\&CODE, \@ARGS)>

The code reference CODE is called for each page with as
arguments ARGS and the page object.

=cut

sub foreach {
	my ($self, $code, $args) = @_;
	for (@{$self->{list}}) {
		if (/:$/) { $self->_recurse($_, $code, $args) }
		else {
			my $page = $self->{rep}->get_page($_);
			next unless $page;
			$code->(($args ? (@$args) : ()), $page);
		}
	}
}

sub _recurse { # dispatch a foreach call to a namespace
	my ($self, $ns, $code, $args) = @_;
	warn "Recursing for namespace: $ns\n";
	for my $name ($self->{rep}->list_pages($ns)) {
		my $r = ($name =~ s/:+$//);
		my $page = $self->{rep}->get_page($ns.$name);
		$code->(($args ? (@$args) : ()), $page) if $page;
		$self->_recurse($ns.$name.':', $code, $args) if $r;
	}
}

=item C<export($TARGET)>

=item C<export(\%OPT)>

Export the pages to TARGET, which should be a repository.
When called with a hash of options, these options are used to
open the target repository.

=cut

sub export {
	my ($self, $opt) = @_;
	my $target;
	if (ref($opt) eq 'HASH') {
		die "For exporting you need to provide at least".
		    "a format, a template and an output root dir.\n"
		    if grep {! length $_} @$opt{qw/format template root/} ;
		$target = ref($$opt{root})
			? $$opt{root}
			: Zim::Repository::Files->new(
				undef, '', $$opt{root}, $$opt{format} ) ;
		$target->{_template} = $$opt{template};
			# FIXME better interface for this
			# do not confuse with the template for new pages
	}
	else { $target = $opt }
	
	my $index = '';
	my @items = (undef, undef, undef);
	my $code = sub {
		my $page = shift;
		return if defined $page and ! $page->exists;

		# convert to linked list
		push @items, $page;
		shift @items;
		$page = $items[1]; # in fact previous page
		return unless $page;
		$target->{_prev} = $items[0];
		$target->{_next} = $items[2];
		
		my $name = $page->name;
		warn "Exporting: $name\n";
		$$opt{callback}->($page) || die "Export Cancelled at page $name\n"
			if $$opt{callback};
		my $target_page = $target->get_page($name);
		$target_page->clone($page);
		$index .= $name."\n";
	} ;
	$self->foreach($code);
	$code->(undef) if $items[2]; # flush last page

	$self->_write_index($target, $index)
		unless defined $$opt{index} and ! $$opt{index} ;
}

sub _write_index { # write a index page for export
	my ($self, $target, $index) = @_;
	my $page = $target->get_page(':index');

	# convert index to wiki list
	# TODO use parse tree list items
	my (@list, @ns);
	for my $page (split /\n/, $index) {
		my @parts = split /:+/, $page;
		shift @parts; # /^:/ => (undef, ..)
		my $match = 1;
		for my $depth (0 .. $#parts-1) {
			next if $match and $ns[$depth] eq $parts[$depth];
			$match = 0;
			push @list,
				("\t"x$depth)."* ",
				['bold', {}, $parts[$depth]], "\n" ;
		}
		@ns = @parts;
		
		my $depth = $#ns;
		push @list,
			("\t"x$depth)."* ",
			['link', {to => $page}, $parts[-1]], "\n" ;
	}

	warn "Writing index\n";
	$page->set_parse_tree( 
		['Document', {},
			['head1', {}, "Document index"],
			['Para', {}, @list],
		] ) ;
}

1;

# TODO
# option "follow links"
# search() - returns a new selection
# list_pages()
# move(TARGET_NAMESPACE)
# copy(TARGET_NAMESPACE)

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2006 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

=cut

