package Zim::Page;

use strict;

our $VERSION = '0.16';

use overload
	'""' => sub { $_[0]->{name} }, # \&name results in bugs ?
	fallback => 'TRUE' ;

=head1 NAME

Zim::Page - Page object for Zim

=head1 DESCRIPTION

This class defines a page object. This is a data container used by
L<Zim::Repository> to represent for example a file.

Since data can be represented in a variety of ways there are multiple
interfaces to access it. These interfaces can be thought as the parent
classes a page inherits from (given multiple-iheritance), but the
interfaces supported by an individual object can change more dynamicly.

This is a general base class that does not support any interfaces
itself and cannot be used on it's own.

=head1 OVERLOAD

This class overloads the '""' operator, so the string version of an object
is the page name.

=head1 METHODS

=over 4

=item C<new(PARENT, NAME)>

Simple constructor. PARENT should be a repository object of class Zim.
NAME is the page name for this object.

=cut

sub new {
	my ($class, $parent, $name) = @_;
	$name =~ s/:+$//;
	die "Can't create $class object without a page name!\n"
		unless length $name;
	$name =~ s/^:?/:/;
	my $self = bless {
		name => $name,
		repository => $parent,
		status => '',
		properties => {read_only => 1},
	}, $class ;
	$self->init();
	return $self;
}

=item C<init()>

Called by C<new()>. To be overloaded.

=cut

sub init { }

=item C<has_interface(NAME)>

Checks whether this object supports a certain interface.

=cut

sub has_interface {
	my ($self, $name) = @_;
	return scalar grep {$_ eq $name} $self->interfaces;
}

=item C<interfaces()>

Returns a list of interfaces supported by this page.

=cut

sub interfaces { } # empty list

=item C<properties()>

Returns a hash with properties. See L</PROPERTIES>.

=cut

sub properties { $_[0]->{properties} }

=item C<name()>

Get or set the full name of the page.

=cut

sub name {
	$_[0]->{name} = $_[1] if @_ > 1;
	return $_[0]->{name};
}

=item C<basename()>

Returns the last part of the page name.

=cut

sub basename {
	my $name = $_[0]->{name};
	$name =~ s/^(.*:+)//;
	return $name;
}

=item C<namespace()>

Returns the namespace to which this page belongs.

=item C<namespaces()>

Like C<namespace()> but returns the namespace path as a list.

=cut

sub namespace {
	my $name = $_[1] || $_[0]->{name};
	#print STDERR "namespace for $name ";
	$name =~ /^(.*:)/;
	$name = $1 || ':';
	$name =~ s/::+/:/g;
	#print STDERR "is $name\n";
	return $name;
}

sub namespaces {
	my $self = shift;
	my $name = $self->namespace(@_);
	$name =~ s/^:+|:+$//g;
	return split /:+/, $name;
}

=item C<status(STRING)>

Set or get a status string for this page.
Typical status strings are 'new' and 'deleted'.

=cut

sub status {
	$_[0]->{status} = $_[1] if @_ > 1;
	return $_[0]->{status};
}

=item C<exists()>

Returns TRUE if the page already exists.

=cut

sub exists { $_[0]->{status} ne 'new' and $_[0]->{status} ne 'deleted' }

=item C<copy(TARGET)>

=item C<move(TARGET)>

=item C<delete()>

The methods C<copy()>, C<move()> and C<delete()> are aliases for the methods
C<copy_page()>, C<move_page()> and C<delete_page()> in the public repository
interface; see L<Zim::Repository>.

=cut

sub copy { $_[0]->{repository}->root->copy_page(@_) }

sub move { $_[0]->{repository}->root->move_page(@_) }

sub delete { $_[0]->{repository}->delete_page(@_) }

=item C<parse_link(LINK)>

Returns the link type and link target for LINK. The type can be 'page',
'file', 'mail' or 'man' - for urls the protocol is used as type.

The target returned is usually the same as LINK but can be different
for example for interwiki links.

=cut

sub parse_link {
	my ($self, $link) = @_;
	if ($link =~ s/^(\w[\w\+\-\.]+)\?//) { # interwiki link
		my $key = $1;
		my $l = $self->{repository}->root->interwiki_lookup($key, $link);
		if (defined $l) { $link = $l }
		else {
			return 'man', $link if $key eq 'man'; # man?zim should work
			return undef;
		}
	}
	$_ = $link;
	
	my $type;
	if    (m#^(\w+[\w\+\-\.]+)://#) { # urls
		$type = $1;
		$type = 'file' if $^O eq 'MSWin32' and $type eq 'file';
	}
	elsif (m#^mailto:|^\S+\@\S+\.\w+$#) { # email
		$link =~ s#^(mailto:)?#mailto:#;
		$type = 'mail';
	}
	else {
		$type =
			m#/#		? 'file' :
			m#:#		? 'page' :
			m#\(\d\w*\)$#	? 'man'  : 'page' ;
	}

	$link = Zim::File->abs_path($link, $self->{properties}{base})
		if $type eq 'file';

	return $type, $link;
}

=item C<resolve_link(NAME)>

Returns a page object for NAME where NAME can be resolved
relative to the current page.

=cut

sub resolve_link { # 'cloning' hack needed when exporting
	my $obj = $_[0]->{cloning} ? $_[0]->{cloning} : $_[0];
	$obj->{repository}->resolve_link(@_)
}

=item C<match_word(WORD)>

TODO: stable api for this

=cut

sub match_word { # TODO optimize by caching found links
	my ($self, $word) = @_;
	return $self->{repository}->can('_match_word')
		? $self->{repository}->_match_word($self, $word)
		: undef ;
}

=item C<clone(SOURCE)>

Import content from object SOURCE into this object.
This should be implemented depending on the interfaces supported
by both SOURCE and TARGET.

=cut

sub clone { die "Can not copy object of class ".ref($_[0])."\n" }

1;

__END__

=back

=head1 PROPERTIES

The page object contains a hash with properties. These can be any kind of data
from the backend that needs to be shared with the interface. Typically it are
config options that can be specified per-page.

For the "formatted" page interface the properties hash is used for the Document
meta attributes in the parse-tree.

Common properties are:

=over 4

=item base (url)

Base directory for files that belong to this page in some way or another.
This is for example used by the interface to resolve the location of image
files that are included in the page.

This value can be undefined when the repository does not allow direct access
to the source files.

TODO: At the moment this is the directory which contains the pae file, this
is open for change in future versions.

Currently only the file:// url is really supported.

=item read_only (boolean)

Tells the interface that this page should not be edited.
Defaults to TRUE.

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

L<Zim>,
L<Zim::Repository>

=cut
