package Zim::History;

use strict;
use Storable qw/nstore retrieve/;

our $VERSION = '0.16';

=head1 NAME

Zim::History - History object for zim

=head1 SYNOPSIS

	use Zim::History;
	
	my $hist = Zim::History->new($HIST_FILE, $PAGEVIEW, $MAX_HIST);
	my $page_record = $hist->get_current;

=head1 DESCRIPTION

This object manages zim's page history. It keeps tree different stacks: one for
the history, one for recently opened pages and one for the namespace.

The history stack is just the chronological order in which pages were opened.

The recent pages stack is derived from the history stack but avoids
all redundancy.

The namespace stack keeps track of the namespaces you opened.

The history is saved in a cache file using L<Storable>.

=head1 METHODS

=over 4

=item C<new(FILE, PAGEVIEW, MAX)>

Constructor. Takes the name of the cache file as an argument
and reads this cache file. The second argument is the pageview
object. When a new record is made for the current page the C<get_state()> method is
called on this object to get state information that needs to be saved.
Third argument is the maximum number of items in the history.

=cut

# We use two stacks here called "hist" and "recent"
# both contain the same kind of records (hashes with state data for each page)
# To keep everything in sync we want only one record per page
# which is linked by reference one or more times.


sub new {
	my ($class, $file, $view, $max) = @_;
	my $self = bless {
		file      => $file,
		PageView  => $view,
		max       => $max,
		point     => 0,
		hist      => [],
		recent    => [],
	}, $class;
	$self->read;
	return $self;
}

=item C<read>

Read the cache file.

=cut

sub read {
	my $self = shift;
	return unless -f $self->{file} and -r _;
	my @cache = @{ retrieve($self->{file}) };
	my $version = shift @cache;
	return unless $version == $VERSION;
	@{$self}{qw/point hist recent/} = @cache;
}

=item C<write>

Write the cache file.

=cut

sub write {
	my $self = shift;
	$self->get_current; # force object to record conversion
	nstore([$VERSION, @{$self}{qw/point hist recent/}], $self->{file});
}

=item C<set_current(PAGE, FROM_HIST)>

Give the page object that is going to be displayed in the 
PageView. Set a new current before updating the PageView
so the History object has time to save the state of the PageView
if necessary.

The FROM_HIST boolean signifies whether this page was loaded
from (this) history and for example as a result of following
a link. The behavior is subtily different when this boolean
is true.

=cut

sub set_current {
	my ($self, $page, $from_hist) = @_;

	my $prev = $self->get_current; # do not use get_current below !
	my $name = $page->name;
	$self->{page} = $page;
	return $self->_set_recent($prev, $from_hist)
		if $prev and $name eq $prev->{name}; # directly redundant
	
	# lookup record, or create new record
	my $rec = $self->_get_record($name) || {
		name => $name,
		namespace => $page->namespace,
		basename => $page->basename,
	};
	
	# update hist stack
	my $point = $self->{point};
	my $hist = $self->{hist};
	if    ($point >= $self->{max}) { shift @$hist }
	elsif (defined $$hist[$point]) { $point++     }
	splice @$hist, $point; # clear forw stack
	$$hist[$point] = $rec;
	$self->{point} = $point;

	$self->_set_recent($rec, $from_hist);
}

sub _set_recent {
	# update recent stack to reflect the current page
	# stuff may have been deleted (fro example because the page was deleted)
	# while the page still is in the history, so check all possibilities
	my ($self, $rec, $from_hist) = @_;
	my $recent = $self->{recent};
	my $name = $rec->{name};
	if ($from_hist) {
		return if grep {$_->{name} eq $name} @$recent; # return if exists
	}
	else {
		@$recent = grep {$_->{name} ne $name} @$recent; # remove redundancy
	}
	shift @$recent if $#$recent >= $self->{max};
	push @$recent, $rec;
}

=item C<get_current()>

Returns the current history object. When possible asks the PageView
objects for the current state information.

=cut

sub get_current {
	my $self = shift;
	my $page = $self->{page};
	my $rec = $self->{hist}[ $self->{point} ];
	#print "rec: >>$rec<<, page: >>$page<<\n";
	return $rec unless $page;
	
	unless ($rec) { # new record
		my $name = $page->name;
		$rec = $self->_get_record($name) || {
			name => $name,
			namespace => $page->namespace,
			basename => $page->basename,
		};
		$self->{hist}[ $self->{point} ] = $rec;
		#use Data::Dumper; print "rec: ", Dumper $rec;
	}

	%$rec = ( %$rec, $self->{PageView}->get_state() );

	return $rec;
}

=item C<delete_recent(PAGE)>

Deletes the entry for PAGE from the recent pages stack.

=cut

sub delete_recent {
	my ($self, $name) = @_;
	#warn "Deleting $name from recent\n";
	@{$self->{recent}} = grep {$_->{name} ne $name} @{$self->{recent}};
}

=item C<back(INT)>

Go back one or more steps in the history stack.
Returns the record for this step or undef on failure.

=cut

sub back {
	my ($self, $i) = @_;
	return if $i < 1 or $i > $self->{point};
	$self->get_current;
	$self->{point} -= $i;
	$self->{page} = undef;
	return $self->{hist}[ $self->{point} ];
}

=item C<forw(INT)>

Go forward one or more steps in the history stack.
Returns the record for this step or undef on failure.

=cut

sub forw {
	my ($self, $i) = @_;
	my $forw = $#{$self->{hist}} - $self->{point};
	return if $i < 1 or $i > $forw;
	$self->get_current;
	$self->{point} += $i;
	$self->{page} = undef;
	return $self->{hist}[ $self->{point} ];
}

=item C<get_history()>

Returns an integer followed by a list of all records in the history stack.
The integer is the index of the current page in this stack.

=cut

sub get_history {
	my $self = shift;
	$self->get_current;
	return $self->{point}, @{$self->{hist}};
}

=item C<get_recent()>

Returns an integer followed by a list of all records in the recent pages stack.
The integer is the index of the current page in this stack.

=cut

sub get_recent {
	my $self = shift;
	my $ref = $self->get_current;
	my $i = 0;
	for (@{$self->{recent}}) {
		last if $_ eq $ref;
		$i++;
	}
	return $i, @{$self->{recent}};
}

=item C<get_namespace>

This method matches the namespace of the current page to that of pages in the
history. The result can be used as a namespace stack.

=cut

sub get_namespace {
	# FIXME set hist_namespace on set_current
	# refine logic on updating records by get_current
	my $self = shift;
	my $current = $self->get_current;
	return unless $current;

	my $namespace = $current->{name};
	#print STDERR "looking for namespace $namespace";
	for (
		reverse(0 .. $self->{point}-1),
		reverse($self->{point} .. $#{$self->{hist}})
	) {
		my $rec = $self->{hist}[$_];
		my $name = $$rec{_namespace} || $$rec{name};
		$namespace = $name if $name =~ /^:*$namespace:/;
	}
	#print STDERR " => $namespace\n";
	$current->{_namespace} = $namespace;

	return $namespace;
}

=item C<get_state()>

Returns a hash ref which contains numbers that tell how many items there are on
the history and namespace stacks in either direction.

=cut

sub get_state {
	my $self = shift;
	my $state = {
		back => $self->{point},
		forw => ($#{$self->{hist}} - $self->{point}),
		up   => 0, # FIXME
		down => 0, # FIXME
	};
	return $state;
}

=item C<get_record(PAGE)>

Returns the history record for a given page object or undef.

=cut

sub get_record {
	my ($self, $page) = @_;
	
	my $name = ref($page) ? $page->name : $page;
	$name =~ s/:+$//;
	
	my $current = $self->get_current;
	return $current if $current->{name} eq $name;

	return $self->_get_record($name);
}

sub _get_record { # used by get_current()
	my ($self, $name) = @_;

	#print STDERR "get_record $name\n";
	for (@{$self->{hist}}, @{$self->{recent}}) {
		#print STDERR "\tfound $_->{name}\n";
		return $_ if $_->{name} eq $name;
	}

	return undef;
}

1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

=cut

