package Zim::GUI::TrayIcon;

use strict;
use Gtk2;
use Gtk2::TrayIcon;

our $VERSION = '0.16';

=head1 NAME

Zim::GUI::TrayIcon - TrayIcon widget for zim

=head1 DESCRIPTION

This module adds an icon in the system tray for zim that implement
"minimise to tray". This way you can always have zim running in the
background.

The context menu for the tray icon lists all pages in the current
history stack.

=head1 EXPORT

None by default.

=head1 METHODS

=over 4

=item C<new(app => PARENT)>

Simple constructor.

=item C<init()>

Method called by the constructor.

=cut

sub new {
	my $class = shift;
	my $self = bless {@_}, $class;
	$self->init();
	return $self;
}

sub init {
	my $self = shift;

	my $icon= Gtk2::TrayIcon->new("Zim ($$)");
	my $box = Gtk2::EventBox->new;
	$box->set_visible_window(0);
	$box->set_above_child(1);
	$box->signal_connect_swapped(
		button_press_event => \&on_button_press_event, $self);
	$box->signal_connect_swapped(
		popup_menu => \&popup_menu, $self);
	$icon->add($box);
	# TODO make icon themeable - use iconfactory instead of iconset
	# see gtk+ reference for gtk_image_new_from_icon_set()
	my $iconset = Gtk2::IconSet->new_from_pixbuf($self->{app}->get_icon);
	my $image = Gtk2::Image->new_from_icon_set(
		$iconset, 'large-toolbar');
	$box->add($image);
	$icon->show_all;
	$self->{icon} = $icon;

	my $tooltips = Gtk2::Tooltips->new;
	$tooltips->set_tip($box, $self->{app}{name});
	
	$self->{app}{settings}{hide_on_delete} = 1;
	$SIG{'USR1'} = sub { $self->toggle_hide_window };
}

=item C<widget()>

Returns the L<Gtk2::TrayIcon> widget.

=cut

sub widget { return $_[0]->{icon} }

sub on_button_press_event {
	my ($self, $event) = @_;
	return unless $event->type eq 'button-press';
	if ($event->button == 1) {
		$self->toggle_hide_window;
	}
	elsif ($event->button == 3) {
		$self->popup_menu($event);
	}
	return 1;
}

=item C<toggle_hide_window()>

Hides the main window of zim if it was visible or unhides it if it
it was invisible.

=cut

sub toggle_hide_window {
	my $self = shift;
	my $window = $self->{app};
	my $iconified = grep {$_ eq 'iconified'} $window->window->get_state;
	if ($window->is_active) {
		@{$self->{app}{settings}}{'x', 'y'} = $window->get_position;
		$window->hide;
	}
	else {
		if ($iconified) {
			# window is either iconified or active on another desktop (KDE)
			# need to hide it so it can be presented on current desktop
			$window->hide;
		}
		$window->present;
	}
}

=item C<popup_menu(EVENT)>

Show a popup menu containing pages in the history and a "quit" button.
EVENT is optional.

=cut

sub popup_menu {
	my ($self, $event) = @_;

	my $menu = Gtk2::Menu->new();

	my $hist = $self->{app}->History;
	my ($idx, @recent) = $hist->get_recent;

	for (qw/Home Today/) {
		my $item = Gtk2::ImageMenuItem->new_with_label($_);
		$item->set_image(
			Gtk2::Image->new_from_stock('gtk-home', 'menu') )
			if $_ eq 'Home' ;
		my $sub = 'on_'.$_;
		$item->signal_connect(
			activate => sub { $self->{app}->$sub() } );
		$menu->add($item);
	}
	
	my $separator = Gtk2::SeparatorMenuItem->new();
	$menu->add($separator);
	
	for (reverse 0 .. $#recent) {
		my $rec = $recent[$_];
		my $item = Gtk2::ImageMenuItem->new_with_label($rec->{basename});
		$item->set_image(
			Gtk2::Image->new_from_stock('gtk-yes', 'menu') )
			if $_ == $idx ;
		$item->signal_connect(
			activate => \&on_popup_menu_activate,
			[$self->{app}, $rec->{name}]  );
		$menu->add($item);
	}
	
	$separator = Gtk2::SeparatorMenuItem->new();
	$menu->add($separator);
	
	my $quit_item = Gtk2::ImageMenuItem->new_from_stock('gtk-quit');
	$quit_item->signal_connect(activate => sub {
			$self->{app}->quit('FORCE');
		} );
	$menu->add($quit_item);
	
	$menu->show_all;
	
	my ($button, $time) = $event
		? ($event->button, $event->time)
		: (0, 0) ;
	$menu->popup(undef, undef, undef, undef, $button, $time);
}

sub on_popup_menu_activate {
	my ($app, $page) = @{ pop(@_) };
	if (! $app->is_active
		&& grep {$_ eq 'iconified'} $app->window->get_state
	) {
		#handle zim on other desktop in KDE
		$app->hide;
	}
	$app->present;
	$app->go_recent($page);
}

1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

=cut

