package Zim::GUI::PageView;

use strict;
use vars '$AUTOLOAD';
use POSIX qw(strftime);
use Gtk2;
use Gtk2::Pango;               # pango constants
use Gtk2::Ex::HyperTextView;   # custom widget
use Gtk2::Ex::HyperTextBuffer; #    "     "
use Gtk2::Gdk::Keysyms;
use File::Spec;
use File::MimeInfo::Magic;
use File::BaseDir qw/xdg_data_files/;
use Zim::File;

eval "use Gtk2::Ex::DesktopEntryMenu";
my $has_mimeapplications = $@ ? 0 : 1;

our $VERSION = '0.16';

# TODO move more logic into Gtk2::Ex::HyperTextBuffer !

=head1 NAME

Zim::GUI::PageView - Page TextView widgets

=head1 DESCRIPTION

This module contains the widgets to display an editable
text buffer containing the current page. It includes a search entry
at the bottom of the TextView, formatting codes for the TextBuffer and 
an undo stack.

=head1 METHODS

Undefined methods are AUTOLOADED to the Gtk2::Ex::HyperTextView object.

=over 4

=cut

our $LINK_ICON = Gtk2->CHECK_VERSION(2, 6, 0) ? 'gtk-connect' : 'gtk-convert';
	# gtk-connect stock item was introduced in 2.6.0

my @ui_actions = (
	# name,          stock id,           label,            accelerator,       tooltip
	[ 'Head1',       undef,              'Head _1',        '<ctrl>1',         'Heading 1',     \&on_tag         ],
	[ 'Head2',       undef,              'Head _2',        '<ctrl>2',         'Heading 2',     \&on_tag         ],
	[ 'Head3',       undef,              'Head _3',        '<ctrl>3',         'Heading 3',     \&on_tag         ],
	[ 'Head4',       undef,              'Head _4',        '<ctrl>4',         'Heading 4',     \&on_tag         ],
	[ 'Head5',       undef,              'Head _5',        '<ctrl>5',         'Heading 5',     \&on_tag         ],
	[ 'Normal',      undef,              '_Normal',        '<ctrl>6',         'Normal',        \&on_tag         ],
	[ 'Bold',        'gtk-bold',         '_Bold',          '<ctrl>B',         'Bold',          \&on_tag         ],
	[ 'Italic',      'gtk-italic',       '_Italic',        '<ctrl>I',         'Italic',        \&on_tag         ],
	[ 'Underline',   'gtk-underline',    '_Underline',     '<ctrl>U',         'Underline',     \&on_tag         ],
	[ 'Strike',      'gtk-strikethrough','Stri_ke',        '<ctrl>K',         'Strike',        \&on_tag         ],
	[ 'Verbatim',    undef,              '_Verbatim',      '<ctrl>T',         'Verbatim',      \&on_tag         ],
	[ 'SplitPage',   undef,              'Spli_t page',    undef,             'Split page',    \&on_SplitPage   ],
	[ 'MergePage',   undef,              '_Merge page',    undef,             'Merge page',    \&on_MergePage   ],
	[ 'Undo',        'gtk-undo',         '_Undo',          '<ctrl>Z',         'Undo',          \&on_Undo        ],
	[ 'Redo',        'gtk-redo',         '_Redo',          '<ctrl><shift>Z',  'Redo',          \&on_Redo        ],
	[ 'Cut',         'gtk-cut',          'Cu_t',           '<ctrl>X',         'Cut',           \&on_clipboard   ],
	[ 'Copy',        'gtk-copy',         '_Copy',          '<ctrl>C',         'Copy',          \&on_clipboard   ],
	[ 'Paste',       'gtk-paste',        '_Paste',         '<ctrl>V',         'Paste',         \&on_clipboard   ],
	[ 'Delete',      'gtk-delete',       '_Delete',          undef,          'Delete',        \&on_Delete      ],
	[ 'EditLink',    'gtk-properties',   '_Edit Link...',    '<ctrl>E',         'Edit link',     \&on_EditLink    ],
	[ 'InsertDate',  undef,              'Insert _Date and Time', '<ctrl>D',         'Insert date',   \&on_InsertDate  ],
	[ 'InsertLink',  $LINK_ICON,         'Insert _Link...',  undef,             'Insert link',   \&on_EditLink    ],
	[ 'InsertImage', undef,              'Insert _Image...', undef,             'Insert image',  \&on_InsertImage ],
	[ 'FindReplace', 'gtk-find-and-replace', '_Replace...', '<ctrl>H', 'Find and Replace', \&on_FindReplace ],
);

my @ui_actions_ro = (
	[ 'Link',        $LINK_ICON,       '_Link',          '<ctrl>L',         'Link',          \&on_Link        ],
	[ 'Find',        'gtk-find',       '_Find...',          '<ctrl>F',         'Find',          \&on_Find        ],
	[ 'FindNext',    undef,            'Find Ne_xt',     '<ctrl>G',         'Find next',     \&on_FindNext    ],
	[ 'FindPrev',    undef,            'Find Pre_vious', '<ctrl><shift>G',  'Find previous', \&on_FindPrev    ],
);

#		<menu action='FileMenu'>
#			<placeholder name='FileMods'>
#				<menuitem action='SplitPage'/>
#				<menuitem action='MergePage'/>
#			</placeholder>
#		</menu>

my $ui_layout_ro = q{<ui>
	<menubar name='MenuBar'>
		<menu action='SearchMenu'>
			<placeholder name='FindItems'>
				<menuitem action='Find'/>
				<menuitem action='FindNext'/>
				<menuitem action='FindPrev'/>
			</placeholder>
		</menu>
	</menubar>
	<accelerator action='Link'/>
</ui>};

my $ui_layout = q{<ui>
	<menubar name='MenuBar'>
		<menu action='EditMenu'>
			<placeholder name='EditPage'>
				<menuitem action='Undo'/>
				<menuitem action='Redo'/>
				<separator/>
				<menuitem action='Cut'/>
				<menuitem action='Copy'/>
				<menuitem action='Paste'/>
				<menuitem action='Delete'/>
				<separator/>
				<menuitem action='EditLink'/>
				<separator/>
			</placeholder>
			<placeholder name='InsertItems'>
				<menuitem action='InsertDate'/>
				<menuitem action='InsertImage'/>
				<menuitem action='InsertLink'/>
			</placeholder>
		</menu>
		<menu action='FormatMenu'>
			<menuitem action='Head1'/>
			<menuitem action='Head2'/>
			<menuitem action='Head3'/>
			<menuitem action='Head4'/>
			<menuitem action='Head5'/>
			<menuitem action='Normal'/>
			<separator/>
			<menuitem action='Bold'/>
			<menuitem action='Italic'/>
			<menuitem action='Underline'/>
			<menuitem action='Strike'/>
			<menuitem action='Verbatim'/>
			<separator/>
			<menuitem action='Link'/>
		</menu>
		<menu action='SearchMenu'>
			<placeholder name='FindItems'>
				<menuitem action='FindReplace'/>
			</placeholder>
		</menu>
	</menubar>
	<toolbar name='ToolBar'>
		<placeholder name='Format'>
			<toolitem action='Link'/>
			<toolitem action='EditLink'/>
			<separator/>
			<toolitem action='Bold'/>
			<toolitem action='Italic'/>
			<toolitem action='Underline'/>
			<toolitem action='Strike'/>
		</placeholder>
		<separator/>
	</toolbar>
</ui>};

=item C<new(app => PARENT)>

Simple constructor.

=item C<init()>

Method called by the constructor.

=cut

my ($k_tab, $k_return, $k_kp_enter, $k_backspace, $k_escape, $k_multiply, $k_home) =
	@Gtk2::Gdk::Keysyms{qw/Tab Return KP_Enter BackSpace Escape KP_Multiply Home/};
#my @k_parse_word = ($k_tab, map ord($_), ' ', qw/. ; , ' "/);

our %UNDO_STEPS = (
	delete     => 'insert',
	insert     => 'delete',
	apply_tag  => 'remove_tag',
	remove_tag => 'apply_tag'
);

$Gtk2::Ex::HyperTextBuffer::TAGS{link} = [foreground => 'blue'];
$Gtk2::Ex::HyperTextBuffer::TAGS{underline} = [background => 'yellow'];

sub new {
	my $class = shift;
	my $self = bless {@_}, $class;
	$self->init();
	return $self;
}

sub init { # called by new()
	my $self = shift;

	my $vbox = Gtk2::VBox->new(0, 0);
	$self->{vbox} = $vbox;
	
	my $scrolled_window = Gtk2::ScrolledWindow->new();
	$scrolled_window->set_policy('automatic', 'automatic');
	$scrolled_window->set_shadow_type('in');
	$vbox->add($scrolled_window);
	$self->{scrolled_window} = $scrolled_window;

	# init TextView
	my $htext = Gtk2::Ex::HyperTextView->new();
	$htext->set_left_margin(10);
	$htext->set_right_margin(5);
	$htext->set_editable(0) if $self->{app}{settings}{read_only};
	$htext->set_tabs( Gtk2::Pango::TabArray->new_with_positions(
		# initial_size, pos_in_pixels, ... allign => position
		1, 0, 'left' => 40 * PANGO_SCALE ) );
	$htext->{link_properties} = [foreground => 'blue']; # TextTag properties
	$htext->signal_connect(link_clicked =>
		sub { $self->{app}->link_clicked($_[1]) }  );
	$htext->signal_connect(link_enter =>
		sub { $self->{app}->push_status("Go to \"$_[1]\"", 'link') }  );
	$htext->signal_connect(link_leave =>
		sub { $self->{app}->pop_status('link') }  );
	$htext->signal_connect_after(toggle_overwrite => \&on_toggle_overwrite, $self);
	$htext->signal_connect(populate_popup => \&on_populate_popup, $self);
	$htext->signal_connect(key_press_event => \&on_key_press_event, $self);
	my $reset_em = sub {
		$self->{buffer}->reset_edit_mode if $self->{buffer};
		return 0;
	};
	$htext->signal_connect(move_cursor => $reset_em);
	$htext->signal_connect(button_press_event => $reset_em);
	$htext->drag_dest_set(['motion', 'highlight'],
		['link', 'copy', 'move'] ); # We would prefer only 'link' here, but KDE needs 'move' 
	my $tlist = Gtk2::TargetList->new();
	if (Gtk2->CHECK_VERSION(2, 6, 0)) {
		$tlist->add_uri_targets(0);
		$tlist->add_text_targets(1);
	}
	else { # 2.4
		$tlist->add_table(
			['text/uri-list', [], 0], map [$_, [], 1],
			qw{UTF8_STRING TEXT COMPOUND_TEXT text/plain} );
	}
	$htext->drag_dest_set_target_list($tlist);
	$htext->signal_connect_swapped(
		drag_data_received => \&on_drag_data_recieved, $self );
	#$htext->signal_connect(drag_motion => \&on_drag_motion); # debug
	$scrolled_window->add($htext);
	$self->{htext} = $htext;

	# init search box
	my $hbox = Gtk2::HBox->new(0, 5);
	$hbox->set_no_show_all(1);
	$hbox->signal_connect(key_press_event => \&on_key_press_event_hbox, $self);
	$vbox->pack_start($hbox, 0, 1, 3);
	$self->{hbox} = $hbox;
	
	my $close_button = Gtk2::Button->new;
	$close_button->set_relief('none');
	$close_button->add(Gtk2::Alignment->new(0.5, 0.5, 0, 0));
	$close_button->child->add(
		Gtk2::Image->new_from_stock('gtk-close', 'small-toolbar') );
	$close_button->signal_connect(clicked => sub { $self->hide_find } );
	$hbox->pack_end($close_button, 0,1,0);
	
	$hbox->pack_start( Gtk2::Label->new(' Find: '), 0,1,0);
	
	my $entry = Gtk2::Entry->new();
	$entry->signal_connect(changed => sub { $self->_find($entry->get_text, 0) });
	$entry->signal_connect_swapped(activate => \&on_activate_entry,  $self);
	$hbox->pack_start($entry, 0, 1, 0);
	$self->{entry} = $entry;

	my $prev_button = $self->{app}->new_button('gtk-go-back', '_Previous');
	$prev_button->set_sensitive(0);
	$prev_button->signal_connect_swapped(clicked => \&find_prev, $self);
	$hbox->pack_start($prev_button, 0, 1, 0);
	$self->{find_prev_button} = $prev_button;
	
	my $next_button = $self->{app}->new_button('gtk-go-forward', '_Next');
	$next_button->set_sensitive(0);
	$next_button->signal_connect_swapped(clicked => \&find_next, $self);
	$hbox->pack_start($next_button, 0, 1, 0);
	$self->{find_next_button} = $next_button;

	# add toolbar buttons and key bindings
	my $accels = $self->{app}{ui}->get_accel_group;
	my $read_only = $self->{app}{settings}{read_only};
	my $actions = Gtk2::ActionGroup->new("PageView");
	$self->{app}{ui}->insert_action_group($actions, 0);
	$actions->add_actions(\@ui_actions_ro, $self);
	$self->{app}{ui}->add_ui_from_string($ui_layout_ro);
	unless ($read_only) {
		$actions->add_actions(\@ui_actions, $self);
		$self->{app}{ui}->add_ui_from_string($ui_layout);
		$accels->connect( # ^Y (identical with the shift-^Z defined above)
			ord('Y'), ['control-mask'], ['visible'], sub {$self->redo} );
	}
	$accels->connect( # alt-/ (identical with the ^F defined above)
		ord('/'), ['mod1-mask'], ['visible'], sub {$self->show_find} );
	$accels->connect( Gtk2::Accelerator->parse('F3'), ['visible'],
		sub { $self->on_FindNext } );

	# prevent our accelerators from triggering while editing another widget
	$htext->signal_connect(focus_in_event  => sub { $actions->set_sensitive(1) });
	$htext->signal_connect(focus_out_event => sub { $actions->set_sensitive(0) });
	
	$htext->set_cursor_visible( $read_only ? 0 : 1 );
}

sub AUTOLOAD {
	my $self = shift;
	$AUTOLOAD =~ s/^.*:://;
	return if $AUTOLOAD eq 'DESTROY';
	return $self->{htext}->$AUTOLOAD(@_);
}

sub on_tag {
	my ($action, $self) = @_;
	my $tag = lc $action->get_name;
	$self->apply_tag($tag);
}

sub on_Link { pop->apply_link }

sub on_SplitPage {
	my $self = pop;
	# TODO prompt level
	$self->{app}->save_page;
	$self->{app}{repository}->split_page($self->{app}{page}, 1);
	$self->{app}->TreeView->{_loaded} = 0;
	$self->{app}->TreeView->load_index;
	$self->{app}->reload;
}

sub on_MergePage {
	my $self = pop;
	$self->{app}->save_page;
	$self->{app}{repository}->merge_page($self->{app}{page});
	$self->{app}->TreeView->{_loaded} = 0;
	$self->{app}->TreeView->load_index;
	$self->{app}->reload;
}

sub on_Undo { pop->undo }

sub on_Redo { pop->redo }

sub on_clipboard {
	my ($action, $self) = @_;
	return unless $self->{htext}->get('has-focus');
	my $signal = lc($action->get_name).'_clipboard';
	$self->{htext}->signal_emit($signal);
}

sub on_Delete {
	my $self = pop;
	return unless $self->{htext}->get('has-focus');
	$self->{htext}->signal_emit('delete_from_cursor', 'chars', 1);
}

sub on_EditLink { pop->edit_link_dialog }

sub on_Find { pop->show_find }

sub on_FindNext { pop->find_next }

sub on_FindPrev { pop->find_prev }

sub on_FindReplace {
	pop->{app}->FindReplaceDialog->show();
}

sub on_InsertDate { # insert date that is linked
	my $self = pop;
	my $settings = $self->{app}{settings};
	my $buffer = $self->get_buffer;
	my $string = strftime( $settings->{date_string}, localtime);
	my $link = $settings->{cal_namespace} . strftime('%Y_%m_%d', localtime);
	$buffer->insert_blocks_at_cursor(['link', {to => $link}, $string]);
}

sub on_InsertImage { pop->insert_image }

=item C<widget()>

Returns the root widget. This should be used to add the object to a container widget.
Also use this widget for things like show_all() and hide_all().

=cut

sub widget { return $_[0]->{vbox} }

sub on_key_press_event_hbox {
	my ($hbox, $event, $self) = @_;
	return 0 unless $event->keyval == $k_escape;
	$self->hide_find;
	return 1;
}

sub on_activate_entry {
	my ($self, $entry) = @_;
	$self->{hbox}->hide;
	$self->_find($entry->get_text, 0);
	$self->{htext}->grab_focus;
}

=item C<find(STRING)>

=item C<find_next()>

=item C<find_prev()>

Search for strings in the buffer.

=cut

sub _find {
	my $self = shift;
	my $succes = $self->{htext}->search(@_);
	# set buttons active depending on whether we found something or not
	$_->set_sensitive($succes)
		for @{$self}{'find_prev_button', 'find_next_button'};
}

sub find {
	$_[0]->{entry}->set_text($_[1]);
	$_[0]->_find($_[1], 0);
}

sub find_next { $_[0]->_find( $_[0]->{entry}->get_text, 1  ) }

sub find_prev { $_[0]->_find( $_[0]->{entry}->get_text, -1 ) }

=item C<show_find()>

Show the search bar at the bottom of the page.

=cut

sub show_find {
	my $self = shift;
	$self->{hbox}->set_no_show_all(0);
	$self->{hbox}->show_all;
	$self->{entry}->grab_focus;
}

=item C<hide_find()>

Hide the search bar.

=cut

sub hide_find {
	my $self = shift;
	$self->{hbox}->hide_all;
	$self->{hbox}->set_no_show_all(1);
	$self->{htext}->grab_focus;
}

sub on_toggle_overwrite {
	my $self = pop;
	my $over = $_[0]->get_overwrite;
	my $sbar = $self->{app}{status2};
	my $id = $sbar->get_context_id('over');
	if ($over) { $sbar->push($id, '  OVR') }
	else       { $sbar->pop($id)           }
}

sub on_drag_motion { # debugging purposes only
	my $context = $_[1];
	my @targets = map $_->name, $context->targets;
	warn "Drag motion with targets: @targets\n";
}

sub on_drag_data_recieved {
	my ($self, $context, $x, $y, $selection, $info, $time) = @_;
	my $type = $selection->target->name;
	my $data = $selection->data;
	#warn "$info ($time): $type: $data\n";
	return unless $type eq 'text/uri-list';
	
	my @files = grep defined($_), split /[\r\n]+/, $data;
	for (@files) {
		s!\%([A-Fa-z0-9]{2})!chr(hex($1))!eg; # url encoding
		eval {$_ = Encode::decode_utf8($_, 1)}; # utf8 decoding
		$_ = $self->_check_rel_path($_) if m#^file:/#;
	}
	@files = map {
		unless (m#^\w[\w\+\-\.]+:/#) {
			my $mtype = mimetype($_);
			($mtype =~ /^image\//)
				? ['image', {src => $_}, $_]
				: ['link',  {to  => $_}, $_]
				, "\n" ;
		}
		else { ['link', {to => $_}, $_], "\n" }
	} @files;
	
	($x, $y) = $self->{htext}->window_to_buffer_coords('widget', $x, $y);
	my $iter = $self->{htext}->get_iter_at_location($x, $y);
	$self->{buffer}->insert_blocks($iter, @files);
}

sub on_populate_popup {
       	# add items to context menu
	# object can be a link to a page, a URL (mail / file / http) or an image (which is a file)
	my ($htext, $menu, $self) = @_;
	my $iter = $htext->get_iter_at_pointer;
	return unless $iter;
	
	# source is the filename
	# link is the name as written in the page source
	my ($type, $source);
	my $link = $htext->get_link_at_iter($iter);
	if (defined $link) {
		($type, $source) = $self->{app}{page}->parse_link($link);
	}
	elsif (my $pixbuf = $iter->get_pixbuf) {
		return unless defined $pixbuf->{image_src};
		($type, $source) = ('image', $pixbuf->{image_src});
		$source = $pixbuf->{image_file};
	}
	else { return }
	
	# Separator
	my $seperator = Gtk2::MenuItem->new();
	$seperator->show;
	$menu->prepend($seperator);
	
	# Edit object
	my $edit_item = Gtk2::MenuItem->new('_Edit Link');
	$edit_item->show;
	$edit_item->signal_connect_swapped(activate => ($type eq 'image')
		? sub { $self->edit_image_dialog($iter) }
		: sub { $self->edit_link_dialog($iter)  }
		, $self);
	$menu->prepend($edit_item);

	# Copy object
	my $copy_item = Gtk2::MenuItem->new(
		($type eq 'mail') ? 'Copy Email Address' : 'Copy _Link' );
	$copy_item->signal_connect(activate => sub {
		my $clipboard = Gtk2::Clipboard->get(
			Gtk2::Gdk::Atom->new('PRIMARY') );
		$clipboard->set_text($link);
	} );
	$copy_item->show;
	$menu->prepend($copy_item);

	if ($type eq 'image' or $type eq 'file') {
		# Open directory
		my $browse_item = Gtk2::MenuItem->new('Open _Directory');
		$browse_item->signal_connect(activate =>
			sub { $self->{app}->open_directory($source) } );
		$browse_item->show;
		$menu->prepend($browse_item);
		
		return unless $has_mimeapplications;
		
		# Separator
		$seperator = Gtk2::MenuItem->new();
		$seperator->show;
		$menu->prepend($seperator);
	
		# Open with...
		Gtk2::Ex::DesktopEntryMenu->populate_menu($menu, $source);
	}
	else {
		# Separator
		$seperator = Gtk2::MenuItem->new();
		$seperator->show;
		$menu->prepend($seperator);
	
		# Open link
		my $open_item = Gtk2::MenuItem->new('_Open link');
		$open_item->signal_connect(activate =>
			sub { $self->{app}->link_clicked($link) }  );
		$open_item->show;
		$menu->prepend($open_item);
	}
	
}

sub on_key_press_event { # some extra keybindings
	# FIXME for more consistent behaviour test for selections
	my ($htext, $event, $self) = @_;
	my $val = $event->keyval;

	if ($self->{app}{settings}{read_only}) {
		return 0 unless $val == ord('/');
		$self->show_find;
		return 1;
	}
	
	if ($val == $k_return or $val == $k_kp_enter) { # Enter
		my $buffer = $htext->get_buffer;
		my $iter = $buffer->get_iter_at_mark($buffer->get_insert());
		return 1 if defined $htext->click_if_link_at_iter($iter);
		$self->parse_word($iter); # end-of-line is also end-of-word
		$iter = $buffer->get_iter_at_mark($buffer->get_insert());
		$self->parse_line($iter) or return 0;
		$htext->scroll_mark_onscreen( $buffer->get_insert );
		return 1;
	}
	elsif (
		$self->{app}{settings}{backsp_unindent} and $val == $k_backspace or
		$val == $k_tab and $event->state eq 'shift-mask'
	) { # BackSpace or Shift-Tab
		my $buffer = $htext->get_buffer;
		my ($start, $end) = $buffer->get_selection_bounds;
		if ($end and $end != $start) {
			my $cont = $self->selection_backspace($start, $end);
			return $val == $k_tab ? 1 : $cont;
		}
		my $iter = $buffer->get_iter_at_mark($buffer->get_insert());
		if ($self->parse_backspace($iter)) {
			$htext->scroll_mark_onscreen( $buffer->get_insert );
			return 1;
		}
	}
	elsif ($val == $k_tab or $val == ord(' ')) { # WhiteSpace
		my $buffer = $htext->get_buffer;
		if ($val == $k_tab) {
			my ($start, $end) = $buffer->get_selection_bounds;
			if ($end and $end != $start) {
				$self->selection_tab($start, $end);
				return 1;
			}
		}
		my $iter = $buffer->get_iter_at_mark($buffer->get_insert());
		my $string = ($val == $k_tab) ? "\t" : ' ';
		if ($self->parse_word($iter, $string)) {
			$htext->scroll_mark_onscreen( $buffer->get_insert );
			return 1;
		}
	}
	elsif ($val == ord('*') or $val == $k_multiply) { # Bullet
		my $buffer = $htext->get_buffer;
		my ($start, $end) = $buffer->get_selection_bounds;
		return 0 if !$end or $end == $start;
		$self->toggle_bullets($start, $end);
		return 1;
	}
	elsif ($val == $k_home and not $event->state >= 'control-mask') { # Home toggle
		my $buffer = $htext->get_buffer;
		my $insert = $buffer->get_iter_at_mark($buffer->get_insert());
		my $start  = $insert->copy;
		$htext->backward_display_line_start($start)
			unless $htext->starts_display_line($start);
		my $begin  = $start->copy;
		my $indent = '';
		while ($indent =~ /^\s*([^\s\w]\s*)?$/) {
			last if $begin->ends_line or ! $begin->forward_char;
			$indent = $start->get_text($begin);
		}
		$indent =~ /^(\s*([^\s\w]\s+)?)/;
		my $back = length($indent) - length($1);
		$begin->backward_chars($back) if $back > 0;
		$insert = ($begin->ends_line || $insert->equal($begin)) ? $start : $begin;
		if ($event->state >= 'shift-mask') {
			$buffer->move_mark_by_name('insert', $insert);
			# leaving the "selection_bound" mark behind
		}
		else { $buffer->place_cursor($insert) }
		return 1;
	}
		
	#else { printf "key %x pressed\n", $val } # perldoc -m Gtk2::Gdk::Keysyms

	return 0;
}

=item C<set_font(STRING)>

Set the default font for the pageview.

=cut

sub set_font {
	my ($self, $string) = @_;
	my $font = Gtk2::Pango::FontDescription->from_string($string);
	$self->{htext}->modify_font($font);
}

=item C<get_state()>

Returns a number of properties that need to be saved in the history.

=cut

sub get_state {
	my $self = shift;
	my $buffer = $self->{buffer} || return;
	my $cursor = $buffer->get_iter_at_mark($buffer->get_insert)->get_offset;
	#my $vscroll = $self->{scrolled_window}->get_vadjustment->get_value;
	#my $hscroll = $self->{scrolled_window}->get_hadjustment->get_value;
	return cursor => $cursor, undo => $self->{undo}, redo => $self->{redo};
		#, vscroll => $vscroll, hscroll => $hscroll;
}

=item C<set_state(property => value, ..)>

Set a number of properties that could be saved in the history.

=cut

sub set_state {
	my $self = shift;
	my %rec = @_;
	
	if (defined $rec{cursor}) {
		$self->{buffer}->place_cursor(
			$self->{buffer}->get_iter_at_offset($rec{cursor}) );
	
		$self->{htext}->scroll_mark_onscreen( $self->{buffer}->get_insert );
	}
	
	#$self->{scrolled_window}->get_vadjustment->set_value($rec{vscroll});
	#$self->{scrolled_window}->get_hadjustment->set_value($rec{hscroll});
	
	$self->{undo} = $rec{undo} || [];
	$self->{redo} = $rec{redo} || [];
}

=item C<load_page(PAGE)>

Load a new page object into the buffer.

=cut

sub load_page {
	my ($self, $page) = @_;
	my $use_spell = defined $self->{app}{objects}{Spell}; # FIXME ugly internals
	
	# clear the old buffer
	$self->{undo_lock} = 1;
	$self->{buffer}->clear if $self->{buffer};
	$self->{app}->Spell->detach($self->{htext}) if $use_spell;
	$self->{_prev_buffer} = $self->{buffer}; # FIXME hack to prevent segfaults
	
	# create a new HyperTextBuffer
	my $buffer = Gtk2::Ex::HyperTextBuffer->new();
	$buffer->create_default_tags;
#	$buffer->create_tag('info', editable => 0); #, foreground => 'red');
	$self->{buffer} = $buffer;
	$self->{htext}->set_buffer($buffer);
	my $tree = $page->get_parse_tree;
	$buffer->set_parse_tree($tree);
	unless ($self->{app}{settings}{read_only}) {
		# connect signals _after_ load_parsetree()
		$buffer->signal_connect(delete_range => \&on_delete_range, $self);
		$buffer->signal_connect_after(insert_text => \&on_insert_text, $self);
		$buffer->signal_connect(modified_changed =>
			sub {$self->{app}->update_status} );
		$buffer->signal_connect(apply_tag => \&on_apply_tag, $self);
		$buffer->signal_connect(remove_tag => \&on_remove_tag, $self);
	}
	$self->{app}->Spell->attach($self->{htext}) if $use_spell;
	$self->{undo_lock} = 0;
	
	if ($page->exists) {
		$buffer->place_cursor(
	 	      	$buffer->get_iter_at_offset(0) );
	}
	else { # new page, place cursor below headers but above footer
		my (undef, $iter) = $buffer->get_bounds;
#		$iter->backward_to_tag_toggle( $buffer->get_tag_table->lookup('info') );
		$buffer->place_cursor( $iter );
	}
	
	$self->{undo} = [];
	$self->{redo} = [];
	
	$self->{htext}->scroll_mark_onscreen( $buffer->get_insert );

	$buffer->set_modified(0);
	$self->_match_all_words($buffer, $page)
		if $self->{app}{settings}{use_autolink} && $page->exists;
}

=item C<modified()>

=item C<modified(BOOLEAN)>

Get or set the modified bit. This bit is set when the buffer
is modified by the user.
It should be reset after succesfully saving the buffer content.

=cut

sub modified {
	return 0 unless defined $_[0]->{buffer};
	$_[0]->{buffer}->set_modified($_[1]) if defined $_[1];
	$_[0]->{buffer}->get_modified;
}

=item C<get_parse_tree(PAGE)>

Returns the parse tree for the current buffer contents.

=cut

sub get_parse_tree {
	my $tree = $_[0]->{buffer}->get_parse_tree;
#	my @head = splice @$tree, 0, 2;
#	@$tree = grep {not ref($_) && $$_[0] eq 'info'} @$tree;
#	unshift @$tree, @head;
	return $tree;
}

=item C<parse_backspace(ITER)>

This method is called when the user types a backspace.
It tries to update the formatting of the current line.
 
When TRUE is returned the widget does not recieve the backspace.

=cut

sub parse_backspace {
	my ($self, $iter) = @_;
	my $buffer = $self->{buffer};
	my $lf = $buffer->get_iter_at_line( $iter->get_line );
	my $line = $buffer->get_text($lf, $iter, 0);
	if ($line =~ s/\t([\*\x{2022}]\s)$/$1/) {
		$buffer->delete($lf, $iter);
		$buffer->insert($lf, $line);
		return 1;
	}
	return 0;
}

=item C<parse_line(ITER)>

This method is called when the user is about to insert a linebreak.
It checks the line left of the cursor of any markup that needs 
updating. It also takes care of autoindenting.

When TRUE is returned the widget does not recieve the linebreak.

=cut

sub parse_line {
	my ($self, $iter) = @_;
	my $buffer = $self->{buffer};
	my $lf = $buffer->get_iter_at_line( $iter->get_line );
	my $line = $buffer->get_text($lf, $iter, 0);
	#print ">>$line<<\n";
	if ($line =~ s/^(=+)\s*(\w)/$2/) { # heading
		my $offset;
		($lf, $offset) = ($lf->get_offset, $iter->get_offset);
		$buffer->insert($iter, "\n"); # keep undo order
		($lf, $iter) = map $buffer->get_iter_at_offset($_), $lf, $offset;
		$iter->forward_char;
		my $h = length($1); # no (7 - x) monkey bussiness here
		$h = 5 if $h > 5;
		$line =~ s/\s+=+\s*$//;
		$offset = $lf->get_offset + length $line;
		$buffer->delete($lf, $iter);
		$buffer->insert_with_tags_by_name($lf, $line, "head$h");
		$iter = $buffer->get_iter_at_offset($offset);
		$buffer->insert($iter, "\n");
		return 1;
	}
	elsif ($line =~ /^(\s*(:?\W+|\d+\W?|\w\W)\s+|\s+)$/) { # empty bullet or list item
		# TODO check previous line for same pattern !
		$buffer->delete($lf, $iter);
	}
	elsif ($line =~ /^(\s*(:?\W+|\d+\W?|\w\W)\s+|\s+)/) { # auto indenting + lists
		my ($indent, $number) = ($1, $2);
		if (length $number) { # numbered list
			$number = ($number =~ /\d/) ? $number+1 : chr(ord($number)+1);
			$indent =~ s/^(\s*)(\d+|\w)/$1$number/;
		}
		my $offset = $iter->get_offset;
		$buffer->insert($iter, "\n");
		$iter = $buffer->get_iter_at_offset($offset);
		$iter->forward_char;
		$buffer->insert($iter, "$indent");
		$self->{htext}->scroll_mark_onscreen( $buffer->get_insert() );
		return 1;
	}
	return 0;
}

=item C<parse_word(ITER, CHAR)>

This method is called after the user ended typing a word.
It checks the word left of the cursor for any markup that
needs updating.

CHAR can be the key that caused a word to end, returning TRUE
makes it never reaching the widget.

=cut

sub parse_word {
	# remember that $char can be empty
	# first insert the char, then replace it, keep undo stack in proper order
	my ($self, $iter, $char) = @_;
	return 0 if $self->_is_verbatim($iter);
	my $buffer = $self->{buffer};
	my $lf = $iter->copy;
	$self->{htext}->backward_display_line_start($lf)
		unless $self->{htext}->starts_display_line($lf);
	my $line = $buffer->get_text($lf, $iter, 0);
	#warn ">>$line<< >>$char<<\n";
	if ($line =~ /^(\s*)[\*\x{2022}](\s*)$/) { # bullet
		return unless $lf->starts_line; # starts_display_line != starts_line
		my ($pre, $post) = ($1, $2.$char);
		my $offset;
		if (defined $char) {
			($lf, $offset) = ($lf->get_offset, $iter->get_offset);
			$buffer->insert($iter, $char); # get undo order good
			($lf, $iter) = map $buffer->get_iter_at_offset($_), $lf, $offset;
			$iter->forward_char;
			$pre .= $1 if $post =~ s/(\t+)$//; # switch tabs
			$post = ' ' unless length $post;
		}
		$line = $pre."\x{2022}".$post;
		$buffer->delete($lf, $iter);
		$buffer->insert($lf, $line);
		return 1;
	}
	elsif (
		$line =~ qr{(?<!\S)(\w[\w\+\-\.]+://\S*)$} # url
		or $line =~ qr{(?<!\S)([\w\.\-\(\):]*:[\w\.\-\(\)]{2,}:*)$} # page
		or $line =~ qr{(?<!\S)((?:~/|~\S*/|\.\.?/|/)\S*)$} # file
		or $self->{app}{settings}{use_camelcase}
		and $line =~ qr{(?<!\S)([A-Z]+[^\WA-Z]+[A-Z]+\w*)$} # CamelCase
	) { # any kind of link
		my $word = $1;
		return 0 unless $word =~ /[^:\dhu]/i; # do not link "10:20h"
		my ($start, $end) = ($iter->copy, $iter->copy);
		$start->backward_chars(length $word);
		return 0 if $start->get_tags or $end->get_tags;
		if (defined $char) {
			($start, $end) = ($start->get_offset, $end->get_offset);
			$buffer->insert($iter, $char);
			($start, $end) = map $buffer->get_iter_at_offset($_), $start, $end;
		}
		$self->{htext}->apply_link(undef, $start, $end);
		return 1;
		
	}
	elsif ( $self->{app}{settings}{use_utf8_ent} &&
		$line =~ /(?<!\S)\\(\w+)$/) { # utf8 chars
		my $word = $1;
		my $chr = _entity($word);
		return 0 unless defined $chr;
		
		if (defined $char) {
			my $offset = $iter->get_offset;
			$buffer->insert($iter, $char);
			$iter = $buffer->get_iter_at_offset($offset)
		}
		my $begin = $iter->copy;
		$begin->backward_chars(1 + length $word);
		$buffer->delete($begin, $iter);
		$buffer->insert($begin, $chr);
		return 1;
	}
#	elsif ($line =~ /^(\t|  )/) { # pre
#		# FIXME \s* => \t
#		$iter->forward_char unless $iter->is_end; # FIXME somthing at end
#		$buffer->apply_tag_by_name('pre', $lf, $iter);
#	}
	elsif ($self->{app}{settings}{use_autolink}) {
		$self->_match_words($buffer, $iter);
	}
	
	return 0;
}

my %_entities;

sub _entity {
	my $key = shift;
	return chr($key) if $key =~ /^\d+$/;
	my $ord;
	if (exists $_entities{$key}) { return $_entities{$key} }
	else {
		for (xdg_data_files(qw/zim entities.list/)) {
			$ord = Zim::File->new($_)->read_hash($key);
			last if defined $ord;
		}
	}
	return unless length $ord;
	$ord =~ /(\d+)/; # ignore comments
	return chr($1);
}

sub _match_words {
	my ($self, $buffer, $iter, $page) = @_;
	return if $iter->starts_line;
	$page ||= $self->{app}{page};
	my $start = $iter->copy;
	$start->backward_chars( $iter->get_line_offset );
	my $line = $start->get_text($iter);
	
	while ($line =~ /\w/) {
		my ($word) = ($line =~ /(\w+)/);
		#warn "Checking: >>$word<<\n";
		while ($_ = $page->match_word($word)) {
			warn "Matching: $_ for >>$word<<\n";
			# match_word returns 1 for single match
			#            and 2 for multiple or partial matches
			if ($_ == 1) {
				my $start = $iter->copy;
				$start->backward_chars(length $line);
				my $end = $start->copy;
				$end->forward_chars(length $word);
				last if $start->get_tags or $end->get_tags;
				$self->{htext}->apply_link(undef, $start, $end);
				last;
			}
			else {
				($word) = ($line =~ /(\Q$word\E\W+\w+)/);
				last unless $word;
			}
		}
		$line =~ s/^\W*\w+\W*//;
		#warn "line: >>$line<<\n";
	}
}

sub _match_all_words { # call _match_words on all lines
	my ($self, $buffer, $page) = @_;
	my ($iter, undef) = $buffer->get_bounds;
	while ($iter->forward_to_line_end) {
		$self->_match_words($buffer, $iter, $page);
	}
}

sub _is_verbatim {
	my ($self, $iter) = @_;
	for ($iter->get_tags) {
		return 1 if lc($_->get_property('name')) eq 'verbatim';
	}
	return 0;
}

=item C<insert_image(IMAGE)>

Inserts an image into the buffer. Without argument prompts the user for a file.

=cut

sub insert_image {
	my ($self, $file) = @_;
	$file = $self->{app}->filechooser_dialog unless $file;
	$file = $self->_check_rel_path($file);
	return unless length $file;
	$self->{buffer}->insert_image_from_file($file);
}

=item C<insert_file(FILE)>

Inserts a link to a file into the buffer.

=cut

sub insert_file {
	my ($self, $file) = @_;
	return unless length $file;
	die 'TODO'
}

sub _check_rel_path {
	# 3 way directory comparison
	# Returns path relative to 'ref' when file is below 'root'
	# returns absolute filename otherwise
	my ($self, $file) = @_;
	my $ref = $self->{app}{page}->properties->{base};
	$ref = Zim::File->abs_path($ref);
	return $file unless defined $ref;
	$file = Zim::File->abs_path($file,$ref);
	return $file if $file =~ m#^\w\w+:/#;
	my $root = $self->{app}{repository}->base;
	$root = Zim::File->abs_path($root);
	#warn "check whether '$file' is below '$ref'\n";

	my @root = split m#/+#, $root;
	my @ref = split m#/+#, $ref;
	my @file = split m#/+#, $file;
	for (@root) { # remove root part - return if no match
		my $dir = shift @file;
		#warn "match $dir $_\n";
		return $file if $dir ne $_;
		$dir = shift @ref;
		return $file if $dir ne $_; # just to be sure
	}
	while (@ref) { # remove common path
		last if $ref[0] ne $file[0];
		shift @ref;
		shift @file;
	}

	if (@ref) { unshift @file, '..' for 0 .. $#ref }
	else      { unshift @file, '.'                 }
	
	return join '/', @file; # un-localize
}

=item C<apply_tag(TAG)>

Applies the tag with the name TAG to any selected text.

=cut

sub apply_tag {
	my ($self, $tag) = @_;
	my $buffer = $self->{buffer};
	my ($start, $end) = $buffer->auto_selection($tag);
	return unless defined $start;

	if ($tag eq 'verbatim') {
		my $Verbatim = $buffer->get_tag_table->lookup('Verbatim');
		$tag = 'Verbatim'
			if $start->get_line != $end->get_line
			or $start->starts_line && $end->ends_line
			or $start->has_tag($Verbatim)
			or $end->has_tag($Verbatim)  ;
	}
	
	$buffer->remove_all_tags($start, $end);
	$buffer->apply_tag_by_name($tag, $start, $end)
		unless $tag eq 'normal';
	$buffer->set_modified(1);

	if ($tag =~ /^head/) { # give headers their own line
		$end = $end->ends_line ? undef : $end->get_offset ;
		$buffer->insert($start, "\n") unless $start->starts_line;
		$buffer->insert($buffer->get_iter_at_offset($end+1), "\n")
			unless ! defined $end;
	}
}

=item C<apply_link(LINK)>

This method is called by the "Link" button or by the ^L keybinding.
It makes any selected text a link. This link is followed immediatly
if the 'follow_new_link' config option is set.

If LINK is undefined the link target is the same as the selected text.

If no text is selected it calls the "New Link" dialog.

In readonly modus the selected text is regarded as a link and
followed immediatly, but no actual link is made

=cut

sub apply_link {
	my ($self, $link) = @_;
	my $buffer = $self->{buffer};
	my ($start, $end) = $buffer->auto_selection('link');
	
	unless (defined $start) {
		return $self->{app}{settings}{read_only}
			? $self->{app}->goto_page_dialog
			: $self->edit_link_dialog ;
	}

	my $text = $buffer->get_text($start, $end, 0);
	$link = $text unless defined $link;
	return undef if $link =~ /\n/;
	
	unless ($self->{app}{settings}{read_only}) {
		my $bit = $link eq $text;
		$buffer->remove_all_tags($start, $end);
		$self->{htext}->apply_link(($bit ? undef : $link), $start, $end);
		$buffer->set_modified(1);
	}

	my ($type, $l) = $self->{app}{page}->parse_link($link);
	$self->{app}->link_clicked($link)
		if $self->{app}{settings}{read_only}
		or $type eq 'page' and $self->{app}{settings}{follow_new_link};
}

=item C<toggle_bullets()>

If selected text is a bullet list this removes the bullets, else it adds
bullets.

=cut

sub toggle_bullets {
	my ($self, $start, $end) = @_;
	my $buffer = $self->{buffer};
	($start, $end) = $buffer->get_selection_bounds unless defined $start;
	return if !$end or $start == $end;
	
	my $text = $self->{buffer}->get_text($start, $end, 1);
	if ($text =~ /^\s*[\*\x{2022}]\s+/m) { # remove bullets
		$text =~ s/^(\s*)[\*\x{2022}]\s+/$1/mg
	}
	else { # set bullets
		$text =~ s/^(\s*)(\S)/$1\x{2022} $2/mg;
	}

	$buffer->replace_selection($text);
}

=item C<selection_tab()>

Puts a tab before every line of a selection.

=cut

sub selection_tab {
	my ($self, $start, $end) = @_;
	($start, $end) = $self->{buffer}->get_selection_bounds unless defined $start;
	return if !$end or $start == $end;
	
	my $buffer = $self->{buffer};
	my $text = $buffer->get_text($start, $end, 1);
	$text =~ s/^/\t/mg;

	my $verbatim = $self->_is_verbatim($start);
	($start, $end) = $buffer->replace_selection($text);
	$self->{buffer}->apply_tag_by_name('verbatim', $start, $end)
		if $verbatim;
}

=item C<selection_backspace()>

Removes a tab for every line of a selection.

=cut

sub selection_backspace {
	my ($self, $start, $end) = @_;
	my $buffer = $self->{buffer};
	($start, $end) = $buffer->get_selection_bounds unless defined $start;
	return if !$end or $start == $end;

	my $text = $self->{buffer}->get_text($start, $end, 1);
	my $verbatim = $self->_is_verbatim($start);
	if ($text =~ s/^\t//mg) {
		($start, $end) = $buffer->replace_selection($text);
		$self->{buffer}->apply_tag_by_name('Verbatim', $start, $end)
			if $verbatim;
		return 1;
	}

	return 0;
}

sub on_insert_text { # buffer, iter, string, length, self
#	(length($string) == 1)
#		? push(@undo_chars, $string)
#		: 
	$_[4]->add_undo('insert', $_[1]->get_offset - length($_[2]), $_[2]);
}

sub on_delete_range { # buffer, begin, end, self
	#print "delete range\n";
	my $string = $_[0]->get_text($_[1], $_[2], 0);
	$_[3]->add_undo('delete', $_[1]->get_offset, $string);
}

sub on_apply_tag { pop->_on_change_tag('apply_tag', @_) }

sub on_remove_tag { pop->_on_change_tag('remove_tag', @_) }

sub _on_change_tag {
	my ($self, $action, undef, $tag, $start, $end) = @_;
	my @off = ($start->get_offset, $end->get_offset);
	if ($tag->{is_link}) {
		$self->add_undo($action, @off, 'L', $tag->{link_data});
	}
	else {
		$self->add_undo($action, @off, $tag->get_property('name'));
	}
}

=item C<add_undo(ACTION, OFFSET, DATA)>

=cut

sub add_undo {
	my $self = shift;
	return if $self->{undo_lock}; # prohibit unwanted recursion
#	flush_undo_chars() if @undo_chars;
	my ($action, $offset, @data) = @_;
#	print "do: $action \@$offset: >>@data<<\n";
	push @{$self->{undo}}, [$action, $offset, @data];
	shift @{$self->{undo}} if @{$self->{undo}} > $self->{app}{settings}{undo_max};
	@{$self->{redo}} = ();
}

#sub flush_undo_chars {
#	return unless @undo_chars;
#	add_undo('insert', 
#}

=item C<undo()>

Undo one editing step in the buffer.

=cut

sub undo {
	my $self = shift;
	my ($undo, $redo) = @{$self}{'undo', 'redo'};
	return unless @$undo;
	my $step = pop @$undo;
	unshift @$redo, [@$step]; # force copy;
	$$step[0] = $UNDO_STEPS{$$step[0]};
	$self->_do_step(@$step);
}

=item C<redo()>

Redo one editing step in the buffer.

=cut

sub redo {
	my $self = shift;
	my ($undo, $redo) = @{$self}{'undo', 'redo'};
	return unless @$redo;
	my $step = shift @$redo;
	push @$undo, $step;
	$self->_do_step(@$step);
}

sub _do_step {
	my ($self, $action, $offset, @data) = @_;
	my $buffer = $self->{buffer};
	my $start = $buffer->get_iter_at_offset($offset);
	$self->{undo_lock} = 1;
	if ($action eq 'insert') {
		$buffer->insert($start, $data[0]);
		$buffer->place_cursor(
			$buffer->get_iter_at_offset($offset + length($data[0])));
	}
	elsif ($action eq 'delete') {
		my $end = $buffer->get_iter_at_offset($offset + length($data[0]));
		$buffer->delete($start, $end);
		$buffer->place_cursor($start);
	}
	elsif ($action eq 'apply_tag') {
		my $end = $buffer->get_iter_at_offset( $data[0] );
		$buffer->remove_all_tags($start, $end);
		if ($data[1] eq 'link') {
			$self->{htext}->apply_link($data[2], $start, $end);
		}
		else { $buffer->apply_tag_by_name($data[1], $start, $end) }
	}
	elsif ($action eq 'remove_tag') {
		my $end = $buffer->get_iter_at_offset( $data[0] );
		$buffer->remove_all_tags($start, $end);
	}
	$buffer->set_modified(1);
	$self->{htext}->scroll_mark_onscreen( $buffer->get_insert );
	$self->{undo_lock} = 0;
}

#sub flush_undo_mini {
#	return unless @undo_mini;
#	my (prev,
#	for (@undo_mini) {
#		my ($action, $offset, $char) = @$_;
#		
#}

=item C<edit_link_dialog(ITER)>

This dialog allows the user to create a link for which
the link target and the link text differ.

If ITER is undefined the cursor is used.

=cut

sub edit_link_dialog {
	my ($self, $iter) = @_;
	my $buffer = $self->{buffer};

	# handle selections
	my ($start, $end) = $buffer->auto_selection('link', $iter);

	my ($text, $link) = ('', '');
	if (defined $start) {
		$link = $self->{htext}->get_link_at_iter($start);
		$text = $self->{buffer}->get_text($start, $end, 0);
		$text = undef if $text =~ /\n/;
	}

	# Run the dialog
	my $val = $self->{app}->run_prompt(
		(defined($start) ? 'Edit Link' : 'Insert Link'),
		['txt', 'link'], {
			txt  => ['Text', 'string', $text],
			link => ['Links to', 'string', $link],
		}, $LINK_ICON, '_Link' )
		or return;
	($text, $link) = @$val;
	return unless $text =~ /\S/ or $link =~ /\S/;
	
	# both entries default to the other
	$link = $text unless $link =~ /\S/;
	$text = $link unless $text =~ /\S/;

	# use delete + insert instead of apply because the text can be different
	if (defined $start) {
		$buffer->delete($start, $end);
	}
	else {
		$start = $buffer->get_iter_at_mark( $buffer->get_insert());
	}
	my $bit = $link eq $text;
	my $_start = $start->get_offset;
	$buffer->insert($start, $text);
	$start = $buffer->get_iter_at_offset($_start);
	$end = $start->copy;
	$end->forward_chars(length $text);
	$buffer->remove_all_tags($start, $end); # because of this we can't use htext->insert_link()
	$self->{htext}->apply_link(($bit ? undef : $link), $start, $end);
	# FIXME redundant code from "apply_link" here
}

=item C<edit_image_dialog()>

=cut

sub edit_image_dialog { # TODO add 'browse' button
	my ($self, $iter) = @_;
	my $pixbuf = $iter->get_pixbuf;
	return unless $pixbuf and defined $pixbuf->{image_src};
	
	my $src = $pixbuf->{image_src};
	my $val = $self->{app}->run_prompt(
		'Edit Image Source',
		['src'], {src => ['Source', 'file', $src]},
		$LINK_ICON, '_Link' )
		or return;

	my ($new_src) = @$val;
	return unless $new_src =~ /\S/ and $new_src ne $src;

	my $buffer = $self->{buffer};
	my $end = $iter->copy;
	$end->forward_char;
	$buffer->delete($iter, $end);
	$buffer->place_cursor($iter);
	$self->{buffer}->insert_image_from_file($new_src);
}

1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

=cut

