package Zim::GUI;

use strict;
use vars qw/$AUTOLOAD %Config/;
use Carp;
use File::BaseDir qw/
	xdg_config_home xdg_config_files
	xdg_data_home xdg_data_dirs xdg_data_files
	xdg_cache_home /;
use File::MimeInfo::Magic;
use Gtk2;
use Gtk2::Gdk::Keysyms;
use Gtk2::SimpleList;
use Zim::File;
use Zim::History;
use Zim::GUI::PathBar;
use Zim::GUI::PageView;

eval "use File::MimeInfo::Applications";
my $has_mimeapplications = $@ ? 0 : 1;
warn "No 'File::MimeInfo::Applications' found, disabling application bindings\n"
	unless $has_mimeapplications;

our $VERSION = '0.16';
our $LONG_VERSION = << "EOT";
zim $VERSION - A desktop wiki and outliner

Copyright (c) 2006 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

Please report bugs to pardus\@cpan.org
EOT

our ($ICON) = xdg_data_files('pixmaps', 'zim.png');
warn "WARNING: Could not find 'zim.png', is \$XDG_DATA_DIRS set properly ?\n"
	unless length $ICON;

=head1 NAME

Zim::GUI - The application object for zim

=head1 SYNOPSIS

	use Zim::GUI;
	use Zim::Repository;
	
	my $zim = Zim::GUI->new(\%SETTINGS);
	
	my $repository = Zim::Repository->new($DIR);
	$zim->set_repository($repository);
	
	$zim->gui_init;
	$zim->gui_show;
	
	Gtk2->main;
	
	exit;

=head1 DESCRIPTION

This is developer documentation, for the user manual try
executing C<zim --doc>. For commandline options see L<zim>(1).

This module provides the application object for the Gtk2 application B<zim>.
The application has been split into several components for which the 
modules can be found in the Zim::GUI:: namespace. This object brings
together these components and manages the settings and data objects.

=head1 EXPORT

None by default.

=head1 METHODS

Undefined methods are AUTOLOADED either to an component or to the main
L<Gtk2::Window> object.

=over 4

=cut

our $LINK_ICON = Gtk2->CHECK_VERSION(2, 6, 0) ? 'gtk-connect' : 'gtk-convert';
	# gtk-connect stock item was introduced in 2.6.0

our %CONFIG = (  # Default config values
	pane_pos	=> 120,
	pane_vis	=> 0,
	statusbar_vis	=> 1,
	toolbar_vis	=> 1,
	pathbar_type	=> 'recent',
	#icon_size	=> 'large-toolbar',
	width		=> 500,
	height		=> 350,
	x		=> undef,
	y		=> undef,
	default_root	=> undef,
	default_home	=> 'Home',
	browser		=> undef,
	file_browser	=> undef,
	email_client	=> undef,
	date_string	=> '%A %d/%m/%Y',
	hist_max	=> 20,
	undo_max	=> 50,
	follow_new_link => 1,
	backsp_unindent => 1,
	use_xdg_cache	=> 0,
	show_spell      => 1,
	spell_language	=> undef,
	cal_namespace	=> ':Date:',
	show_cal	=> 0,
	plugins		=> '',
	use_camelcase	=> 1,
	use_utf8_ent	=> 1,
	use_autolink	=> 0,
	textfont	=> undef,
);

our %DEFAULTS = (
	file_browser => q/rox '%s'/,
	browser => q/firefox '%s'/,
	email_client => q/thunderbird '%s'/,
);

my @ui_actions = (
	# name,          stock id,  label
	[ 'FileMenu',    undef,     '_File'         ],
	[ 'EditMenu',    undef,     '_Edit'         ],
	[ 'ViewMenu',    undef,     '_View'         ],
	[ 'SearchMenu',  undef,     '_Search'       ],
	[ 'FormatMenu',  undef,     'For_mat'       ],
	[ 'ToolsMenu',   undef,     '_Tools'        ],
	[ 'GoMenu',      undef,     '_Go'           ],
	[ 'HelpMenu',    undef,     '_Help'         ],
	[ 'PathBarMenu', undef,     'P_athbar type' ],
	# name,       stock id,         label,          accelerator,   tooltip
	[ 'OpenRep',  'gtk-open',       '_Open...',     '<ctrl>O',     'Open',         \&on_OpenRep  ],
	[ 'Save',     'gtk-save',       '_Save',        '<ctrl>S',     'Save page',    \&on_Save     ],
	[ 'SaveCopy', 'gtk-save-as',    'S_ave A Copy...',   '<shift><ctrl>S', 'Save a copy',    \&on_SaveCopy ],
	[ 'Export',   undef,            'E_xport...',      undef,         'Export',       \&on_Export   ],
	[ 'EmailPage', undef,		'_Send to...',	'<ctrl>M',	'Send to...',	\&on_EmailPage ],
	[ 'RenamePage', undef,		'_Rename page...', 'F2',          'Rename page',  \&on_RenamePage ],
	[ 'DeletePage', undef,		'_Delete page', undef,		'Delete page',  \&on_DeletePage ],
	[ 'Close',    'gtk-close',      '_Close',       '<ctrl>W',     'Close window', \&on_Close    ],
	[ 'Quit',     'gtk-quit',       '_Quit',        '<ctrl>Q',     'Quit',         \&on_Quit     ],
	[ 'Search',   'gtk-find',       '_Search...',	'<shift><ctrl>F', 'Search',    \&on_Search   ],
	[ 'SearchBL', undef,            'Search _Backlinks...',	undef, 'Search Back links',    \&on_SearchBL   ],
	[ 'Prefs',    'gtk-preferences','Pr_eferences', undef,         undef,          \&on_Prefs    ],
	[ 'Reload',   'gtk-refresh',    '_Reload',      '<ctrl>R',     'Reload',       \&on_Reload   ],
	[ 'OpenDir',  'gtk-open',       'Open _folder', undef,         'Open folder',  \&on_OpenDir  ],
	[ 'RBIndex',  undef,            'Re-build index', undef,       undef,          \&on_RBIndex   ],
	[ 'Back',     'gtk-go-back',    '_Back',        '<alt>Left',   'Go back',      \&on_Back     ],
	[ 'Forward',  'gtk-go-forward', '_Forward',     '<alt>Right',  'Go forward',   \&on_Forward  ],
	[ 'RBack',    undef,            '_Back',        '<alt>Page_Down', undef,       \&on_RBack    ],
	[ 'RForward', undef,            '_Forward',     '<alt>Page_Up',   undef,       \&on_RForward ],
	[ 'Parent',   'gtk-go-up',      '_Parent',      '<alt>Up',     'Go to parent', \&on_Parent   ],
	[ 'Child',    'gtk-go-down',    '_Child',       '<alt>Down',   'Go to child',  \&on_Child    ],
	[ 'Home',     'gtk-home',       '_Home',        '<alt>Home',   'Go home',      \&on_Home     ],
	[ 'JumpTo',   'gtk-jump-to',    '_Jump to...',  '<ctrl>J',     'Jump to page', \&on_JumpTo   ],
	['Today',     undef,        'To_day',      '<alt>D',         'Today',         \&on_Today ],
	[ 'Help',     'gtk-help',       '_Contents',        'F1',          'Help contents',         \&on_Help     ],
	[ 'About',    'gtk-about',      '_About',       undef,         'About',        \&on_About    ],
);

my @ui_toggle_actions = (
	# name,         stock id,    label,        accelerator, tooltip 
	[ 'TToolBar',   undef,       '_Toolbar',   undef,       'Show toolbar',   \&on_TToolBar,   0],
	[ 'TStatusBar', undef,       '_Statusbar', undef,       'Show statusbar', \&on_TStatusBar, 0],
	[ 'TPane',      'gtk-index', 'Side _Pane', 'F9',        'Show side pane', \&on_TPane,      0],
	['TCalendar',   'gtk-paste', 'Calen_dar',  '<ctrl><shift>D', 'Show calendar', \&on_TCalendar, 0],
);

my @ui_radio_actions = (
	# name,          stock id, label,           accelerator, tooltip, value
	[ 'PBRecent',    undef,    '_Recent pages', undef,       '',      0],
	[ 'PBHistory',   undef,    '_History',      undef,       '',      1],
	[ 'PBNamespace', undef,    '_Namespace',    undef,       '',      2],
	[ 'PBHidden',    undef,    'H_idden',       undef,       '',      3],
);

my %pathbar_map = ( 'recent' => 0, 'history' => 1, 'namespace' => 2, 'hidden' => 3);

my $ui_layout = q{<ui>
	<menubar name='MenuBar'>
		<menu action='FileMenu'>
			<menuitem action='OpenRep'/>
			<separator/>
			<menuitem action='Save'/>
			<menuitem action='SaveCopy'/>
			<menuitem action='Export'/>
			<separator/>
			<placeholder name='PrintActions'></placeholder>
			<menuitem action='EmailPage'/>
			<separator/>
			<placeholder name='FileMods'></placeholder>
			<separator/>
			<menuitem action='Close'/>
			<menuitem action='Quit'/>
		</menu>
		<menu action='EditMenu'>
			<placeholder name='EditPage'></placeholder>
			<separator/>
			<placeholder name='InsertItems'></placeholder>
			<separator/>
			<menuitem action='Prefs'/>
		</menu>
		<menu action='ViewMenu'>
			<menuitem action='TToolBar'/>
			<menuitem action='TStatusBar'/>
			<menuitem action='TPane'/>
			<menu action='PathBarMenu'>
				<menuitem action='PBRecent'/>
				<menuitem action='PBHistory'/>
				<menuitem action='PBNamespace'/>
				<menuitem action='PBHidden'/>
			</menu>
			<separator/>
			<menuitem action='TCalendar'/>
			<placeholder name='PluginItems'></placeholder>
			<separator/>
			<menuitem action='Reload'/>
		</menu>
		<menu action='FormatMenu'></menu>
		<menu action='SearchMenu'>
			<placeholder name='FindItems'></placeholder>
			<separator/>
			<menuitem action='Search'/>
			<menuitem action='SearchBL'/>
		</menu>
		<menu action='ToolsMenu'>
			<menuitem action='OpenDir'/>
			<menuitem action='RBIndex'/>
		</menu>
		<placeholder name='PluginMenus'></placeholder>
		<menu action='GoMenu'>
			<menuitem action='Back'/>
			<menuitem action='Forward'/>
			<menuitem action='Parent'/>
			<separator/>
			<menuitem action='Today'/>
			<placeholder name='PluginItems'></placeholder>
			<separator/>
			<menuitem action='Home'/>
			<menuitem action='JumpTo'/>
		</menu>
		<menu action='HelpMenu'>
			<menuitem action='Help'/>
			<menuitem action='About'/>
		</menu>
	</menubar>
	<toolbar name='ToolBar'>
		<placeholder name='File'></placeholder>
		<separator/>
		<placeholder name='Edit'></placeholder>
		<separator/>
		<placeholder name='View'>
			<toolitem action='TPane'/>
		</placeholder>
		<separator/>
		<placeholder name='Go'>
			<toolitem action='Home'/>
			<toolitem action='Back'/>
			<toolitem action='Forward'/>
		</placeholder>
		<separator/>
		<placeholder name='Search'></placeholder>
		<separator/>
		<placeholder name='Format'></placeholder>
		<separator/>
		<toolitem action='TCalendar'/>
		<placeholder name='Tools'></placeholder>
	</toolbar>
	<accelerator action='Child'/>
	<accelerator action='RBack'/>
	<accelerator action='RForward'/>
</ui>};
my $rw_ui_layout = q{<ui>
<menubar name='MenuBar'>
		<menu action='FileMenu'>
			<placeholder name='FileMods'>
				<menuitem action='RenamePage'/>
				<menuitem action='DeletePage'/>
			</placeholder>
		</menu>
</menubar>
</ui>};

our $CURRENT;

=item C<new(SETTINGS)>

Simple constructor.

=cut

sub new {
	my ($class, $settings) = @_;
	
	%$settings = (%CONFIG, %$settings); # set defaults
	$$settings{root} ||= $$settings{default_root};
	$$settings{home} ||= $$settings{default_home};
	my $self = bless {settings => $settings}, $class;
	$self->load_config;

	$self->{settings}{pathbar_type} = 'recent'
		if $self->{settings}{pathbar_type} eq 'trace';
		# reset 0.08 default value

	$self->{_message_timeout} = -1;
	$self->{_save_timeout} = -1;
	
	$self->{icon_file} = $ICON;
	
	return $self;
}

sub DESTROY {
	my $self = shift;
	for (qw/_save_timeout _message_timeout/) {
		my $timeout = $self->{$_};
		Glib::Source->remove($timeout) if $timeout >= 0;
	}
}

sub AUTOLOAD {
	my $self = shift;
	$AUTOLOAD =~ s/^.*:://;
	return if $AUTOLOAD eq 'DESTROY';
	croak "No such method: Zim::$AUTOLOAD" unless ref $self;
	#warn join ' ', "Zim::AUTOLOAD called for $AUTOLOAD by: ", caller, "\n";
	return $self->{objects}{$AUTOLOAD}
		if exists $self->{objects}{$AUTOLOAD};
		
	if (exists $self->{autoload}{$AUTOLOAD}) {
		# Delayed initialization of components
		#warn "Autoloading: $AUTOLOAD\n";
		my $obj = delete $self->{autoload}{$AUTOLOAD};
		eval {
			if (ref($obj) eq 'CODE') { $obj = $obj->(app => $self) }
			else { # expect $load to be a class
				eval "require $obj";
				die $@ if $@;
				$obj = $obj->new(app => $self);
			}
		};
		$self->exit_error($@) if $@;
		$self->{objects}{$AUTOLOAD} = $obj;
		return $obj
	}
	
	# Call method on default widget
	return $self->{window}->$AUTOLOAD(@_);
}

=item C<set_repository(REPOSITORY)>

Set the repository object.

=cut

sub set_repository {
	my ($self, $repository) = @_;
	
	my $dir = $repository->{dir}; # FIXME what if there is no dir
	unless ($self->{name}) { # needed for trayicon tooltip
		my @dirs = grep length($_),
			File::Spec->splitdir($dir);
		$self->{name} = pop @dirs;
	}

	if (my $icon = $repository->config->{icon}) {
		my $file = File::Spec->rel2abs($icon, $dir);
		($file) = xdg_data_files('pixmaps', $icon) unless -f $file;
		$self->{icon_file} = $file if defined $file and -f $file;
	}

	$self->{repository} = $repository;
	$repository->{date_string} = $self->{settings}{date_string};
		# Tmp hack till more template support is written
}

=item C<gui_init()>

This method initializes all GUI objects that make up the application.

=cut

sub gui_init {
	my $self = shift;
	
	## Setup the window
	my $window = Gtk2::Window->new('toplevel');
	$window->set_default_size(@{$self->{settings}}{'width', 'height'});
	$window->signal_connect_swapped(delete_event => \&on_delete_event, $self);
	$window->signal_connect(destroy => sub { Gtk2->main_quit });
	$window->set_icon(
		Gtk2::Gdk::Pixbuf->new_from_file($self->{icon_file}) );
	$window->set_title('Zim');
	$self->{window} = $window;

	$SIG{'USR1'} = sub { # toggle window on `kill -USR1`
		my $iconified = grep {$_ eq 'iconified'}
			$window->window->get_state;
		$iconified ? $window->present : $window->iconify ;
	}; # defined _before_ plugins are loaded - so TrayIcon can change this
	
	#$window->signal_connect(hide => sub { # FIXME does not work - patched in TrayIcon
	#		@{$self->{settings}}{'x', 'y'} = $window->get_position;
	#		warn "hiding window: @{$self->{settings}}{'x', 'y'}\n";
	#	} );
	$window->signal_connect(show => sub {
			my ($x, $y) = @{$self->{settings}}{'x','y'};
			#warn "showing window: $x, $y\n";
			$window->move($x,$y) if defined($x) and defined ($y);
		} );
			

	my $vbox = Gtk2::VBox->new(0, 0);
	$window->add($vbox);
	$self->{vbox} = $vbox;

	## Setup actions and ui for menubar and toolbar
	my $actions = Gtk2::ActionGroup->new("Actions");
	$actions->add_actions(\@ui_actions, $self);
	$actions->add_toggle_actions(\@ui_toggle_actions, $self);
	$actions->add_radio_actions(\@ui_radio_actions,
		$pathbar_map{$self->{settings}{pathbar_type}}, \&on_TPathBar, $self);
	#$actions->get_action('Save')->set_sensitive(0) if $self->{settings}{read_only};
	#$actions->get_action('Back')->set_sensitive(0);
	#$actions->get_action('Forward')->set_sensitive(0);
	#$actions->get_action('Parent')->set_sensitive(0);
	_gtk_action_set_sensitive($actions->get_action('Save'), 0)
		if $self->{settings}{read_only};
	_gtk_action_set_sensitive($actions->get_action('Back'), 0);
	_gtk_action_set_sensitive($actions->get_action('Forward'), 0);
	_gtk_action_set_sensitive($actions->get_action('Parent'), 0);
	_gtk_action_set_sensitive($actions->get_action('EmailPage'), 0);
	$self->{actions} = $actions;
	
	my $ui = Gtk2::UIManager->new;
	$ui->insert_action_group($actions, 0);
	$window->add_accel_group($ui->get_accel_group);
	$ui->add_ui_from_string($ui_layout);
	$ui->add_ui_from_string($rw_ui_layout) unless $self->{settings}{read_only};
	$self->{menubar} = $ui->get_widget("/MenuBar");
	$vbox->pack_start($self->{menubar}, 0,1,0);
	$self->{toolbar} = $ui->get_widget("/ToolBar");
	$vbox->pack_start($self->{toolbar}, 0,1,0);
	$self->{toolbar}->hide;
	$self->{toolbar}->set_no_show_all(1);
	$self->{ui} = $ui;

	## General window layout
	my $hpaned = Gtk2::HPaned->new();
	$hpaned->set_position($self->{settings}{pane_pos});
	$vbox->pack_start($hpaned, 1,1,0);
	$self->{hpaned} = $hpaned;

	my $l_vbox = Gtk2::VBox->new(0, 0);
	$hpaned->add1($l_vbox);
	$l_vbox->set_no_show_all(1);
	$self->{l_vbox} = $l_vbox;

	my $r_vbox = Gtk2::VBox->new(0, 0);
	$hpaned->add2($r_vbox);
	$self->{r_vbox} = $r_vbox;

	my $shbox = Gtk2::HBox->new(0, 2);
	$shbox->set_no_show_all(1);
	$vbox->pack_start($shbox, 0,1,0);
	$self->{statusbar} = $shbox;
	
	## Status bar
	my $s1 = Gtk2::Statusbar->new;
	$s1->set_has_resize_grip(0);
	$shbox->pack_start($s1, 1,1,0);
	$self->{status1} = $s1;

	my $frame = Gtk2::Frame->new();
	$frame->set_shadow_type('in');
	$s1->pack_end($frame, 0,1,0);
	my $ebox = Gtk2::EventBox->new();
	$frame->add($ebox);
	my $l1 = Gtk2::Label->new();
	$l1->set_size_request(120,10);
	$l1->set_alignment(0, 0.5);
	$l1->set_mnemonic_widget( ($actions->get_action('SearchBL')->get_proxies)[0] );
	$ebox->add($l1);
	$self->{statusl} = $l1;
	
	my $s2  = Gtk2::Statusbar->new;
	$s2->set_size_request(80,10);
	$shbox->pack_end($s2, 0,1,0);
	$self->{status2} = $s2;
	
	$ebox->signal_connect_swapped(button_press_event => \&on_click_backlinks, $self);

	## Build side pane
	$self->{autoload}{TreeView} = sub {
		my %args = @_;
		eval "require Zim::GUI::TreeView";
		die $@ if $@;
		my $tree_view = Zim::GUI::TreeView->new(%args);
		$tree_view->load_index();
		$tree_view->signal_connect(row_activated =>
			sub { $self->toggle_pane(0) if $self->{_pane_visible} == -1 } );
		$l_vbox->add($tree_view->widget);
		return $tree_view;
	};

	## Build content of the main area
	my $path_bar = Zim::GUI::PathBar->new(app => $self);
	$r_vbox->pack_start($path_bar->widget, 0,1,5);
	$self->{objects}{PathBar} = $path_bar;

	my $page_view = Zim::GUI::PageView->new(app => $self);
	$page_view->set_font($self->{settings}{textfont})
		if defined $self->{settings}{textfont};
	$r_vbox->pack_end($page_view->widget, 1,1,0);
	$self->{objects}{PageView} = $page_view;

	# FIXME shouldn't the repository manage this directory ?
	if ($self->{settings}{use_xdg_cache}) {
		my $root = File::Spec->rel2abs($self->{repository}{dir});
		my ($vol, $dirs) = File::Spec->splitpath($root, 'NO_FILE');
		my @dirs = File::Spec->splitdir($dirs);
		$self->{settings}{hist_file} ||= File::Spec->catfile( 
			xdg_cache_home, 'zim', join ':', $vol, @dirs );
	}
	else {
		my $root = File::Spec->rel2abs($self->{repository}{dir});
		$self->{settings}{hist_file} ||= File::Spec->catfile($root, '.zim.history'); 
	}
	
	eval {
		my $history = Zim::History->new(
			$self->{settings}{hist_file}, $page_view,
			$self->{settings}{hist_max} );
		$self->{objects}{History} = $history;
	};
	$self->exit_error($@) if $@;

	## Some wiring
	my $accels = $ui->get_accel_group;
	$accels->connect( # Ctrl-Space
		ord(' '), ['control-mask'], ['visible'],
		sub {
			if ($self->{settings}{pane_vis}) {
				my $tree_view = $self->TreeView;
				if ($tree_view->has_focus) { $page_view->grab_focus }
				else                       { $tree_view->grab_focus }
			}
			else {
				my $vis = $self->{_pane_visible} ? 0 : -1;
				$self->toggle_pane($vis);
			}
		} );
	$accels->connect( Gtk2::Accelerator->parse('F5'), ['visible'],
		sub { $self->on_Reload } );

	
	$self->signal_connect('page_loaded', sub {
		my $self = shift;
		my $state = $self->History->get_state;
		#$self->{actions}->get_action('Back')->set_sensitive($$state{back} ? 1:0);
		#$self->{actions}->get_action('Forward')->set_sensitive($$state{forw} ? 1:0);
		_gtk_action_set_sensitive($self->{actions}->get_action('Back'), $$state{back} ? 1:0);
		_gtk_action_set_sensitive($self->{actions}->get_action('Forward'), $$state{forw} ? 1:0);
		my $namespace = ($self->{page}->namespace =~ /[^:]/) ? 1 : 0;
		#$self->{actions}->get_action('Parent')->set_sensitive($namespace);
		_gtk_action_set_sensitive($self->{actions}->get_action('Parent'), $namespace);
	} );
	$self->signal_connect('page_loaded', sub {
		my $self = shift;
		my $has_src = grep {$_ eq 'source'} $self->{page}->interfaces;
		_gtk_action_set_sensitive($self->{actions}->get_action('EmailPage'), $has_src);
	} );

	## List autoloaded components
	$self->{autoload}{$_} = 'Zim::GUI::'.$_
		for qw/SearchDialog FindReplaceDialog Calendar ExportDialog/ ;

	## Load plugins
	$self->plug($_) for grep length($_), split /,/, $self->{settings}{plugins};

	## Try saving the program on system signals
	for my $sig (qw/TERM HUP PIPE/) {
		$SIG{$sig} = sub {
			$self->quit('FORCE');
			$self->exit_error(undef,"Signal $sig caught\n");
		};
	}

	# Try saving on desktop exit
#	$window->get_display->signal_connect( closed => sub {
#			print "Display closed\n";
#		} );
}

sub _gtk_action_set_sensitive { # **sigh**
	my ($action, $bit) = @_;
	if (Gtk2->CHECK_VERSION(2, 6, 0)) { $action->set_sensitive($bit) }
	else { $_->set_sensitive($bit) for $action->get_proxies }
}


=item C<gui_show()>

=cut

sub gui_show {
	my $self = shift;
	
	## Show all widgets .. well most of them
	$self->{window}->show_all;
	$self->{statusbar}->set_no_show_all(0);
	$self->{l_vbox}->set_no_show_all(0);
	$self->{toolbar}->set_no_show_all(0);
#	my ($x, $y) = @{$self->{settings}}{'x','y'};
#	$window->move($x,$y) if defined($x) and defined ($y);
	
	## Find a page to load
	unless ($self->{page}) {
		my $page = $self->{objects}{History}->get_current
			|| $self->{settings}{home};
		$page = $self->{repository}->resolve_page($page) unless ref $page;
		$self->load_page($page);
		# FIXME what if we are read_only and page does not exist ?
		# 	load_apge returns undef
		# 	first try home, but might also not exist
		# 	need page that always exists ... like about or version page
	}
	
	## Toggle some widgets
	$self->{message_lock} = 1;
	my ($actions, $settings) = ($self->{actions}, $self->{settings});
	$actions->get_action('TToolBar')->set_active($settings->{toolbar_vis});
	$actions->get_action('TStatusBar')->set_active($settings->{statusbar_vis});
	$actions->get_action('TPane')->set_active($settings->{pane_vis});
	$actions->get_action('TCalendar')->set_active($settings->{show_cal});
	$self->toggle_path_bar($self->{settings}{pathbar_type});
	$self->{message_lock} = 0;

	$self->{objects}{PageView}->grab_focus;

	## set save interval
	$self->{_save_timeout} = 
		Glib::Timeout->add(5000, sub {$self->save_page; 1})
		unless $self->{settings}{read_only};
	
	$self->signal_emit('gui_show');
}

=item C<quit()>

Exit the main loop.

=cut

sub quit {
	my $self = shift;
	$self->on_delete_event(@_) && return;
	Gtk2->main_quit;
}

sub on_delete_event {
	# Called when the window is deleted
	# return true to avoid destruction
	my $self = shift;
	my $force = uc(shift) eq 'FORCE';

	eval {
		my ($settings, $window) = @{$self}{'settings', 'window'};
		@{$settings}{'width', 'height'} = $window->get_size;
		@{$settings}{'x', 'y'} = $window->get_position if $window->visible;
			# position is 0,0 when window is hidden
	};
	
	if ($self->{settings}{hide_on_delete} and !$force) {
		$self->{window}->hide;
		return 1;
	}
	
	$self->{save_lock} = 0; # just to be sure
	$self->save_page() || return 1;
	$self->save_config;

	unless ($self->{settings}{read_only}) {
		my ($vol, $dirs, undef) =
			File::Spec->splitpath($self->{settings}{hist_file});
		my $dir = File::Spec->catpath($vol, $dirs);
		$self->{objects}{History}->write if -d $dir;
			# Only safe history if directory exists ..
	}

	return 0;
}

=item C<current()>

=cut

sub current { return $CURRENT }

=item C<plug(PLUGIN)>

=item C<unplug(PLUGIN)>

=cut

sub plug { # TODO add plugin to config
	my ($self, $name) = @_;
	return $self->error_dialog("Plugin already loaded: $name")
		if $self->{objects}{$name};
		
	my ($script) = xdg_data_files('zim', 'plugins', $name.'.pl');
	unless (length $script) {
		$self->unplug($name);
		return $self->error_dialog("No such plugin: $name");
	}

	#warn "Loading plugin $name from $script\n";
	local $CURRENT = $self;
	my $return = do $script;
	if ($@ or ! defined $return) {
		$self->unplug($name);
		return $self->error_dialog("Failed to load $name plugin", $@ || $!);
	}

	return 1;
}

sub unplug {
	my ($self, $name) = @_;
	$self->{settings}{plugins} =~ s/\Q$name\E,*//;
	$self->del_object($name);
}

=item C<add_object(NAME => OBJECT)>

=item C<del_object(NAME)>

=cut

sub add_object {
	my ($self, $name, $obj) = @_;
	croak "Object '$name' already exists" if $self->{objects}{$name};
	$self->{objects}{$name} = $obj;
	return 1;
}

sub del_object { return delete ${$_[0]{objects}}{$_[1]} }

sub on_click_backlinks {
	my ($self, $event) = @_;
	return if $event->type ne 'button-press';
	
	if ($event->button == 1) {
		if ($self->SearchDialog->{dialog}) { $self->SearchDialog->hide }
		else { $self->on_SearchBL }
	}
	return unless $event->button == 3;
	
	my @blinks = $self->{page}->_list_backlinks;
	return unless @blinks;
	
	my $menu = Gtk2::Menu->new();
	for my $l (@blinks) {
		my $item = Gtk2::ImageMenuItem->new_with_label($l);
		$item->signal_connect(activate =>
			sub { $self->link_clicked($l) }  );
		$menu->add($item);
	}

	$menu->show_all;
	
	my ($button, $time) = $event
		? ($event->button, $event->time)
		: (0, 0) ;
	$menu->popup(undef, undef, undef, undef, $button, $time);
}

sub on_OpenRep {
	my $self = pop;
	$self->prompt_repository_dialog(
		sub { $self->exec_new_window('--name', @_) } );
}

sub on_Save {
	my $self = pop;
	$self->save_page('FORCE');
	$self->save_config;
	$self->History->write;
}

sub on_SaveCopy { pop->save_copy }

sub on_Export { pop->ExportDialog->show() }

sub on_RenamePage { pop->rename_page() }

sub on_DeletePage { pop->delete_page() }

sub on_Close { pop->quit() }

sub on_Quit { pop->quit('FORCE') }

sub on_Search { pop->SearchDialog->show }

sub on_SearchBL {
	my $self = pop;
	$self->SearchDialog->search($self->{page}->name);
}

sub on_Prefs { pop->preferences_dialog }

sub on_Reload { pop->reload }

sub on_OpenDir {
	my $self = pop;
	my $dir = $self->{page}->properties->{base};
	return unless length $dir;
	#warn "Opening $dir\n";
	$self->link_clicked($dir);
}

sub on_EmailPage {
	# TODO do _not_ use the get_source interface here, let PageView convert to plain text !
	# Using this interface here may screw up mtime detection
	my $self = pop;
	my $subject = $self->{page}->name;
	my $io = $self->{page}->get_source('r');
	my $src = join '', <$io>;
	$src =~ s/(\W)/sprintf '%%%02x', ord $1/eg; # url encoding
	$self->open_url("mailto:?subject=$subject&body=".$src);
}

sub on_RBIndex {
	my $self = pop;
	$self->{repository}->_flush_cache;
	my $tree = $self->TreeView;
	$tree->{_loaded} = 0;
	$tree->load_index;
	$self->reload;
}

sub on_TToolBar {
	my $self = pop;
	my $vis = $self->{toolbar}->visible ? 0 : 1;
	$vis ? $self->{toolbar}->show_all : $self->{toolbar}->hide_all ;
	$self->{settings}{toolbar_vis} = $vis;
}

sub on_TStatusBar {
	my $self = pop;
	my $vis = $self->{statusbar}->visible ? 0 : 1;
	$vis ? $self->{statusbar}->show_all : $self->{statusbar}->hide_all ;
	$self->{settings}{statusbar_vis} = $vis;
}

sub on_TPane {
	my $self = pop;
	my $vis = $self->{l_vbox}->visible ? 0 : 1;
	$self->toggle_pane($vis)
}

sub on_TPathBar {
	my $self = pop;
	my $type = lc pop->get_name;
	$type =~ s/^pb//;
	$self->toggle_path_bar($type);
}

sub on_TCalendar { pop->Calendar->on_TCalendar }

sub on_Back { pop->go_back(1) }

sub on_Forward { pop->go_forw(1) }

sub on_RBack {
	my $self = pop;
	my ($i, @recent) = $self->History->get_recent;
	return unless $i > 0;
	my $rec = $recent[$i-1];
	$self->load_page($rec);
}

sub on_RForward {
	my $self = pop;
	my ($i, @recent) = $self->History->get_recent;
	return unless $i < $#recent;
	my $rec = $recent[$i+1];
	$self->load_page($rec);
}

sub on_Parent {
	my $self = pop;
	my $namespace = $self->{page}->namespace;
	return if $namespace eq ':';
	$self->load_page($namespace);
}

sub on_Child {
	my $self = pop;
	my $namespace = $self->History->get_namespace;
	my $name = $self->{page}->name;
	return unless $namespace =~ /^(:*$name:+[^:]+)/;
	$self->load_page($1);
}

sub on_Today { pop->Calendar->on_Today }

sub on_Home {
	my $self = pop;
	my $home = $self->{repository}->resolve_page($self->{settings}{home});
	$self->load_page($home);
}

sub on_JumpTo { pop->jump_page_dialog }

sub on_Help { pop->show_help }

sub on_About { pop->about_dialog }

=item C<widget()>

Returns the root window widget.
Use this widget for things like show_all() and hide_all().

=cut

sub widget { return $_[0]->{window} }

=item C<signal_connect(NAME, CODE, DATA)>

=cut

sub signal_connect {
	my ($self, $signal, $code, @data) = @_;
	$self->{_signals}{$signal} ||= [];
	push @{$self->{_signals}{$signal}}, [$code, @data];
}

=item C<signal_emit(NAME, ARGS)>

=cut

sub signal_emit {
	my ($self, $signal, @args) = @_;
	return unless exists $self->{_signals}{$signal};
	for my $handler (grep defined($_), @{$self->{_signals}{$signal}}) {
		my ($code, @data) = @$handler;
		eval { $code->($self, @args, @data) };
		warn if $@;
	}
}

=item C<load_config>

Read config file.

=cut

sub load_config {
	my $self = shift;
	my $name = $self->{settings}{read_only} ? 'conf.ro' : 'conf.rw' ;
	my ($file) = xdg_config_files('zim', $name);
	return unless defined $file;
	
	my $conf = Zim::File->new($file)->read_config();
	for (keys %$conf) {
		next unless length $$conf{$_};
		$self->{settings}{$_} = $$conf{$_};
	}
}

=item C<save_config>

Save config file.

=cut

sub save_config {
	my $self = shift;
	$self->{settings}{pane_pos} = $self->{hpaned}->get_position;
	
	my $dir  = File::Spec->catdir(xdg_config_home, 'zim');
	my $name = $self->{settings}{read_only} ? 'conf.ro' : 'conf.rw' ;

	my %conf = %{$self->{settings}};
	delete $conf{$_} for grep {! exists $CONFIG{$_}} keys %conf;
	Zim::File->new($dir, $name)->write_config(\%conf);
}

sub _mkdir {
	my ($vol, $dirs) = (@_ == 1)
		 ? (File::Spec->splitpath(shift(@_), 'NO_FILE')) : (@_) ;
	my @dirs = File::Spec->splitdir($dirs);
	my $path = File::Spec->catpath($vol, shift @dirs);
	mkdir $path or die "Could not create dir $path\n"
		if length $path and ! -d $path;
	while (@dirs) {
		$path = File::Spec->catdir($path, shift @dirs);
		mkdir $path or die "Could not create dir $path\n"
			unless -d $path;
	}
}

=item C<link_clicked(LINK)>

Loads a page in zim or opens an external url in a browser.

LINK is considered to be either an page name, an url or a file name.
Page names are resolved as relative links first.
Dispatches to C<open_file()> or C<open_url()> when
LINK is a file or an url.

=cut

sub link_clicked {
	my ($self, $link, $fallback) = @_;
	#warn "link clicked: >>$link<<\n";
	return warn "Warning: You tried to folow an empty link.\n"
		unless length $link;

	my ($type, $l) = $self->{page}->parse_link($link);
	return $self->error_dialog("Link undefined: $link")
	       unless defined $type;
	
	if ($type eq 'page') {
		my $page = $self->{page}->resolve_link($l);
		$page = $self->{repository}->resolve_page($l)
			if $fallback and ! $page and $l =~ ':'; # FIXME
		return $self->error_dialog("Page does not exist:\n$link")
			unless $page;
		$self->load_page($page);
	}
	elsif ($type eq 'file') {
		my $f = Zim::File->localize($l);
		$self->open_file($f);
	}
	elsif ($type eq 'man') { $self->show_help(':man:'.$l) }
	else                   { $self->open_url($l)          }
}

=item C<open_file(FILE)>

Opens FILE with the apropriate program.
Calls C<open_directory()> when FILE is a directory.

=cut

sub open_file {
	my ($self, $file) = @_;
	return $self->open_directory($file) if $file =~ m#/$# or -d $file;
	my $mt = mimetype($file);
	
	return $self->open_directory($file)
		unless $mt and $has_mimeapplications; # FIXME prompt application
	
	my ($app) = mime_applications($mt);

	return $self->open_directory($file) unless $app; # FIXME - idem

	unless (fork) { # child process
		warn 'Executing '.$app->get_value('Name')." on $file\n";
		eval { $app->exec($file) };
		exit 1;
	}
}

=item C<open_directory(DIR)>

Opens DIR in the file browser. Open the parent directory when
DIR turns out to be a file.

=cut

sub open_directory {
	my ($self, $dir) = @_;
	
	if (! -d $dir) { # strip filename if $dir is a file
		my ($vol, $dirs, undef) = File::Spec->splitpath($dir);
		$dir = File::Spec->catpath($vol, $dirs);
	}
	
	my $browser = $self->{settings}{file_browser};
	unless ($browser) { # prompt for a browser
		my $val = $self->run_prompt(
			'Choose file browser',
			['cmd'], {cmd => ['Command', 'string', $DEFAULTS{file_browser}]},
			undef, undef,
			'Please enter a file browser' );
		($browser) = @$val;
		$browser = undef unless $browser =~ /\S/;
		return $self->error_dialog('You have no browser configured')
			unless defined $browser;
		$self->{settings}{file_browser} = $browser;
	}
	
	$browser =~ s/\%s/$dir/ or $browser .= ' '.$dir;
	$self->_exec($browser);
}

=item C<open_url(URL)>

Opens URL in the web browser.

=cut

sub open_url {
	my ($self, $url) = @_;
	
	$url =~ /^(\w[\w\+\-\.]+):/
		or return $self->error_dialog("Not an url: $url");
	my $proto = $1;
	
	if ($proto eq 'zim') { # special case
		$url =~ s#\?(.*)$##;
		my $page = $1;
		$url =~ s#^zim:#file:#i;
		my $dir = Zim::File->parse_uri($url);
		return $self->exec_new_window($dir, $page);
	}
	
	my ($app, $title, $string, $error) = ($proto eq 'mailto')
		? (
			'email_client',
			'Choose Email Client',
			'Please enter an email client',
			'You have no email client configured'
		) : (
			'browser',
			'Choose Browser',
			'Please enter a browser to open external links',
			'You have no browser configured'
		) ;

	my $browser = $self->{settings}{$app};
	$browser ||= $ENV{BROWSER} unless $proto eq 'mailto';
	unless ($browser) { # prompt for a browser
		my $val = $self->run_prompt(
			$title,
			['cmd'], {cmd => ['Command', 'string', $DEFAULTS{$app}]},
			undef, undef,
			$string );
		($browser) = @$val;
		$browser = undef unless $browser =~ /\S/;
		return $self->error_dialog($error)
			unless defined $browser;
		$self->{settings}{$app} = $browser;
	}
	
	$browser =~ s/\%s/$url/ or $browser .= ' '.$url;
	$self->_exec($browser);
}

sub _exec {
	my ($self, @args) = @_;
	warn "Executing: @args\n";
	unless (fork) { # child process
		exec @args;
		exit 1; # just to be sure
	}
}

=item C<go_back(INT)>

Go back one or more steps in the history stack.

=cut

sub go_back {
	my $self = shift;
	my $i = shift || 1;
	my $rec = $self->{objects}{History}->back($i) || return;
	$self->load_page($rec);
}

=item C<go_forw(INT)>

Go forward one or more steps in the history stack.

=cut

sub go_forw {
	my $self = shift;
	my $i = shift || 1;
	my $rec = $self->{objects}{History}->forw($i) || return;
	$self->load_page($rec);
}

=item C<go_recent(NAME)>

Go to the recent page NAME. This is almost identical to C<load_page(NAME)>
but does not change the recent pages stack.

=cut

sub go_recent {
	my ($self, $name) = @_;
	my $rec = $self->{objects}{History}->get_record($name);
	$self->load_page($rec || $name);
}

=item C<reload()>

Save and reload the current page.

=cut

sub reload {
	my $self = shift;
	$self->load_page(
		$self->{objects}{History}->get_current );
}

=item C<load_page(PAGE)>

Loads a new page in the PageView, updates history etc. when
necessary. PAGE should be either an absolute page name, a history record
or a page object.

use C<link_clicked()> for relative page names, urls etc.

=cut

sub load_page {
	my ($self, $page) = @_;
	
	if ($self->{page}) {
		$self->{save_lock} = 0; # just to be sure
		$self->save_page || return;
	}

	goto &_load_page;
}

sub _load_page { # load _without_ saving first
	my ($self, $page) = @_;
	
	my $prevpage = $self->{page};
	
	my ($rec, $name);
	($name, $page, $rec) = 
		(ref($page) eq 'HASH') ? ($page->{name}, undef, $page) :
		ref($page)             ? ($page->name,   $page, undef) : ($page, undef, undef) ;
	my $from_hist = defined $rec;
	
	unless ($page) { # get object
		return warn "Warning: You tried to load an empty name.\n" unless length $name;
		
		eval { $page = $self->{repository}->get_page($name) };
		return $self->error_dialog("Could not load page $name", $@) if $@;
	}
	return $self->error_dialog("Page does not exist:\n$page")
		if ! defined $page
		or $self->{settings}{read_only} and ! $page->exists ;
	
	unless ($rec) { # get history record
		$rec = $self->{objects}{History}->get_record($name);
	}
	
	$self->{objects}{History}->set_current($page, $from_hist);
	eval { $self->{objects}{PageView}->load_page($page) };
	if ($@) {
		$self->exit_error("Could not load page $name", $@);
		# TODO handle more gratiously
	}
	
	$self->{page} = $page;
	$self->{objects}{PageView}->set_state(%$rec) if defined $rec;
	$self->{objects}{History}->delete_recent($prevpage->name)
		unless !defined $prevpage or $prevpage->exists;
	$self->signal_emit('page_loaded', $name);
	
	$self->{window}->set_title("$name - Zim");
	$self->update_status();
	my $links = scalar $page->_list_backlinks;
	$self->{statusl}->set_text_with_mnemonic("  $links _Back links");
}

=item C<save_page(FORCE)>

Check if the current page needs to be saved.
If FORCE is true the page is saved whether it was modified or not.

=cut

# TODO split out to save_page_if_modified() and save_page()
# save_page_if_modified(FORCE) overrules save_lock
# save_page() just saves it, no questions asked
# (maybe call the first one check_save_page() ?)

# TODO part of this logic belongs in the PageView component

sub save_page {
	my ($self, $force) = @_;
	my $modified = $self->{objects}{PageView}->modified;
	return 0 if $self->{app}{settings}{read_only};
	unless ($force) {
		return 1 if ! $modified;
		return 0 if $self->{save_lock};
	}
	
	$self->{save_lock} = 1; # Used to block autosave while "could not save" dialog
	
	my $page = $self->{page};
	my $tree = $self->{objects}{PageView}->get_parse_tree;
	
	eval { # FIXME maybe this logic belongs in PageView
		my $existed = $page->exists;
		if (@$tree > 2) { # content
			$page->set_parse_tree($tree);
			$self->signal_emit('page_created', $page->name) unless $existed;
		}
		else {
			$page->delete;
			$self->signal_emit('page_deleted', $page->name);
		}
	};
	unless ($@) {
		$self->{objects}{PageView}->modified(0);
		$self->signal_emit('page_saved', $page->name);
	}
	else {
		my $error = $@;
		my $answer = $self->prompt_question(
			'Could not save - Zim', 'error',
			"<b>Could not save page</b>\n\n$error",
			['discard', undef, '_Discard changes'],
			['copy', 'gtk-save-as', '_Save a copy...' ],
			2 );
		if ($answer eq 'discard') {
			$self->_load_page($self->{page}); # load page wihtout saving
		}
		elsif ($answer eq 'copy') {
			return $self->save_copy; # recurs inderectly
		}
		else { return 0 }
	}
		
	$self->update_status();
	$self->{save_lock} = 0;
	return 1;
}

=item C<save_copy()>

=cut

sub save_copy {
	my $self = shift;
	return unless $self->{page};

	# prompt page
	my $new = $self->{page}->name . '_(Copy)' ;
	my $val = $self->run_prompt(
		'Save Copy - Zim',
		['page'], {page => ['Page', 'page', $new]},
		'gtk-save-as', '_Save Copy',
		'Please enter a name to save' );
	return unless defined $val;
	($new) = @$val;
	return unless length $new;

	my $page = $self->{page}->resolve_link($new);
	
	# check if page exists
	if ($page->exists()) {
		my $answer = $self->prompt_question(
			'Page exists - Zim', 'warning',
			"Page '$new'\nalready exists.",
			['cancel', 'gtk-cancel', undef],
			['overwrite', undef, '_Overwrite'] );
		return unless $answer eq 'overwrite';
	}
	
	# change page and save buffer
	$self->{page} = $page;
	$self->save_page('FORCE');
	$self->load_page($page->name); # trick into reloading
}

=item C<rename_page(FROM, TO, UPDATE)>

Wrapper for C<< Zim::Repository->move_page() >>.

Move page from FROM to TO. If TO is undefined a dialog is shown to ask for a page name.

UPDATE is a boolean telling to update all links to this page.

=cut

sub rename_page {
	my ($self, $from, $to, $update) = @_;

	$from = $self->{page}->name unless defined $from;
	
	my $move_current = ($from eq $self->{page}->name);
	if ($move_current) { # FIXME clearer interface for this
		$self->{save_lock} = 0 if $move_current;
		$self->save_page or return;
	}
	else { $self->save_page }
	
	unless (defined $to) {
		($to, $update) = $self->prompt_rename_page_dialog($from);
		return unless defined $to;
		$to = $self->{repository}->resolve_page($to);
		# TODO check for things like ':'
	}

	# Get backlinks
	my $rfrom = $move_current ? $self->{page} : $self->{repository}->get_page($from); # FIXME ugly lookup
	my @backlinks = $rfrom->_list_backlinks();
	
	# Move page
	#warn "Move '$from' to '$to'\n";
	my $page;
	eval { $page = $self->{repository}->move_page($from, $to) };
	return $self->error_dialog("Could not rename $from to $to\n\n$@") if $@;
	# FIXME error re-eval page object :(

	if ($update) { # Update backlinks
		my ($dialog, $bar, $label) = $self->_new_progress_bar(
			"Updating links", 'Updating..' );
		my $continue = 1;
		$dialog->signal_connect(response => sub {$continue = 0});
		my $i = 1;
		my $callback = sub {
			my $page = shift;
			$bar->pulse;
			$label->set_text('Updating links in ' . $page);
			while (Gtk2->events_pending) { Gtk2->main_iteration_do(0) }
			return $continue;
		};
		eval {
			for (@backlinks) {
				$callback->($_) or last;
				$self->{repository}->get_page($_)->_update_links(
					'page', $from => $to );
			}
		};
		$self->error_dialog($@) if $@;
		$dialog->destroy;
	}
	
	($from, $to) = map {ref($_) ? $_->name : $_} $from, $to;
	$self->signal_emit('page_renamed', $from => $to);
	$self->load_page($page) if $move_current;
	
	$self->update_status;
	return 1;
}

=item C<delete_page(PAGE)>

Wrapper for C<< Zim::Repository->delete_page >>.
Asks the user for confirmation.

If PAGE is undefined the current page is deleted.

=cut

sub delete_page {
	# FIXME option to delete a complete sub-tree
	# FIXME integrate with the save_page() logic, now we have 
	# redundant code here
	my ($self, $page) = @_;
	$page = $self->{page}->name unless defined $page;
	
	my $name = ref($page) ? $page->name : $page;
	my $delete_current = ($name eq $self->{page}->name);
	$self->save_page; # Mind you, overriding save lock could lock us here
	
	my $answer = $self->prompt_question(
		'Delete page - Zim', 'warning',
		"Are you sure you want to\ndelete '$name' ?",
		['no',  'gtk-cancel', undef],
		['yes', 'gtk-delete', undef]  );
	return unless $answer eq 'yes';
	
	$page = $self->{page} if $delete_current; # make sure we have the right object
	eval { $self->{repository}->delete_page($page) };
	return $self->error_dialog("Could not delete $name\n\n$@") if $@;

	# TODO trigger this with signals
	$self->{objects}{History}->delete_recent($name);
	$self->signal_emit('page_deleted', $name);

	if ($delete_current) {
		eval { $self->{objects}{PageView}->load_page($page) };
		if ($@) {
			$self->exit_error("Could not load page $name\n\n$@");
			# TODO handle more gratiously
			# .. switching read_only would help here
		}
	}

	$self->update_status;
}

=item C<message(STRING)>

Flash a message in the statusbar for a short time.

=cut

sub message {
	my ($self, $str) = @_;
	return if $self->{message_lock};
	my $timeout = $self->{_message_timeout};
	$self->push_status($str, 'message');
	Glib::Source->remove($timeout) if $timeout >= 0;
	$timeout = Glib::Timeout->add(2500, sub {
		$self->pop_status('message');
		$self->{_message_timeout} = -1;
		return 0; # removes timeout
	} );
	$self->{_message_timeout} = $timeout;
}

=item C<update_status()>

Sets the statusbar to display the current page name and some other
information.

=cut

sub update_status {
	my $self = shift;
	my $stat = ' '.$self->{page}->name;
	$stat .= '*' if $self->PageView->modified;
	if ($_ = $self->{page}->status()) { $stat .= '  -  '.uc($_) }
	$stat .= ' [readonly]' if $self->{settings}{read_only};
	$self->push_status($stat, 'page');
}

=item C<push_status(STRING, CONTEXT)>

Put STRING in the status bar.

=cut

sub push_status {
	my ($self, $str, $id) = @_;
	my $statusbar = $self->{status1};
	$id = $statusbar->get_context_id($id);
	$statusbar->pop($id);
	$statusbar->push($id, $str);
}

=item pop_status(CONTEXT)

Removes a string from the status bar.

=cut

sub pop_status {
	my ($self, $id) = @_;
	my $statusbar = $self->{status1};
	$id = $statusbar->get_context_id($id);
	$statusbar->pop($id);
}

=item C<new_button(STOCK, TEXT)>

Creates a button with a stock image but different text.

=cut

sub new_button {
	my ($self, $stock, $text) = @_;
	my $hbox = Gtk2::HBox->new;
	$hbox->pack_start(
		Gtk2::Image->new_from_stock($stock, 'button'), 0,0,0);
	$hbox->pack_start(
		Gtk2::Label->new_with_mnemonic($text), 1,1,0);
	my $button = Gtk2::Button->new();
	$button->add(Gtk2::Alignment->new(0.5,0.5, 0,0));
	$button->child->add($hbox);
	return $button;
}

=item C<new_prompt(TITLE, FIELDS, DATA, BUTTON_STOCK, BUTTON_TEXT, TEXT)>

Generates a dialog asking for one or more fields of input.
Returns the dialog widget and a list with Gtk2::Entry objects.

TITLE is the dialog title.

FIELDS is an array ref giving the order of the input fields.

DATA is a hash ref containing definitions of the input fields.
The key is the name used in FIELDS, the value an array ref with a label text,
a data type and a value.
At the moment only the "string", "page", "file" and "dir" data types are treated special, all other will be ignored silently.

TEXT, BUTTON_STOCK and BUTTON_TEXT are optional.

=cut

sub new_prompt {
	my ($self, $title, $fields, $data, $stock, $string, $text) = @_;
	
	## Setup dialog
	my $dialog = Gtk2::Dialog->new(
		$title, $self->{window},
		[qw/modal destroy-with-parent no-separator/],
		'gtk-cancel'  => 'cancel',
	);
	$dialog->set_resizable(0);
	#$dialog->vbox->set_border_width(12); # FIXME
	$dialog->set_icon($self->{window}->get_icon);
	
	$stock ||= 'gtk-ok';
	my $button = $string
		? $self->new_button($stock, $string)
		: Gtk2::Button->new_from_stock($stock);
	$dialog->add_action_widget($button, 'ok');
	# $dialog->set_default_response('ok'); FIXME

	if (defined $text) {
		my $label = Gtk2::Label->new($text);
		my $align = Gtk2::Alignment->new(0,0.5, 0,0);
		$align->add($label);
		$dialog->vbox->add($align);
	}
	
	## Generate table with input fields
	my $table = Gtk2::Table->new(scalar(@$fields), 3);
	$table->set_border_width(5);
	$table->set_row_spacings(5);
	$table->set_col_spacings(12);
	$dialog->vbox->add($table);
	
	my @entries;
	for my $i (0 .. $#$fields) {
		my @f = @{ $$data{$$fields[$i]} };
		
		my $label = Gtk2::Label->new($f[0].':');
		my $align = Gtk2::Alignment->new(0,0.5, 0,0);
		$align->add($label);
		
		my $entry = Gtk2::Entry->new();
		$entry->set_text($f[2]) if defined $f[2];
		$entry->signal_connect(
			activate => sub { $dialog->response('ok') } );
		push @entries, $entry;
		
		$table->attach_defaults($align, 0,1, $i,$i+1);
		$table->attach_defaults($entry, 1,2, $i,$i+1);

		if ($f[1] eq 'file' or $f[1] eq 'dir') {
			my $is_dir = ($f[1] eq 'dir');
			my $button = $self->new_button('gtk-open', 'Browse');
			$button->signal_connect( clicked => sub {
				my $val = $self->filechooser_dialog(
					$entry->get_text(), $is_dir );
				$entry->set_text($val);
			} );
			$table->attach_defaults($button, 2,3, $i,$i+1);
		}
		elsif ($f[1] eq 'page') {
			$self->set_page_completion($entry);
		}
	}
	
	$dialog->show_all;
	return $dialog, \@entries;
}

=item C<set_page_completion(ENTRY)>

Attach page completions code to a L<Gtk2::Entry> object.

=cut

sub set_page_completion {
	my ($self, $entry) = @_;
	my $completion = Gtk2::EntryCompletion->new;
	my $model = Gtk2::ListStore->new('Glib::String');
	$completion->set_model($model);
	$completion->set_text_column(0);
	$completion->set_inline_completion(1)  if Gtk2->CHECK_VERSION(2, 6, 0);
	$entry->set_completion($completion);
	$entry->signal_connect(changed => \&_update_completion, $self);
}

sub _update_completion {
	my ($entry, $self) = @_;
	my $ns = $entry->get_text;
	$ns =~ s/[^:]+$//;
	return if defined $entry->{_ns} and $entry->{_ns} eq $ns;
	$entry->{_ns} = $ns;
	
	my $_ns = length($ns)
		? $self->{repository}->resolve_namespace($ns)
		: $self->{page}->namespace() ;
		#warn "Complete namespace: $_ns\n";
	
	my $model = $entry->get_completion->get_model;
	$model->clear;
	for ($self->{repository}->list_pages($_ns)) {
		s/_/ /g;
		my $iter = $model->append();
		$model->set($iter, 0 => $ns.$_);
		#warn "Appended: $ns$_\n";
	}
}

=item C<run_prompt(..)>

Wrapper around C<new_prompt()> that runs the dialog and
returns a list with input values. Returns undef on 'cancel'.

=cut

sub run_prompt {
	my $self = shift;
	my ($dialog, $entries) = $self->new_prompt(@_);

	my $values = ($dialog->run eq 'ok')
		? [map $_->get_text, @$entries]
		: undef ;
	$dialog->destroy;

	return $values;
}

=item toggle_pane(BOOLEAN)

Show or hide the side pane with the index tree.
If BOOLEAN is "-1" the pane will only be toggled on untill a page is selected.

=cut

sub toggle_pane { # TODO does not update action state :(
	my ($self, $pane_visible) = @_;
	$self->{_pane_visible} = $pane_visible;
	my $widget = $self->{l_vbox};
	my $hpaned = $self->{hpaned};
	if ($pane_visible) {
		$hpaned->set_position($self->{settings}{pane_pos});
		my $tree_view = $self->TreeView; # possibly autoloaded here
		$widget->show_all;
		$tree_view->grab_focus;
	}
	else {
		$self->{settings}{pane_pos} = $hpaned->get_position();
		$widget->hide_all;
		$self->{objects}{PageView}->grab_focus;
	}
	$self->{settings}{pane_vis} = $pane_visible
		unless $pane_visible == -1;
}

=item C<toggle_path_bar(TYPE)>

Set or cycle the type of the pathbar.

=cut

sub toggle_path_bar {
	my $self = shift;
	my $type = $self->{settings}{pathbar_type};
	$type = shift || (
		($type eq 'recent'   ) ? 'history'   :
		($type eq 'history'  ) ? 'namespace' :
		($type eq 'namespace') ? 'hidden'    : 'recent' );
	$self->{settings}{pathbar_type} = $type;
	
	my $path_bar = $self->{objects}{PathBar};
	if (grep {$type eq $_} qw/recent history namespace/) {
		$path_bar->widget->show_all;
		$path_bar->set_type($type);
	}
	elsif ($type eq 'hidden') {
		$path_bar->widget->hide_all;
		$path_bar->clear_items;
	}
	else { warn "unknown pathbar_type: $type\n" }

	$self->message("Pathbar type: $type");
}



# TODO the various dialogs could be out-sourced using Autoloader
# also a lot of code could be reduced by haveing a generic
# _prompt_dialog(title => ... text => ... fields => { })

=item C<exit_error(ERROR)>

Like C<error_dialog> but exits afterwards.

=cut

sub exit_error {
	my $self = shift;
	my ($text1, $text2) = @_;
	if (defined $text1) { $self->error_dialog($text1, $text2) }
	else {
		$text2 ||= "Unknown error";
		warn "zim: $text2\n";
	}
	unlink $self->{pidfile} if defined $self->{pidfile};
	exit 1;
}

=item C<error_dialog(ERROR)>

This method is used to display errors.

=cut

sub error_dialog {
	my ($self, $text1, $text2) = @_;
	$text2 ||= $@ || $text1;
	warn "zim: $text2\n";
	my $window = $self->{window};
	$window = undef unless defined $window and $window->visible;
		# window might not yet be realized
	my $dialog = Gtk2::MessageDialog->new_with_markup(
		$window, 'modal', 'error', 'ok', $text1 );
		# parent, flags, type, buttons, message
	$dialog->run;
	$dialog->destroy;
	return undef;
}

=item C<jump_page_dialog()>

Prompts the user for a page name. This page is then opened.

=cut

sub jump_page_dialog {
	my $self = shift;
	my $page = $self->{page}->namespace;

	$page = '' if $page eq ':';
	my $values = $self->run_prompt(
		'Jump to',
		['page'], {page => ['Jump to Page', 'page', $page]},
		'gtk-jump-to', undef, undef )
		or return;
	($page) = @$values;
	return unless $page =~ /\S/;
		
	$self->link_clicked($page, 1);
}

=item C<prompt_rename_page_dialog>

=cut

sub prompt_rename_page_dialog {
	my ($self, $page) = @_;
	
	my ($dialog, $entries) = $self->new_prompt(
		'Rename page',
		['page'], {page => ['Rename to', 'page', $page]},
		'gtk-save', '_Rename'  );
	
	my ($entry) = @$entries;
	$entry->select_region(length($1), -1) if $page =~ /^(.+:)/;

	my $rpage = ($page eq $self->{page}->name) ? $self->{page} : $self->{repository}->get_page($page); # FIXME ugly lookup
	my $nlinks = scalar $rpage->_list_backlinks;
	#my $check = Gtk2::CheckButton->new("_Update $nlinks pages linking here");
	#if ($nlinks > 0) { $check->set_active(1)    }
	#else             { $check->set_sensitive(0) }
	#$check->show;
	#$dialog->vbox->add($check);
	
	if ($dialog->run eq 'ok') { $page = $entry->get_text }
	else                      { $page = undef            }
	my $update = 0; #$check->get_active;
	$dialog->destroy;

	$page = undef unless $page =~ /\S/;
	return defined($page) ? ($page, $update) : ();
}

=item C<prompt_question(TITLE, TYPE, TEXT, BUTTONS ..., TIME)>

Runs a dialog displaying TEXT

BUTTONS is a list of array references, each containing a name, a stock item
name and/or text. The id of the button that was pressed is returned.

TYPE can either be 'error', 'warning', 'question', 'info' or C<undef>.

TIME is an optional argument, it gives a timeout in seconds. This is used
for popups that can popup while the user is typing to prevent accidental
triggering of a accelerator.

=cut

sub prompt_question {
	my ($self, $title, $type, $text, @buttons) = @_;
	my $time = pop @buttons unless ref $buttons[-1];
	
	my $dialog = Gtk2::Dialog->new(
		$title, $self->{window},
	       	[qw/modal destroy-with-parent no-separator/],
	);
	$dialog->set_resizable(0);
	$dialog->set_icon($self->{window}->get_icon);

	my @button_widgets;
	for (0 .. $#buttons) {
		my ($id, $stock, $string) = @{$buttons[$_]};
		my $button = (defined($stock) && ! defined($string))
			? Gtk2::Button->new_from_stock($stock)
			: $self->new_button($stock, $string)   ;
		$button->set_sensitive(0);
		$dialog->add_action_widget($button, $_);
		push @button_widgets, $button;
	}
	
	my $hbox = Gtk2::HBox->new(0,12);
	$hbox->set_border_width(12);
	$dialog->vbox->pack_start($hbox, 0,0,0);

	if (defined $type) {
		my $image = Gtk2::Image->new_from_stock(
			"gtk-dialog-$type", 'dialog' );
		$image->set_alignment(0.0, 0.5); # valign=top
		$hbox->pack_start($image, 0,0,0);
	}
	if (defined $text) {
		my $label = Gtk2::Label->new($text);
		$label->set_use_markup(1);
		$label->set_selectable(1);
		$label->set_alignment(0.0, 0.0); # align left top corner
		$hbox->add($label);
	}

	$dialog->show_all;
	if ($time) {
		Glib::Timeout->add( $time*1000,
			sub { $_->set_sensitive(1) for @button_widgets; 0 } );
	}
	else { $_->set_sensitive(1) for @button_widgets }
	my $id = $dialog->run;
	$dialog->destroy;
	
	return $buttons[$id][0];
}

=item C<about_dialog()>

This dialog tells you about the version of zim you are using,
the copyright and the current config file. Most importantly 
it has a button to open the manual.

=cut

sub about_dialog { # TODO use Gtk2::AboutDialog for gtk > 2.6
	my $self = shift;
	
	my $dialog = Gtk2::Dialog->new(
		'About Zim', $self->{window},
	       	[qw/modal destroy-with-parent no-separator/],
		'gtk-close' => 'close',
	);
	$dialog->set_resizable(0);
	$dialog->set_border_width(5);
	$dialog->set_icon($self->{window}->get_icon);
	my $button = $self->new_button('gtk-help', '_More');
	$dialog->add_action_widget($button, 'help');
	$dialog->set_default_response('close');
	
	$dialog->vbox->add(
		Gtk2::Image->new_from_file($ICON) );

	my $text = $LONG_VERSION;
	$text =~ s/^(.*)$/\n<b>$1<\/b>/m;
	my $label = Gtk2::Label->new();
	$label->set_markup($text);
	$label->set_justify('center');
	$dialog->vbox->add($label);
	
	$dialog->show_all;
	my $response = $dialog->run;
	$dialog->destroy;

	$self->show_help('zim:about') if $response eq 'help';
}

=item C<show_help(PAGE)>

Show a window showing the documentation. PAGE is optional.

=cut

sub show_help {
	my ($self, $page) = @_;
	my @args = ('--doc', $page ? $page : ());
	$self->exec_new_window(@args);
}

=item  C<exec_new_window(..)>

Executes a new process for $0, this gives a detached window.
Any arguments are passed on to the new process.

=cut

sub exec_new_window {
	my ($self, @args) = @_;
	
	if (ref($self->{window}) !~ /Dialog/) {
		@{$self->{settings}}{'width', 'height'} = $self->{window}->get_size;
	}

	$self->_exec($^X, $0, @args);
}

=item C<filechooser_dialog(FILE)>

Ask the user for a filename. FILE is the suggested filename.

=cut

sub filechooser_dialog {
	my ($self, $file, $dir, $title) = @_;
	
	my $dialog;
	$title ||= $dir ? 'Select Folder' : 'Select File' ;
	# if (Gtk2->CHECK_VERSION(2, 4, 0) and $Gtk2::VERSION >= 1.040) {
	$dialog = Gtk2::FileChooserDialog->new(
		$title, $self->{window}, 'open',
		'gtk-cancel' => 'cancel',
		'gtk-ok'     => 'ok'
	);
	# }
	#else { # old & ugly interface
	#	$dialog = Gtk2::FileSelection->new($title);
	#}
	$dialog->set_icon($self->{window}->get_icon) if $self->{window};
	$dialog->set_action('select-folder') if $dir;
	if (defined $file) {
		$file =~ s#^file:(//localhost)?/+#/#;
		$dialog->set_filename(File::Spec->rel2abs($file));
	}
	elsif (defined $self->{page}) {
		my $dir = $self->{page}->properties->{base};
		if (defined $dir) {
			$dir =~ s#^file:(//localhost)?/+#/#;
			$dialog->set_current_folder(File::Spec->rel2abs($dir));
		}
	}
	$dialog->signal_connect('response', sub {
		$file = $_[1] eq 'ok' ? $dialog->get_filename : undef;
		$dialog->destroy;
	} );
	$dialog->run;

	return $file;
}

=item C<prompt_repository_dialog(CALLBACK)>

This dialog does not run "modal" so it needs a callback because the
calling code can not wait for it to return.

=cut

sub prompt_repository_dialog { # TODO improve visual layout
	my $self = shift;
	my $callback = shift;

	#warn "window: $self->{window}\n";
	my $dialog = Gtk2::Dialog->new(
		"Open repository - Zim", $self->{window},
	       	[qw/destroy-with-parent no-separator/],
		'gtk-help'   => 'help',
		'gtk-cancel' => 'cancel',
		'gtk-open'   => 'ok',
	);
	#$dialog->set_type_hint('utility');
	$dialog->set_default_size(100,250);
	$dialog->set_border_width(5);
	$dialog->set_icon(
		Gtk2::Gdk::Pixbuf->new_from_file($ICON) );
	$dialog->set_default_response('ok');
	$self->{window} ||= $dialog; # parent might not yet be initialized
	
	$dialog->vbox->set_spacing(5);
	$dialog->vbox->pack_start(
		Gtk2::Label->new('Please choose a repository'), 0,1,0 );
	
	my $hbox = Gtk2::HBox->new(0, 12);
	$dialog->vbox->add($hbox);
	
	my $list = Gtk2::SimpleList->new('Repository' => 'text');
	$list->set_headers_visible(0);
	$list->get_selection->set_mode('browse');
	$list->signal_connect(row_activated => sub { $dialog->response('ok') });
	#$dialog->vbox->pack_start($list, 1,1,0);
	$hbox->add($list);
	
	# load list
	my ($source, $order) = ({}, []);
	my ($file) = xdg_config_files('zim', 'repositories.list');
	if (defined $file) {
		$file = Zim::File->new($file);
		($source, $order) = $file->read_config;
		@{$list->{data}} = @$order;
	}
	
	#my $hbox = Gtk2::HBox->new(1,10);
	#my $add_button = Gtk2::Button->new_from_stock('gtk-add');
	#my $rem_button = Gtk2::Button->new_from_stock('gtk-remove');
	#$hbox->add($add_button);
	#$hbox->add($rem_button);
	#$dialog->vbox->pack_start($hbox, 0,1,0);
	my $vbox = Gtk2::VButtonBox->new();
	$vbox->set_layout('start');
	$hbox->pack_start($vbox, 0,0,0);
	
	my @buttons = map Gtk2::Button->new_from_stock($_),
		qw/gtk-add gtk-remove/;
	splice @buttons, 1, 0, $self->new_button('gtk-properties', 'Change');
	$vbox->add($_) for @buttons;
	
	my $modified = 0;
	my $modify = sub {
		my ($i, $name, $dir);
		if (pop @_) { #boolean user data
			($i) = $list->get_selected_indices;
			($name) = @{$list->{data}[$i]};
			$dir = $$source{$name};
		}
		else { $i = scalar @{$list->{data}} }
		($name, $dir) = $self->prompt_new_repository_dialog($name, $dir);
		return unless defined $name;
		$$source{$name} = $dir;
		splice @{$list->{data}}, $i, 1, $name;
		$modified = 1;
		$list->select($i);
	};
	$buttons[0]->signal_connect(clicked => $modify, 0);
	$buttons[1]->signal_connect(clicked => $modify, 1);
	$buttons[2]->signal_connect(clicked => sub {
			my ($i) = $list->get_selected_indices;
			splice @{$list->{data}}, $i, 1;
			$modified = 1;
		} );
	
	$dialog->signal_connect(response => sub {
		my ($dialog, $response) = @_;
		
		# Save list
		if ($modified) {
			@$order = map $$_[0], @{$list->{data}};
			$file = Zim::File->new(
				xdg_config_home, 'zim', 'repositories.list' );
			$file->write_config($source, $order);
		}

		if ($response eq 'help') {
			return $self->show_help('zim:usage:repositories');
		}
		elsif ($response ne 'ok') {
			$dialog->destroy;
			Gtk2->main_quit if $self->{window} eq $dialog;
			return;
		}

		my ($i) = (@{$list->{data}} == 1)
			? (0) : ($list->get_selected_indices);
		return $self->error_dialog("Please select a repository first.\n")
			unless defined $i;
	
		my ($name) = @{$list->{data}[$i]};
		my $src = Zim::File->abs_path($$source{$name}, $ENV{HOME});
		$callback->($name, $src);
		$dialog->destroy;
	} );
	
	$dialog->show_all;
}

=item C<prompt_new_repository_dialog(WINDOW)>

=cut

sub prompt_new_repository_dialog { # TODO add "browse" button for dir
	my $self = shift;
	my $val = $self->run_prompt(
		'New repository - Zim',
		['name', 'dir'], {
			name => ['Name', 'string', $_[0]],
			dir  => ['Directory', 'dir', $_[1]],
		}, undef, undef,
		'Please give a directory to store your pages' )
		or return undef;
	
	my ($name, $dir) = @$val;
	return undef unless $dir =~ /\S/;
	$name = $dir unless $name =~ /\S/;
	
	return ($name, $dir);
}

sub _new_progress_bar {
	my ($self, $title, $label) = @_;
	my $dialog = Gtk2::Dialog->new(
		$title, $self->{window},
	   	[qw/destroy-with-parent no-separator/],
		'gtk-cancel' => 'cancel',
	);
	$dialog->set_resizable(0);
	$dialog->vbox->set_spacing(5);
	$dialog->vbox->set_border_width(10);
	$label = Gtk2::Label->new($label);
	$dialog->vbox->add($label);
	my $bar = Gtk2::ProgressBar->new;
	$dialog->vbox->add($bar);
	$dialog->show_all;
	return ($dialog, $bar, $label);
}

=item C<preferences_dialog()>

=cut

sub preferences_dialog {
	my $self = shift;
	$self->save_page || return;

	my $dialog = Gtk2::Dialog->new(
		"Preferences - Zim", $self->{window},
	       	[qw/destroy-with-parent no-separator/],
		'gtk-help'   => 'help',
		'gtk-cancel' => 'cancel',
		'gtk-ok'     => 'ok',
	);
	$dialog->set_resizable(0);
	
	my $tabs = Gtk2::Notebook->new;
	$dialog->vbox->add($tabs);
	
	## General tab
	my $vbox1 = Gtk2::VBox->new(0,5);
	$vbox1->set_border_width(5);
	$tabs->append_page($vbox1, 'General');
	
	my @settings = qw/date_string browser file_browser email_client/;
	my @labels = (
		'Date format',
		'Browser',
		'File browser',
		'Email client',
		'Font'	);
	my @entries;
	my $table = Gtk2::Table->new(scalar(@labels), 2);
	$table->set_row_spacings(5);
	$table->set_col_spacings(12);
	$vbox1->pack_start($table,0,1,0);
	my $i = 0;
	for (@labels) {
		my $label = Gtk2::Label->new($_.':');
		my $align = Gtk2::Alignment->new(0,0.5, 0,0);
		$align->add($label);
		$table->attach_defaults($align, 0,1, $i,$i+1);
		if (/Font/) {
			my $font = $self->{settings}{textfont} ||
				$self->PageView->get_style->font_desc->to_string;
			#warn "font: $font\n";
			my $button = Gtk2::FontButton->new_with_font($font);
			$button->signal_connect(font_set => sub {
					my $string = $button->get_font_name;
					$self->{settings}{textfont} = $string;
					$self->PageView->set_font($string);
			} );
			$table->attach_defaults($button, 1,2, $i,$i+1);
		}
		else {
			my $entry = Gtk2::Entry->new;
			$entry->set_text( $self->{settings}{$settings[$i]} );
			push @entries, $entry;
			$table->attach_defaults($entry, 1,2, $i,$i+1);
		}
		$i++;
	}

	## Editing tab
	my $vbox3 = Gtk2::VBox->new(0,5);
	$vbox3->set_border_width(5);
	$tabs->append_page($vbox3, 'Editing');
	my @edit_settings = qw/follow_new_link use_camelcase use_utf8_ent/;
	my @edit_labels = (
		'Follow new link',
		'Use CamelCase',
		'Use utf8 entities' );
	my @toggles;
	$table = Gtk2::Table->new(scalar(@edit_labels), 1);
	$table->set_row_spacings(5);
	$table->set_col_spacings(12);
	$vbox3->pack_start($table, 0,1,0);
	$i = 0;
	for (@edit_labels) {
		my $box = Gtk2::CheckButton->new_with_label($_);
		$box->set_active($self->{settings}{$edit_settings[$i]});
		$table->attach_defaults($box, 0,1, $i,$i+1);
		$i++;
		push @toggles, $box;
	}

	## Plugins tab
	my $vbox2 = Gtk2::VBox->new(0,5);
	$vbox2->set_border_width(5);
	$tabs->append_page($vbox2, 'Plugins');

	$vbox2->pack_start(Gtk2::Label->new(
		"You need to restart the application\nfor plugin changes to take effect."), 0,1,0 );
	my @plugins;
	my @plugged = split /,/, $self->{settings}{plugins};
	for (xdg_data_home(), xdg_data_dirs()) {
		my $dir = File::Spec->catdir($_, 'zim', 'plugins');
		next unless -d $dir;
		for my $f (Zim::File->list_dir($dir)) {
			next unless $f =~ s/\.pl$//;
			next if grep {$$_[0] eq $f} @plugins;
			my $bit = grep {$_ eq $f} @plugged;
			push @plugins, [$f, $bit];
		}
	}
	@plugins = sort {$$a[0] cmp $$b[0]} @plugins;
	
	$table = Gtk2::Table->new(scalar(@plugins), 1);
	$table->set_row_spacings(5);
	$table->set_col_spacings(12);
	$vbox2->pack_start($table, 0,1,0);
	$i = 0;
	for (@plugins) { # TODO add descriptions
		my $box = Gtk2::CheckButton->new_with_label($$_[0]);
		$box->set_active($$_[1]);
		$table->attach_defaults($box, 0,1, $i,$i+1);
		$i++;
		push @$_, $box;
	}
	
	## Show it all
	$dialog->show_all;
	while ($_ = $dialog->run) {
		if ($_ eq 'help') {
			$self->show_help(':zim:usage:preferences');
			next;
		}
		last unless $_ eq 'ok';
		
		# set setting
		$self->{settings}{$settings[$_]} = $entries[$_]->get_text
			for 0 .. $#settings;
			
		unless (length $self->{settings}{date_string}) {
			$self->{settings}{date_string} = '%A %d/%m/%Y';
			$self->{repository}{date_string} = '%A %d/%m/%Y';
		}
		else { $self->{repository}{date_string} = $self->{settings}{date_string} } # Tmp hack till more fully template support gets implemented
		
		# set editing settings
		$self->{settings}{$edit_settings[$_]} = $toggles[$_]->get_active || '0'
			for 0 .. $#edit_settings;
		
		# set plugins
		@plugged = map $$_[0], grep $$_[2]->get_active, @plugins;
		$self->{settings}{plugins} = join ',', @plugged;

		last;
	}
	$dialog->destroy;
	$self->save_config;
}

1;

__END__

=back

=head1 BUGS

Please mail the author if you find any bugs.

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

L<zim>(1),
L<Zim::Repository>

=cut

