package Zim::File;

use strict;
use Carp;
use File::Glob ':glob';
use File::Spec;
use File::Copy ();
use Encode;

require Cwd;

our $VERSION = '0.16';

use overload
	'""' => \&path,
	fallback => 'TRUE' ;

unless (length $ENV{HOME}) {
	my ($home) = bsd_glob('~', GLOB_TILDE);
	$ENV{HOME} = $home if length $home;
}

=head1 NAME

Zim::File - OO wrapper for files

=head1 SYNOPSIS

	$file = Zim::File->new('./Changes');
	$text = $changes."\n\n".$file->read;
	$file->write($text);

=head1 DESCRIPTION

This package provides an object for handling files.
It bundles some snippets of code that are needed by several modules.

=head1 OVERLOAD

This class overloads the '""' operator, so the string version of an object
is the file path.

=head1 METHODS

=over 4

=item C<new(FILENAME)>

Simple constructor, uses L<File::Spec>'s C<catfile> when multiple arguments are given.

=cut

sub new {
	my $class = shift;
	my $name = (@_ == 1) ? shift : File::Spec->catfile(@_);
	croak "Invalid filename: '$name'" unless $name =~ /\S/;
	bless {path => $class->abs_path($name)}, $class;
}

=item C<path>

Returns the full path to the file.

=cut

sub path { $_[0]->{path} }

=item C<file>

Returns the filename part of the path.

=cut

sub file {
	my $self = shift;
	my $file = @_ ? File::Spec->rel2abs($_[0]) : $self->{path};
	my (undef, undef, $name) = File::Spec->splitpath($file);
	return $name;
}

=item C<dir>

Returns the directory part of the path.

=cut

sub dir {
	my $self = shift;
	my $file = @_ ? File::Spec->rel2abs($_[0]) : $self->{path};
	my ($vol, $dir, undef) = File::Spec->splitpath($file);
	return File::Spec->catpath($vol, $dir, '');
}

=item C<read()>

Returns the contents of the file. Takes list context in account.
Returns empty when the file does not exist.

=cut

sub read {
	my $self = shift;
	my $file = $self->{path};
	return unless -f $file;
	CORE::open FILE, "<$file" or die "Could not read file: $file\n";
	binmode FILE, ':utf8' unless $] < 5.008;
	my @text = wantarray ? (<FILE>) : join('', <FILE>);
	close FILE;
	return wantarray ? (@text) : $text[0];
}

=item C<read_config()>

Read and parses contents.
Returns a hash with key value mappings and a list which indicates the order
in which the keys were found.

=cut

sub read_config {
	my $self = shift;
	my (%config, @order);
	for ($self->read) {
		/^\s*\#/ || ! /^\s*(.+?)=(.*)$/ and next;
		$config{$1} = $2;
		push @order, $1;
	}
	return wantarray ? (\%config, \@order) : \%config;
}

=item C<read_hash(KEY)>

Searches for KEY in a plain text hash.

=cut

sub read_hash {
	my ($self, $key) = @_;
	my $value = undef;
	my $fh = $self->open;
	while (<$fh>) {
		/^\s*$key/ or next;
		/^\s*$key\s+(.+?)\s*$/ or next;
		$value = $1;
		last;
	}
	close $fh;
	return $value;
}

=item C<write(CONTENT, ..)>

Write CONTENT to file.
Creates file and parent directories when needed.

=item C<append(CONTENT, ..)>

Append content to file.
Creates file and parent directories when needed.

=cut

sub write {
	my $self = shift;
	$self->_write('>', $self->{path}, @_);
}

sub append {
	my $self = shift;
	$self->_write('>>', $self->{path}, @_);
}

sub _write {
	my $self = shift;
	my $mode = shift;
	my $file = shift;
	$self->make_dir;
	CORE::open FILE, $mode.$file or die "Could not write file: $file\n";
	binmode FILE, ':utf8' unless $] < 5.008;
	print FILE @_;
	close FILE or die "Could not write file: $file\n";
}

=item C<write_config(\%CONFIG, \@ORDER)>

Writes a list of keys and values to file.
A second argument can be used to enforce the order of the keys in the
config file; this is optional.

=cut

sub write_config {
	my ($self, $config, $order) = @_;
	$order ||= [sort keys %$config];
	$self->write(map "$_=$$config{$_}\n", @$order);
}

=item C<remove()>

Delete the file.
Removes parent directories when empty.

=cut

sub remove {
	my $self = shift;
	my $file = $self->{path};
	unlink $file or die "Could not delete file: $file\n";
	$self->remove_dir;
}

=item C<exists()>

Returns boolean.

=cut

sub exists { -f $_[0]->{path} }

=item C<stat()>

Returns a hash reference with stat values.

=cut

my @stat = qw/dev ino mode nlink uid gid rdev size
		atime mtime ctime blksize blocks/ ;

sub stat {
	my $self = shift;
	my @val = stat $self->{path};
	return { map {$stat[$_] => $val[$_]} 0 .. 12 };
}

=item C<open(MODE)>

Returns an L<IO::File> object.

=cut

sub open {
	my ($self, $mode, $cb) = @_;
	$mode ||= 'r';
	my $file = $self->{path};
	my $handle = IO::File::Zim->new($file, $mode) || die "$file: $!\n";
	$handle->set_callback($cb) if defined $cb;
	binmode $handle, ':utf8' if defined $handle and not $] < 5.008;
		# no utf8 binmode for perl 5.6 and earlier
		# no OO call because IO::File OO version broken in perl 5.8.7
	return $handle;
}

=item C<grep(REGEXP)>

Line based grep function.

=cut

sub grep {
	my ($self, $regexp) = @_;
	my $file = $self->{path};
	return unless -f $file;
	#warn "Grepping for /$regexp/ in $file\n";
	my @match;
	CORE::open FILE, "<$file" or die "Could not read file: $file\n";
	while (<FILE>) {
		push @match, ($_ =~ /($regexp)/);
	}
	close FILE;
	#warn "\t=> @match\n";
	return @match;
}

=back

=head2 Class Methods

The following methods don't need an object to work on. They can also be called
as for example C<< Zim::File->make_dir($dir) >>. Of course they can also be
called on a Zim::File object.

=over 4

=item C<abs_path(NAME, REFERENCE)>

Turn a relative filename or a file uri into a local path name
and cleans up this path to a logical form. It also does tilde 
expansion.

If REFERENCE is omitted the current dir is used.

Calls C<parse_uri()> first when needed.

=cut

sub abs_path {
	my ($class, $file, $ref) = @_;
	#warn "abs_path: $name ($ref)\n";
	$file = $class->parse_uri($file) if $file =~ m#file:/#;
	return $file if $file =~ m#^\w\w+:/#;

	if ($file =~ m#^(~([^/]*))#) {
		my ($home) = length($2) ? bsd_glob($1, GLOB_TILDE) : $ENV{HOME};
		$file =~ s#^(~[^/]*)#$home#;
	}
	elsif ($file !~ m#^/#) {
		$ref ||= Cwd::cwd();
		$file = $ref.'/'.$file;
	}
	
	$file =~ s#(?<=[^/])/+\.(?![^/])##g; # remove /.
	while ($file =~ s#[^/]+/+\.\.(?![^/])##) {} # remove foo/..
	$file =~ s#^(/\.\.)+(?![^/])##; # remove leading /../
	$file =~ s#//+#/#g; # /// => /
	return $file;
}

=item C<parse_uri(URI)>

Method for parsing file uris, returns either a normal path or a (new) uri.
No url decoding is done here.

=cut

# support file:/path file://localhost/path and file:///path

sub parse_uri {
	shift; # class
	my $file = shift;
	
	$file =~ m#^file:/# or return $file;
	$file =~ s#^file:(?://localhost/+|/|///+)([^/]+)#/$1#i or return $file;

	return $file;
}

=item C<path2uri(FILE)>

Turns a path into an uri. Does no checks.

=cut

sub path2uri {
	shift; # class
	my $file = shift;
	$file =~ s#^/?#file:///#;
	return $file;
}

=item C<localize(FILE)>

Returns platform dependend version of FILE.

=cut

sub localize {return pop} # default does nothing, placeholder for win32

=item C<copy(SOURCE, DEST)>

Copy content from one file to another.
Creates parent directories if needed.

=cut

sub copy {
	my $self = $_[0];
	
	my $dest = pop;
	my $source = pop;
	$dest = $dest->path if ref $dest;
	$source = $source->path if ref $source;
	croak "Invalid file name: $dest\n" unless $dest =~ /\S/;
	croak "Invalid file name: $source\n" unless $source =~ /\S/;
	
	$self->make_dir($self->dir($dest));
	File::Copy::copy($source, $dest)
		or die "Could not copy '$source' to '$dest'\n";
}

=item C<move(SOURCE, DEST)>

Move file contents from one place to another.
Takes care of creating and cleaning up parent directories.

=cut

sub move {
	my $self = $_[0];
	
	my $dest = pop;
	my $source = pop;
	$dest = $dest->path if ref $dest;
	$source = $source->path if ref $source;
	croak "Invalid file name: $dest\n" unless $dest =~ /\S/;
	croak "Invalid file name: $source\n" unless $source =~ /\S/;
	
	$self->make_dir($self->dir($dest));
	#warn "moving $source => $dest\n";
	File::Copy::move($source, $dest)
		or die "Could not move '$source' to '$dest'\n";
	$self->remove_dir($self->dir($source));
}

=item C<list_dir(DIR)>

Returns a list of items for DIR. Ignores hidden files.

=cut

sub list_dir {
	my ($self, $dir) = @_;
	$dir = $self->dir if ! defined $dir and ref $self;
	croak "Invalid dir name: $dir" unless $dir =~ /\S/;
	
	opendir DIR, $dir or die "Could not list dir $dir\n";
	my @items = grep {! /^\./} readdir DIR;
	closedir DIR;

	eval {$_ = Encode::decode_utf8($_, 1)} for @items;
	return @items;
}

=item C<make_dir(DIR)>

Create DIR is it doesn't exist already. Subtle differences with L<File::Path>.

=cut

sub make_dir {
	my ($self, $dir) = @_;
	$dir = $self->dir if ! defined $dir and ref $self;
	croak "Invalid dir name: $dir" unless $dir =~ /\S/;
	
	return if -d $dir;
	
	my ($vol, $dirs) = File::Spec->splitpath(
		File::Spec->rel2abs($dir), 'DIR');
	my @dirs = File::Spec->splitdir($dirs);
	my $path = File::Spec->catpath($vol, shift @dirs, '');
	mkdir $path or die "Could not create dir: $path\n"
		unless ! length $path or -d $path;
	while (@dirs) {
		$path = File::Spec->catdir($path, shift @dirs);
		mkdir $path or die "Could not create dir $path\n"
			unless -d $path;
	}
}

=item C<remove_dir(DIR)>

Removes a directory and it's parents when they are empty.
Fails silently.

=cut

sub remove_dir {
	my ($self, $dir) = @_;
	$dir = $self->dir if ! defined $dir and ref $self;
	croak "Invalid dir name: $dir" unless $dir =~ /\S/;
	
	rmdir $dir or return; # fails when non-empty
	
	my ($vol, $dirs) = File::Spec->splitpath(
		File::Spec->rel2abs($dir), 'DIR');
	my @dirs = File::Spec->splitdir($dirs);
	while (@dirs) {
		pop @dirs;
		$dir = File::Spec->catdir($vol, @dirs);
		#warn "rmdir $dir\n";
		rmdir $dir or last; # fails when non-empty
	}
}

package IO::File::Zim;

use IO::File;

our @ISA = qw/IO::File/;

# This class is a hack in order to provide a callback
# when the filehandle is closed. This is needed to fix
# stuff like saving the 'mtime' _after_ the file was writen.

# It uses a global hash to save object attributes because
# IO::File objects are filehandle references, not hashes.

# It uses a indirect subroutine definition to circumvent
# the Pod::Cover test

our %_objects;

*set_callback = sub {
	my $self = shift;
	$_objects{"$self"}{callback} = shift;
};

*close = sub {
	my $self = shift;
	my $re = $self->SUPER::close(@_);
	return $re unless exists $_objects{"$self"};
	my $obj = delete $_objects{"$self"};
	$obj->{callback}->() if defined $obj->{callback};
	return $re;
};

1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2006 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

L<Zim>

=cut

