package Zim::History;

use strict;
use Storable qw/nstore retrieve/;

our $VERSION = '0.12';

=head1 NAME

Zim::History - History object for zim

=head1 SYNOPSIS

	use Zim::History;
	
	my $hist = Zim::History->new($HIST_FILE, $PAGEVIEW, $MAX_HIST);
	my $page_record = $hist->get_current;

=head1 DESCRIPTION

This object manages zim's page history. It keeps tree different stacks: one for
the history, one for recently opened pages and one for the namespace.

The history stack is just the chronological order in which pages were opened.

The recent pages stack is derived from the history stack but avoids
all redundancy.

The namespace stack keeps track of the namespaces you opened.

The history is saved in a cache file using L<Storable>.

=head1 METHODS

=over 4

=item C<new(FILE, PAGEVIEW, MAX)>

Constructor. Takes the name of the cache file as an argument
and reads this cache file. The second argument is the pageview
object. When a new record is made for the current page the C<get_state()> method is
called on this object to get state information that needs to be saved.
Third argument is the maximum number of items in the history.

=cut

sub new {
	my ($class, $file, $view, $max) = @_;
	my $self = bless {
		file      => $file,
		PageView  => $view,
		max       => $max,
		point     => 0,
		hist      => [],
		recent    => [],
	}, $class;
	$self->read;
	return $self;
}

=item C<read>

Read the cache file.

=cut

sub read {
	my $self = shift;
	return unless -f $self->{file} and -r _;
	my @cache = @{ retrieve($self->{file}) };
	my $version = shift @cache;
	return unless $version == $VERSION;
	@{$self}{qw/point hist recent/} = @cache;
}

=item C<write>

Write the cache file.

=cut

sub write {
	my $self = shift;
	$self->get_current; # force object to record conversion
	nstore([$VERSION, @{$self}{qw/point hist recent/}], $self->{file});
}

=item C<set_current(PAGE)>

Give the page object that is going to be displayed in the 
PageView. Set a new current before updating the PageView
so the History object has time to save the state of the PageView
if necessary.

=cut

sub set_current {
	my ($self, $page) = @_;

	my $current = $self->get_current;
	my $name = $page->name;
	$self->{page} = $page; # set _after_ calling get_current()
	return if $current and $name eq $current->{name}; # directly redundant

	# update hist stack
	my $point = $self->{point};
	my $hist = $self->{hist};
	if    ($point >= $self->{max}) { shift @$hist }
	elsif (defined $$hist[$point]) { $point++     }
	splice @$hist, $point; # clear forw stack
	$$hist[$point] = undef;
	$self->{point} = $point;
	# note: do not call get_current below this line

	# update recent stack
	my $recent = $self->{recent};
	for (@$recent) { return if $_->{name} eq $name } # check redundancy
	my $rec = { map {$_ => $page->$_} qw/name namespace basename/ };
	$$hist[$point] = $rec; # force linking of ref
	shift @$recent if $#$recent >= $self->{max};
	push @$recent, $rec;

	#use Data::Dumper; print 'hist, recent: ', Dumper($hist, $recent);
}

=item C<get_current()>

Returns the curernt history object. When possible asks the PageView
objects for the current state information.

=cut

sub get_current {
	my $self = shift;
	my $page = $self->{page};
	my $rec = $self->{hist}[ $self->{point} ];
	#print "rec: >>$rec<<, page: >>$page<<\n";
	return $rec unless $page;
	
	unless ($rec) { # new record
		my $name = $page->name;
		$rec = $self->_get_record($name) || {
			name => $name,
			map {$_ => $page->$_} qw/namespace basename/  };
		$self->{hist}[ $self->{point} ] = $rec;
		#use Data::Dumper; print "rec: ", Dumper $rec;
	}

	%$rec = ( %$rec, $self->{PageView}->get_state() );

	return $rec;
}

=item C<back(INT)>

Go back one or more steps in the history stack.
Returns the record for this step or undef on failure.

=cut

sub back {
	my ($self, $i) = @_;
	return if $i < 1 or $i > $self->{point};
	$self->get_current;
	$self->{point} -= $i;
	$self->{page} = undef;
	return $self->{hist}[ $self->{point} ];
}

=item C<forw(INT)>

Go forward one or more steps in the history stack.
Returns the record for this step or undef on failure.

=cut

sub forw {
	my ($self, $i) = @_;
	my $forw = $#{$self->{hist}} - $self->{point};
	return if $i < 1 or $i > $forw;
	$self->get_current;
	$self->{point} += $i;
	$self->{page} = undef;
	return $self->{hist}[ $self->{point} ];
}

=item C<get_history()>

Returns an integer followed by a list of all records in the history stack.
The integer is the index of the current page in this stack.

=cut

sub get_history {
	my $self = shift;
	$self->get_current;
	return $self->{point}, @{$self->{hist}};
}

=item C<get_recent()>

Returns an integer followed by a list of all records in the recent pages stack.
The integer is the index of the current page in this stack.

=cut

sub get_recent {
	my $self = shift;
	my $name = $self->get_current->{name};
	my $i = 0;
	for (@{$self->{recent}}) {
		#print "Comparing $name with $_->{name}\n";
		last if $_->{name} eq $name;
		$i++;
	}
	return $i, @{$self->{recent}};
}

=item C<get_namespace>

This method matches the namespace of the current page to that of pages in the
history. The result can be used as a namespace stack.

=cut

sub get_namespace {
	# FIXME set hist_namespace on set_current
	# refine logic on updating records by get_current
	my $self = shift;
	my $current = $self->get_current;
	return unless $current;

	my $namespace = $current->{name};
	#print STDERR "looking for namespace $namespace";
	for (
		reverse(0 .. $self->{point}-1),
		reverse($self->{point} .. $#{$self->{hist}})
	) {
		my $rec = $self->{hist}[$_];
		my $name = $$rec{_namespace} || $$rec{name};
		$namespace = $name if $name =~ /^:*$namespace:/;
	}
	#print STDERR " => $namespace\n";
	$current->{_namespace} = $namespace;

	return $namespace;
}

=item C<get_state()>

Returns a hash ref which contains numbers that tell how many items there are on
the history and namespace stacks in either direction.

=cut

sub get_state {
	my $self = shift;
	my $state = {
		back => $self->{point},
		forw => ($#{$self->{hist}} - $self->{point}),
		up   => 0, # FIXME
		down => 0, # FIXME
	};
	return $state;
}

=item C<get_record(PAGE)>

Returns the history record for a given page object or undef.

=cut

sub get_record {
	my ($self, $page) = @_;
	
	my $name = ref($page) ? $page->name : $page;
	$name =~ s/:+$//;
	
	my $current = $self->get_current;
	return $current if $current->{name} eq $name;

	return $self->_get_record($name);
}

sub _get_record { # used by get_current()
	my ($self, $name) = @_;

	#print STDERR "get_record $name\n";
	for (@{$self->{hist}}, @{$self->{recent}}) {
		#print STDERR "\tfound $_->{name}\n";
		return $_ if $_->{name} eq $name;
	}

	return undef;
}

1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

=cut

