package Zim::File;

use strict;
use Carp;
use File::Spec;
use File::Copy ();
use IO::File;
use Encode;

our $VERSION = '0.01';

use overload
	'""' => \&path,
	fallback => 'TRUE' ;

=head1 NAME

Zim::File - OO wrapper for files

=head1 SYNOPSIS

	$file = Zim::File->new('./Changes');
	$text = $changes."\n\n".$file->read;
	$file->write($text);

=head1 DESCRIPTION

This package provides an object for handling files.
It bundles some snippets of code that are needed by several modules.

=head1 OVERLOAD

This class overloads the '""' operator, so the string version of an object
is the file path.

=head1 METHODS

=over 4

=item C<new(FILENAME)>

Simple constructor, uses L<File::Spec>'s C<catfile> when multiple arguments are given.

=cut

sub new {
	my $class = shift;
	my $name = (@_ == 1) ? shift : File::Spec->catfile(@_);
	croak "Invalid filename: '$name'" unless $name =~ /\S/;
	$name = File::Spec->rel2abs($name);
	bless \$name, $class;
}

=item C<path>

Returns the full path to the file.

=cut

sub path { ${$_[0]} }

=item C<file>

Returns the filename part of the path.

=cut

sub file {
	my $self = shift;
	my $file = @_ ? File::Spec->rel2abs($_[0]) : $$self;
	my (undef, undef, $name) = File::Spec->splitpath($file);
	return $name;
}

=item C<dir>

Returns the directory part of the path.

=cut

sub dir {
	my $self = shift;
	my $file = @_ ? File::Spec->rel2abs($_[0]) : $$self;
	my ($vol, $dir, undef) = File::Spec->splitpath($file);
	return File::Spec->catpath($vol, $dir, '');
}

=item C<read()>

Returns the contents of the file. Takes list context in account.
Returns empty when the file does not exist.

=cut

sub read {
	my $self = shift;
	my $file = $$self;
	return unless -f $file;
	open FILE, "<$file" or die "Could not read file: $file\n";
	my @text = wantarray ? (<FILE>) : join('', <FILE>);
	close FILE;
	return wantarray ? (@text) : $text[0];
}

=item C<read_config()>

Read and parses contents.
Returns a hash with key value mappings and a list which indicates the order
in which the keys were found.

=cut

sub read_config {
	my $self = shift;
	my (%config, @order);
	for ($self->read) {
		/^\s*\#/ || ! /^\s*(.+?)=(.*)$/ and next;
		$config{$1} = $2;
		push @order, $1;
	}
	return wantarray ? (\%config, \@order) : \%config;
}

=item C<write(CONTENT, ..)>

Write CONTENT to file.
Creates file and parent directories when needed.

=item C<append(CONTENT, ..)>

Append content to file.
Creates file and parent directories when needed.

=cut

sub write {
	my $self = shift;
	$self->_write('>', $$self, @_);
}

sub append {
	my $self = shift;
	$self->_write('>>', $$self, @_);
}

sub _write {
	my $self = shift;
	my $mode = shift;
	my $file = shift;
	$self->make_dir;
	open FILE, $mode.$file or die "Could not write file: $file\n";
	print FILE @_;
	close FILE or die "Could not write file: $file\n";
}

=item C<write_config(\%CONFIG, \@ORDER)>

Writes a list of keys and values to file.
A second argument can be used to enforce the order of the keys in the
config file; this is optional.

=cut

sub write_config {
	my ($self, $config, $order) = @_;
	$order ||= [sort keys %$config];
	$self->write(map "$_=$$config{$_}\n", @$order);
}

=item C<remove()>

Delete the file.
Removes parent directories when empty.

=cut

sub remove {
	my $self = shift;
	unlink $$self or die "Could not delete file: $$self\n";
	$self->remove_dir;
}

=item C<exists()>

Returns boolean.

=cut

sub exists { -f ${$_[0]} }

=item C<open(MODE)>

Returns an L<IO::File> object.

=cut

sub open {
	my $self = shift;
	my $handle = IO::File->new($$self, @_);
	binmode $handle, ':utf8' if defined $handle and not $] < 5.008;
		# no utf8 binmode for perl 5.6 and earlier
		# no OO call because IO::File OO version broken in perl 5.8.7
	return $handle;
}

=back

=head2 Class Methods

The following methods don't need an object to work on. They can also be called
as for example C<< Zim::File->make_dir($dir) >>. Of course they can also be
called on a Zim::File object.

=over 4

=item C<copy(SOURCE, DEST)>

Copy content from one file to another.
Creates parent directories if needed.

=cut

sub copy {
	my $self = $_[0];
	
	my $dest = pop;
	my $source = pop;
	$dest = $$dest if ref $dest;
	$source = $$source if ref $source;
	croak "Invalid file name: $dest\n" unless $dest =~ /\S/;
	croak "Invalid file name: $source\n" unless $source =~ /\S/;
	
	$self->make_dir($self->dir($dest));
	File::Copy::copy($source, $dest)
		or die "Could not copy '$source' to '$dest'\n";
}

=item C<move(SOURCE, DEST)>

Move file contents from one place to another.
Takes care of creating and cleaning up parent directories.

=cut

sub move {
	my $self = $_[0];
	
	my $dest = pop;
	my $source = pop;
	$dest = $$dest if ref $dest;
	$source = $$source if ref $source;
	croak "Invalid file name: $dest\n" unless $dest =~ /\S/;
	croak "Invalid file name: $source\n" unless $source =~ /\S/;
	
	$self->make_dir($self->dir($dest));
	#warn "moving $source => $dest\n";
	File::Copy::move($source, $dest)
		or die "Could not move '$source' to '$dest'\n";
	$self->remove_dir($self->dir($source));
}

=item C<list_dir(DIR)>

Returns a list of items for DIR. Ignores hidden files.

=cut

sub list_dir {
	my ($self, $dir) = @_;
	$dir = $self->dir if ! defined $dir and ref $self;
	croak "Invalid dir name: $dir" unless $dir =~ /\S/;
	
	opendir DIR, $dir or die "Could not list dir $dir\n";
	my @items = grep {! /^\./} readdir DIR;
	closedir DIR;

	return map Encode::decode_utf8($_, 1), @items;
}

=item C<make_dir(DIR)>

Create DIR is it doesn't exist already. Subtle differences with L<File::Path>.

=cut

sub make_dir {
	my ($self, $dir) = @_;
	$dir = $self->dir if ! defined $dir and ref $self;
	croak "Invalid dir name: $dir" unless $dir =~ /\S/;
	
	return if -d $dir;
	
	my ($vol, $dirs) = File::Spec->splitpath(
		File::Spec->rel2abs($dir), 'DIR');
	my @dirs = File::Spec->splitdir($dirs);
	my $path = File::Spec->catpath($vol, shift @dirs, '');
	mkdir $path or die "Could not create dir: $path\n"
		unless ! length $path or -d $path;
	while (@dirs) {
		$path = File::Spec->catdir($path, shift @dirs);
		mkdir $path or die "Could not create dir $path\n"
			unless -d $path;
	}
}

=item C<remove_dir(DIR)>

Removes a directory and it's parents when they are empty.
Fails silently.

=cut

sub remove_dir {
	my ($self, $dir) = @_;
	$dir = $self->dir if ! defined $dir and ref $self;
	croak "Invalid dir name: $dir" unless $dir =~ /\S/;
	
	rmdir $dir or return; # fails when non-empty
	
	my ($vol, $dirs) = File::Spec->splitpath(
		File::Spec->rel2abs($dir), 'DIR');
	my @dirs = File::Spec->splitdir($dirs);
	while (@dirs) {
		pop @dirs;
		$dir = File::Spec->catdir($vol, @dirs);
		#warn "rmdir $dir\n";
		rmdir $dir or last; # fails when non-empty
	}
}

1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2006 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

L<Zim>

=cut

