package Zim::Repository;

use strict;
use Carp;
use UNIVERSAL qw/isa/;
use File::Spec;

our $VERSION = '0.12';

$Carp::CarpLevel = 1;

=head1 NAME

Zim::Repository - A wiki repository

=head1 SYNOPSIS

	use Zim::Repository;
	
	my $repo = Zim::Repository->new(undef, ':', $dir);
	my $page = $repo->get_page('Home');

=head1 DESCRIPTION

This class defines the public interface to the document repository
as used by L<zim>(1). It acts as a dispatcher that manages one or
more child objects to handle page requests. It also adds some tests
and thus tries to isolate the handlers from errors in the GUI.

The interface which child objects should support is defined in
L<Zim::Repository::Base>.

=head1 METHODS

=head2 Public Methods

The following methods can be used by the GUI.

=over 4

=item new(PARENT, NAMESPACE, DIR, ...)

Constructor. PARENT can be a parent repository object or undef.
NAMESPACE is the prefix for all pages managed by this repository.

DIR is the root namespace of the repository and might contain
a onfig file.

=cut

sub new {
	my ($class, $parent, $namespace, $dir, $format) = @_;
		# passing along $format here is a hack ! FIXME
	$namespace =~ s/:?$/:/;
	$dir = File::Spec->rel2abs($dir);
	my $self = bless {
		parent => $parent,
		namespace => $namespace,
		dir => $dir,
		config => {},
	}, $class;
	
	return $self unless length $dir;
	
	# load config file
	my $conf = File::Spec->catfile($dir, '.zim.config');
	if (-f $conf and -r _) {
		$self->{file} = $conf;
		open CONFIG, "<$conf" or die "Could not open: $conf\n";
		while (<CONFIG>) {
			chomp;
			next unless /\S/;
			my ($opt, $val) = split '=', $_, 2;
			if ($opt =~ /^:/) { # opt is namespace
				my ($class, $arg) = split '=', $val, 2;
				my @arg = split ',', $arg if defined $arg;
				$self->add_child($opt, $class, @arg);
			}
			else { $self->{config}{$opt} = $val }
		}
		close CONFIG;
	}
	
	# default config
	$self->add_child(':', 'File', $dir, $format)
		unless defined $self->{':'};

	return $self;
}

=item C<config()>

Returns a hash with config options.

=cut

sub config { $_[0]->{config} }

## 3 methods for checking arguments ##

sub _check_page {
	if (ref $_[0]) {
		croak "Object \"$_[0]\" is not a Zim::Page"
	       		unless isa $_[0], 'Zim::Page';
	}
	elsif ($_[1]) { croak "\"$_[0]\" is not a page object" }
	else { goto \&_check_page_name }
}

sub _check_page_name {
	croak "\"$_[0]\" is not a valid page name"
		unless $_[0] =~ /^:/ and $_[0] =~ /[^:]/;
}

sub _check_namespace {
	croak "\"$_[0]\" is not a valid namespace"
		unless $_[0] =~ /^:/ and $_[0] =~ /:$/;
}

=item C<list_pages(NAMESPACE)>

Lists pages in NAMESPACE. Sub-namespaces have a trailing ':'
in the listing.

If a page is not present in this list that does not mean that it
doesn't exists. This list is used for hierarchical views of the
page structure; not all repositories need to support a hierarchical order.

=cut

sub list_pages { # TODO also list plugin namespaces + parents
	my ($self, $namespace) = @_;
	$namespace =~ s/:?$/:/;
	_check_namespace($namespace);
	$namespace =~ $self->{regex};
	return $self->{":$1"}->list_pages($namespace);
}

=item get_page(NAME)

Returns an object of a class inheriting from L<Zim::Page>.
When you ask for a page that doesn't exists yet, you should get a new object.
In case it is not possible to create the page in question C<undef> is returned.

=cut

sub get_page {
	my ($self, $name) = @_;
	_check_page_name($name);
	$name =~ $self->{regex};
	#warn "Dispatching get_page('$name') to ':$1'\n";
	return $self->{":$1"}->get_page($name);
}

=item C<resolve_page(NAME)>

Like C<get_page()> but now expecting NAME to be user input,
so NAME is sanitized first.

=cut

sub resolve_page {
	my ($self, $name) = @_;
	$name =~ s/^:?/:/; # simple fixup
	$name =~ $self->{regex};
	#warn "Dispatching resolve_page('$name') to ':$1'\n";
	return $self->{":$1"}->resolve_page($name);
}

=item copy_page(SOURCE, TARGET)

Copy page SOURCE to TARGET. Both arguments can be either page names
or objects. Returns the (new) TARGET object.

=item move_page(SOURCE, TARGET)

Move page SOURCE to TARGET. Both arguments can be either page names
or objects. Returns the (new) TARGET object.

=cut

sub copy_page { _clone('copy_page', @_) }

sub move_page { _clone('move_page', @_) }

sub _clone { # Copy and move are almost the same
	my ($method, $self, $source, $target) = @_;
	_check_page($source);
	_check_page($target);
	
	$source = $self->get_page($source) || die "Could not open page '$source'\n"
		unless ref $source;
	$target = $self->get_page($target) || die "Could not create page '$target'\n"
	       	unless ref $target;
	
	die "Page '$target' exists\n" if $target->exists;
	
	if ($source->{repository} eq $target->{repository}) {
		$source->{repository}->$method($source, $target);
	}
	else {
		$target->clone($source);
		$source->{repository}->delete_page($source)
			if $method eq 'move_page';
	}

	return $target;
}

=item delete_page(PAGE)

Delete PAGE. PAGE can be either a page name or a page object.

=cut

sub delete_page {
	my ($self, $page) = @_;
	_check_page($page);
	$page = $self->get_page($page) unless ref $page;
	$page->{repository}->delete_page($page);
	return $page;
}

=item C<export(PAGE, %OPTS)>

TODO stable api for this

=cut

# TODO use verbose option

sub export {
	my ($self, $page, %opts) = @_;
	for (qw/format template root/) {
		die "for exporting you need at least to provide a format, a template and an output root dir.\n" unless length $opts{$_};
	}

	$page = $page->name if ref $page;
	my $namespace = ($page =~ /:$/) ? $page : '';
	#print STDERR "export root: $opts{root}\n";
	my $exporter = ref($opts{root}) ? $opts{root} :
		Zim::Repository::File->new(undef, '', $opts{root}, $opts{format});
	#print STDERR "export template: $opts{template}\n";
	$exporter->{_template} = $opts{template}; # FIXME better interface for this - do not confuse with the template for new pages
	
	my $r = defined($opts{recurs}) ? $opts{recurs} : 1 ;
	if (length $namespace) {
		$self->_export_namespace($exporter, $page, $r);
	}
	else {
		$self->_export_page($exporter, $page);
	}

}


sub _export_namespace {
	my ($self, $exporter, $namespace, $recurs) = @_;
	_check_namespace($namespace);
	
	my $index = 1;
	my @pages;
	for my $page ($self->list_pages($namespace)) {
		$index = 0 if $page =~ /^index/i;
		$self->_export_namespace($exporter, $namespace.$page, $recurs)
			if $recurs and $page =~ /:$/; # recurs
		push @pages, $page
			if $self->_export_page($exporter, $namespace.$page);
	}

	if ($index && @pages) { # FIXME option for this
		print STDERR "Writing index for $namespace\n";
		#print STDERR join(', ', @pages), "\n";
		$index = $exporter->get_page($namespace.'index');
		$index->set_parse_tree( ['Document', {},
			['head1', {}, "Document index for $namespace"],
			['Para', {},
				map {("* ", ['link', {to => $_}, $_], "\n")} @pages
			]
		] );
	}
	
}

sub _export_page {
	my ($self, $exporter, $page) = @_;
	_check_page_name($page);
	
	my $source = $self->get_page($page);
	return 0 unless $source->exists;
	print STDERR "Exporting: $page\n";
	
	my $target = $exporter->get_page($page);
	$target->clone($source);
	
	return 1;
}

=item C<add_child(NAMESPACE, CHILD, ...)>

This will connect a new child object to this repository under a certain
namespace. CHILD is a class name implementing the repository interface.
The class will be looked for in C<Zim::Repository::>.
All remaining arguments will be passed on to the constructor of the object.

=cut

sub add_child {
	my ($self, $namespace, $child, @args) = @_;
	$namespace =~ s/:?$/:/;
	_check_namespace($namespace);
	#warn "Adding child  type '$child' for $namespace @args\n";
	
	$child = "Zim::Repository::$child" unless $child =~ /::/;
	eval "use $child ();";
	die if $@;
	$child = $child->new($self, $namespace, @args);
	$self->{$namespace} = $child;

	my $regex = join '|', reverse
		sort {length($a) <=> length($b)} grep s/^://, keys %$self;
	$self->{regex} = qr#$self->{namespace}(?i)($regex)#;
	#warn "regex: $self->{regex}\n";
}

=back

=head2 Utility Methods

These methods can be used by the interface but do not need to be reproduced
in child handlers.

=over 4

=item C<split_page(PAGE, LEVEL)>

Splits a page into multiple pages in the like-named namespace.
Splits the page at headings up to level LEVEL (1..5 default 1).

This method can only be used for page objects that support the
"formatted" interface like L<Zim::Page::Text>.

=cut

# TODO - maybe have a module with pare-tree logic would make this method a bit smaller ?

sub split_page {
	my ($self, $page, $max) = @_;
	_check_page($page);
	$page = $self->get_page($page) unless ref $page;
	$max ||= 1;

	die "Page does not exist\n" unless $page;
	die "Page ".$page->name." does not support the 'formatted' interface\n"
		unless $page->has_interface('formatted');

	# Build nested structure of headings
	my $root = [0, $page->basename, []]; # [level, title, [children], content]
	my @stack = ($root);
	my $tree = $page->get_parse_tree;
	for my $part (@$tree[2 .. $#$tree]) {
		if (! ref $part or $$part[0] !~ /^head/) { push @{$stack[-1]}, $part }
		else { # TODO check head not empty
			$$part[0] =~ /^head(\d+)/;
			my $level = $1;
			if ($level > $max) {
				push @{$stack[-1]}, $part;
				next;
			}
			
			my $title = _to_string($part);
			if ($stack[-1] eq $root) { # match head to page name - TODO better compare logic
				my $name = $title;
				$name =~ s/\W/_/;
				if (lc($name) eq lc($$root[1])) {
					push @{$stack[-1]}, $part;
					next;
				}
			}
			else { # find parent
				until ($stack[-1][0] < $level) { pop @stack }
			}
			
			my $item = [$level, $title, [], $part];
			push @{$stack[-1][2]}, $item; # attach to parent
			push @stack, $item; # become the new head of the tree
		}
	}
	
	# Write files
	_save_head_page($self, $page->name, $_) for @{$$root[2]};
	# Write root files last .. if anything goes wrong it should still be there
	$tree = [ @$tree[0, 1], @$root[3 .. $#$root] ];
	$page->set_parse_tree($tree);
}

sub _to_string { # convert parse tree node to plain text (no formatting)
	my $ref = shift;
	my $string = '';
	my @todo = @$ref[2 .. $#$ref];
	while (@todo) {
		my $n = shift @todo;
		if (ref $n) { unshift @todo, $$n[2 .. $#$n] }
		else        { $string .= $n                 }
	}
	return $string;
}

sub _save_head_page {
	# TODO promote heading => lvl 1
	my ($self, $path, $ref) = @_;
	$path = $self->resolve_page($path.':'.$$ref[1]);
	my $page = $self->get_page($path);
	my $tree = $page->exists ? ['Document', {}] : $page->get_parse_tree;
	push @$tree, @$ref[3 .. $#$ref];
	$page->set_parse_tree($tree);
	_save_head_page($self, $path, $_) for @{$$ref[2]};
}

=item C<merge_page(NAME)>

TODO - work in progress

=cut

sub merge_page {
	my ($self, $page) = @_;
	_check_page($page);
	$page = $self->get_page($page) unless ref $page;

	die "Could not open page\n" unless $page;
	die "Page ".$page->name." does not support the 'formatted' interface\n"
		unless $page->has_interface('formatted');

	my @pages = _list_pages($self, $page->name, 1);
	warn join "\n", "Merging:", map "\t@$_", @pages;

	# demote levels => lvl
}

sub _list_pages {
	my ($self, $name, $lvl) = @_;
	$name =~ s/:$//;
	my @list;
	for ($self->list_pages($name)) {
		my $p = "$name:$_";
		my $r = ($p =~ s/:$//);
		push @list, [$lvl, $p];
		push @list, _list_pages($self, $p, $lvl+1) if $r; # recurs
	}
	return @list;
}

1;

__END__

=back

=head1 BUGS

Please mail the author if you find any bugs.

=head1 AUTHOR

Jaap Karssenberg || Pardus [Larus] E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

L<Zim>, L<Zim::Repository::File>, L<Zim::Repository::DBI>

=cut
