package Zim;

use strict;
use vars qw/$AUTOLOAD %Config/;
use Gtk2;
use Gtk2::Gdk::Keysyms;
use Gtk2::SimpleList;
use Zim::History;
use Zim::Components::TreeView;
use Zim::Components::PathBar;
use Zim::Components::PageView;
#use Zim::Components::SearchPane;

# TODO make components configable/autoloading like plugins

our $VERSION = '0.12';
our $LONG_VERSION = << "EOT";
zim $VERSION - A desktop wiki and outliner

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

Please report bugs to pardus\@cpan.org
EOT

=head1 NAME

Zim - The application object for zim

=head1 SYNOPSIS

	use Zim;
	use Zim::Repository;
	
	my $zim = Zim->new(\%SETTINGS);
	
	my $repository = Zim::Repository->new($DIR);
	$zim->set_repository($repository);
	
	$zim->main_loop;
	
	exit;

=head1 DESCRIPTION

This is developer documentation, for the user manual try
executing C<zim --doc>. For commandline options see L<zim>(1).

This module provides the application object for the Gtk2 application B<zim>.
The application has been split into several components for which the 
modules can be found in the ZIM::Components:: namespace. This object brings
together these components and manages the settings and data objects.

=head1 EXPORT

None by default.

=head1 METHODS

Undefined methods are AUTOLOADED either to an component or to the main
L<Gtk2::Window> object.

=over 4

=cut

our $LINK_ICON = Gtk2->CHECK_VERSION(2, 6, 0) ? 'gtk-connect' : 'gtk-convert';
	# gtk-connect stock item was introduced in 2.6.0

our %CONFIG = (  # Default config values
	pane_pos	=> 120,
	pane_vis	=> 0,
	statusbar_vis	=> 1,
	toolbar_vis	=> 1,
	pathbar_type	=> 'recent',
	#icon_size	=> 'large-toolbar',
	width		=> 500,
	height		=> 350,
	x		=> undef,
	y		=> undef,
	default_root	=> undef,
	default_home	=> 'Home',
	browser		=> undef,
	file_browser	=> undef,
	date_string	=> '%A %d/%m/%Y',
	hist_max	=> 20,
	recent_show	=> 7,
	undo_max	=> 50,
	follow_new_link => 1,
	backsp_unindent => 1,
	use_xdg_cache	=> 0,
	use_tray_icon	=> 0,
	use_gtkspell	=> 0,
	show_spell      => 1,
	spell_language	=> undef,
	use_calendar	=> 0,
	cal_namespace	=> ':Date:',
	show_cal	=> 0,
);

my @ui_actions = (
	# name,          stock id,  label
	[ 'FileMenu',    undef,     '_File'         ],
	[ 'EditMenu',    undef,     '_Edit'         ],
	[ 'ViewMenu',    undef,     '_View'         ],
	[ 'InsertMenu',  undef,     '_Insert'       ],
	[ 'FormatMenu',  undef,     'For_mat'       ],
	[ 'ToolsMenu',   undef,     '_Tools'        ],
	[ 'GoMenu',      undef,     '_Go'           ],
	[ 'HelpMenu',    undef,     '_Help'         ],
	[ 'PathBarMenu', undef,     'P_athbar type' ],
	# name,       stock id,         label,          accelerator,   tooltip
	[ 'OpenRep',  'gtk-open',       '_Open...',     '<ctrl>O',     'Open',         \&on_OpenRep  ],
	[ 'Save',     'gtk-save',       '_Save',        '<ctrl>S',     'Save page',    \&on_Save     ],
	[ 'Rename',   undef,            '_Rename page', 'F2',          'Rename page',  \&on_Rename   ],
	[ 'Close',    'gtk-close',      '_Close',       '<ctrl>W',     'Close window', \&on_Close    ],
	[ 'Quit',     'gtk-quit',       '_Quit',        '<ctrl>Q',     'Quit',         \&on_Quit     ],
	[ 'Reload',   'gtk-refresh',    '_Reload',      '<ctrl>R',     'Reload',       \&on_Reload   ],
	[ 'OpenDir',  'gtk-open',       'Open _folder', undef,         'Open folder',  \&on_OpenDir  ],
	[ 'Back',     'gtk-go-back',    '_Back',        '<alt>Left',   'Go back',      \&on_Back     ],
	[ 'Forward',  'gtk-go-forward', '_Forward',     '<alt>Right',  'Go forward',   \&on_Forward  ],
	[ 'RBack',    undef,            '_Back',        '<alt>Page_Down', undef,       \&on_RBack    ],
	[ 'RForward', undef,            '_Forward',     '<alt>Page_Up',   undef,       \&on_RForward ],
	[ 'Parent',   'gtk-go-up',      '_Parent',      '<alt>Up',     'Go to parent', \&on_Parent   ],
	[ 'Child',    'gtk-go-down',    '_Child',       '<alt>Down',   'Go to child',  \&on_Child    ],
	[ 'Home',     'gtk-home',       '_Home',        '<alt>Home',   'Go home',      \&on_Home     ],
	[ 'JumpTo',   'gtk-jump-to',    '_Jump to...',  '<ctrl>J',     'Jump to page', \&on_JumpTo   ],
	[ 'Help',     'gtk-help',       '_Help',        'F1',          'Help',         \&on_Help     ],
	[ 'About',    'gtk-about',      '_About',       undef,         'About',        \&on_About    ],
);
my @ui_toggle_actions = (
	# name,         stock id,    label,        accelerator, tooltip 
	[ 'TToolBar',   undef,       '_Toolbar',   undef,       'Show toolbar',   \&on_TToolBar,   0],
	[ 'TStatusBar', undef,       '_Statusbar', undef,       'Show statusbar', \&on_TStatusBar, 0],
	[ 'TPane',      'gtk-index', 'Side _Pane', 'F9',        'Show side pane', \&on_TPane,      0],
);
my @ui_radio_actions = (
	# name,          stock id, label,           accelerator, tooltip, value
	[ 'PBRecent',    undef,    '_Recent pages', undef,       '',      0],
	[ 'PBHistory',   undef,    '_History',      undef,       '',      1],
	[ 'PBNamespace', undef,    '_Namespace',    undef,       '',      2],
	[ 'PBHidden',    undef,    'H_idden',       undef,       '',      3],
);
my %pathbar_map = ( 'recent' => 0, 'history' => 1, 'namespace' => 2, 'hidden' => 3);

my $ui_layout = q{<ui>
	<menubar name='MenuBar'>
		<menu action='FileMenu'>
			<menuitem action='OpenRep'/>
			<separator/>
			<menuitem action='Save'/>
			<separator/>
			<placeholder name='FileMods'>
				<menuitem action='Rename'/>
			</placeholder>
			<separator/>
			<menuitem action='Close'/>
			<menuitem action='Quit'/>
		</menu>
		<menu action='EditMenu'></menu>
		<menu action='ViewMenu'>
			<menuitem action='TToolBar'/>
			<menuitem action='TStatusBar'/>
			<menuitem action='TPane'/>
			<menu action='PathBarMenu'>
				<menuitem action='PBRecent'/>
				<menuitem action='PBHistory'/>
				<menuitem action='PBNamespace'/>
				<menuitem action='PBHidden'/>
			</menu>
			<separator/>
			<placeholder name='PluginItems'>
			</placeholder>
			<separator/>
			<menuitem action='Reload'/>
		</menu>
		<menu action='InsertMenu'></menu>
		<menu action='FormatMenu'></menu>
		<menu action='ToolsMenu'>
			<menuitem action='OpenDir'/>
		</menu>
		<placeholder name='PluginMenus'></placeholder>
		<menu action='GoMenu'>
			<menuitem action='Back'/>
			<menuitem action='Forward'/>
			<menuitem action='Parent'/>
			<separator/>
			<placeholder name='PluginItems'>
			</placeholder>
			<separator/>
			<menuitem action='Home'/>
			<menuitem action='JumpTo'/>
		</menu>
		<menu action='HelpMenu'>
			<menuitem action='Help'/>
			<menuitem action='About'/>
		</menu>
	</menubar>
	<toolbar name='ToolBar'>
		<placeholder name='View'>
			<toolitem action='TPane'/>
		</placeholder>
		<separator/>
		<placeholder name='Go'>
			<toolitem action='Home'/>
			<toolitem action='Back'/>
			<toolitem action='Forward'/>
		</placeholder>
		<separator/>
	</toolbar>
	<accelerator action='Child'/>
	<accelerator action='RBack'/>
	<accelerator action='RForward'/>
</ui>};

=item C<new(SETTINGS)>

Simple constructor.

=cut

sub new {
	my ($class, $settings) = @_;
	
	%$settings = (%CONFIG, %$settings); # set defaults
	$$settings{root} ||= $$settings{default_root};
	$$settings{home} ||= $$settings{default_home};
	my $self = bless {settings => $settings}, $class;
	$self->load_config;

	$self->{settings}{pathbar_type} = 'recent'
		if $self->{settings}{pathbar_type} eq 'trace';
		# reset 0.08 default value

	$self->{_message_timeout} = -1;
	$self->{_save_timeout} = -1;
	
	return $self;
}

sub DESTROY {
	my $self = shift;
	for (qw/_save_timeout _message_timeout/) {
		my $timeout = $self->{$_};
		Glib::Source->remove($timeout) if $timeout >= 0;
	}
}

sub AUTOLOAD {
	my $self = shift;
	$AUTOLOAD =~ s/^.*:://;
	return if $AUTOLOAD eq 'DESTROY';
	#warn join ' ', "Zim::AUTOLOAD called for $AUTOLOAD by: ", caller, "\n";
	return $self->{objects}{$AUTOLOAD} if exists $self->{objects}{$AUTOLOAD};
	return $self->{window}->$AUTOLOAD(@_);
}

=item C<set_repository(REPOSITORY)>

Set the repository object.

=cut

sub set_repository { $_[0]->{repository} = $_[1] }

=item C<gui_init()>

This method initializes all GUI objects that make up the application.

=cut

sub gui_init {
	my $self = shift;

	## Setup the window
	my $window = Gtk2::Window->new('toplevel');
	$window->set_default_size(@{$self->{settings}}{'width', 'height'});
	$window->signal_connect_swapped(delete_event => \&on_delete_event, $self);
	$window->signal_connect(destroy => sub { Gtk2->main_quit });
	$window->set_icon(
		Gtk2::Gdk::Pixbuf->new_from_file($self->{settings}{icon_file}) );
	$window->set_title('Zim');
	$self->{window} = $window;

	#$window->signal_connect(hide => sub { # FIXME does not work - patched in TrayIcon
	#		@{$self->{settings}}{'x', 'y'} = $window->get_position;
	#		warn "hiding window: @{$self->{settings}}{'x', 'y'}\n";
	#	} );
	$window->signal_connect(show => sub {
			my ($x, $y) = @{$self->{settings}}{'x','y'};
	#		warn "showing window: $x, $y\n";
			$window->move($x,$y) if defined($x) and defined ($y);
		} );
			

	my $vbox = Gtk2::VBox->new(0, 0);
	$window->add($vbox);
	$self->{vbox} = $vbox;

	## Setup actions and ui for menubar and toolbar
	my $actions = Gtk2::ActionGroup->new("Actions");
	$actions->add_actions(\@ui_actions, $self);
	$actions->add_toggle_actions(\@ui_toggle_actions, $self);
	$actions->add_radio_actions(\@ui_radio_actions,
		$pathbar_map{$self->{settings}{pathbar_type}}, \&on_TPathBar, $self);
	#$actions->get_action('Save')->set_sensitive(0) if $self->{settings}{read_only};
	#$actions->get_action('Back')->set_sensitive(0);
	#$actions->get_action('Forward')->set_sensitive(0);
	#$actions->get_action('Parent')->set_sensitive(0);
	_gtk_action_set_sensitive($actions->get_action('Save'), 0)
		if $self->{settings}{read_only};
	_gtk_action_set_sensitive($actions->get_action('Back'), 0);
	_gtk_action_set_sensitive($actions->get_action('Forward'), 0);
	_gtk_action_set_sensitive($actions->get_action('Parent'), 0);
	$self->{actions} = $actions;
	
	my $ui = Gtk2::UIManager->new;
	$ui->insert_action_group($actions, 0);
	$window->add_accel_group($ui->get_accel_group);
	$ui->add_ui_from_string($ui_layout);
	$self->{menubar} = $ui->get_widget("/MenuBar");
	$vbox->pack_start($self->{menubar}, 0,1,0);
	$self->{toolbar} = $ui->get_widget("/ToolBar");
	$vbox->pack_start($self->{toolbar}, 0,1,0);
	$self->{toolbar}->hide;
	$self->{toolbar}->set_no_show_all(1);
	$self->{ui} = $ui;

	## General window layout
	my $hpaned = Gtk2::HPaned->new();
	$hpaned->set_position($self->{settings}{pane_pos});
	$vbox->pack_start($hpaned, 1,1,0);
	$self->{hpaned} = $hpaned;

	my $l_vbox = Gtk2::VBox->new(0, 0);
	$hpaned->add1($l_vbox);
	$l_vbox->set_no_show_all(1);
	$self->{l_vbox} = $l_vbox;

	my $r_vbox = Gtk2::VBox->new(0, 0);
	$hpaned->add2($r_vbox);
	$self->{r_vbox} = $r_vbox;

	my $statusbar = Gtk2::Statusbar->new;
	$statusbar->set_no_show_all(1);
	$vbox->pack_start($statusbar, 0,1,0);
	$self->{statusbar} = $statusbar;
	
	## Build the content of the side pane
#	my $l_top_hbox = Gtk2::HBox->new(0, 0);
#	$l_vbox->pack_start($l_top_hbox, 0,1,0);
	
#	my $pane_combo = Gtk2::ComboBox->new_text;
#	$pane_combo->signal_connect(changed =>
#		sub { $self->show_in_pane($pane_combo->get_active_text()) } );
#	# FIXME get_active_text is since 2.6
#	$l_top_hbox->add($pane_combo);
#	$self->{pane_combo} = $pane_combo;

	my $close_button = Gtk2::Button->new;
	$close_button->set_relief('none');
	$close_button->add(Gtk2::Alignment->new(0.5, 0.5, 0, 0));
	$close_button->child->add(
		Gtk2::Image->new_from_stock('gtk-close', 'small-toolbar') );
	$close_button->signal_connect(clicked =>
		sub { $actions->get_action('TPane')->set_active(0) } );
#	$l_top_hbox->pack_end($close_button, 0,1,0);
	
	my $notebook = Gtk2::Notebook->new;
	$notebook->set_show_tabs(0);
	$notebook->set_show_border(0);
	$l_vbox->add($notebook);
	$self->{notebook} = $notebook;
	
	my $tree_view = Zim::Components::TreeView->new(app => $self);
	$tree_view->signal_connect(row_activated =>
		sub { $self->toggle_pane(0) if $self->{_pane_visible} == -1 } );
	$self->add_to_pane($tree_view->widget, 'Index');
	$self->{objects}{TreeView} = $tree_view;

	#my $search_pane = Zim::Components::SearchPane->new(app => $self);
	#$self->add_to_pane($search_pane->widget, 'Search');
	#$self->{objects}{SearchPane} = $search_pane;
	
#	$pane_combo->set_active(0); # FIXME should be setting
	
	## Build content of the main area
	my $path_bar = Zim::Components::PathBar->new(app => $self);
	$r_vbox->pack_start($path_bar->widget, 0,1,5);
	$self->{objects}{PathBar} = $path_bar;

	my $page_view = Zim::Components::PageView->new(app => $self);
	$r_vbox->add($page_view->widget);
	$self->{objects}{PageView} = $page_view;

	if ($self->{settings}{use_xdg_cache}) {
		my $root = File::Spec->rel2abs($self->{settings}{root_dir});
		my ($vol, $dirs) = File::Spec->splitpath($root, 'NO_FILE');
		my @dirs = File::Spec->splitdir($dirs);
		$self->{settings}{hist_file} ||= File::Spec->catfile( 
			File::BaseDir->xdg_cache_home, 'zim', join ':', $vol, @dirs );
	}
	else {
		my $root = File::Spec->rel2abs($self->{settings}{root_dir});
		$self->{settings}{hist_file} ||= File::Spec->catfile($root, '.zim.history'); 
	}
	my $history = Zim::History->new(
		$self->{settings}{hist_file}, $page_view,
		$self->{settings}{hist_max} );
	$self->{objects}{History} = $history;

	## Some wiring
	$ui->get_accel_group->connect( # Ctrl-Space
		ord(' '), ['control-mask'], ['visible'],
		sub {
			if ($self->{settings}{pane_vis}) {
				if ($tree_view->has_focus) { $page_view->grab_focus }
				else                       { $tree_view->grab_focus }
			}
			else {
				my $vis = $self->{_pane_visible} ? 0 : -1;
				$self->toggle_pane($vis);
				$vis ? $tree_view->grab_focus : $page_view->grab_focus ;
			}
		} );
	
	$self->signal_connect('page_loaded', sub {
		my $self = shift;
		my $state = $self->History->get_state;
		#$self->{actions}->get_action('Back')->set_sensitive($$state{back} ? 1:0);
		#$self->{actions}->get_action('Forward')->set_sensitive($$state{forw} ? 1:0);
		_gtk_action_set_sensitive($self->{actions}->get_action('Back'), $$state{back} ? 1:0);
		_gtk_action_set_sensitive($self->{actions}->get_action('Forward'), $$state{forw} ? 1:0);
		my $namespace = ($self->{page}->namespace =~ /[^:]/) ? 1 : 0;
		#$self->{actions}->get_action('Parent')->set_sensitive($namespace);
		_gtk_action_set_sensitive($self->{actions}->get_action('Parent'), $namespace);
	} );

	## Some Optional features
	if ($self->{settings}{use_tray_icon}) {
		eval {
			eval 'use Zim::Components::TrayIcon'; die $@ if $@;
			my $trayicon = Zim::Components::TrayIcon->new(app => $self);
			$self->{objects}{TrayIcon} = $trayicon;
		} ;
		$self->error_dialog(
			"Failed to load TrayIcon,\n".
			"do you have Gtk2::TrayIcon installed ?", $@) if $@;
	}
	if ($self->{settings}{use_gtkspell}) {
		eval {
			eval 'use Zim::Components::Spell'; die $@ if $@;
			my $spell = Zim::Components::Spell->new(app => $self);
			$self->{objects}{Spell} = $spell;
		} ;
		$self->error_dialog(
			"Failed to load spell checking,\n".
			"do you have Gtk2::Spell installed ?", $@) if $@;
	}
	if ($self->{settings}{use_calendar}) {
		eval {
			eval 'use Zim::Components::Calendar'; die $@ if $@;
			my $cal = Zim::Components::Calendar->new(app => $self);
			$self->{objects}{Calendar} = $cal;
		} ;
		$self->error_dialog("Failed to load calendar", $@) if $@;
	}

	## Show all widgets .. well most of them
	$window->show_all;
	$statusbar->set_no_show_all(0);
	$l_vbox->set_no_show_all(0);
	$self->{toolbar}->set_no_show_all(0);
#	my ($x, $y) = @{$self->{settings}}{'x','y'};
#	$window->move($x,$y) if defined($x) and defined ($y);
	
	## Try saving the program on system signals
	for my $sig (qw/TERM HUP PIPE/) {
		$SIG{$sig} = sub {
			warn "Signal $sig caught\n";
			$self->quit('FORCE');
			exit 1;
		};
	}

	# Try saving on desktop exit
#	$window->get_display->signal_connect( closed => sub {
#			print "Display closed\n";
#		} );
}

sub _gtk_action_set_sensitive { # **sigh**
	my ($action, $bit) = @_;
	if (Gtk2->CHECK_VERSION(2, 6, 0)) { $action->set_sensitive($bit) }
	else { $_->set_sensitive($bit) for $action->get_proxies }
}

sub on_OpenRep {
	my $self = pop;
	my $rep = $self->prompt_repository_dialog;
	$self->exec_new_window($rep) if defined $rep;
}

sub on_Save { pop->save_page('FORCE') }

sub on_Rename { pop->rename_page() }

sub on_Close { pop->quit() }

sub on_Quit { pop->quit('FORCE') }

sub on_Reload { pop->reload }

sub on_OpenDir {
	my $self = pop;
	my $dir = $self->{page}->properties->{base};
	return unless length $dir;
	#warn "Opening $dir\n";
	$self->link_clicked($dir);
}

sub on_TToolBar {
	my $self = pop;
	my $vis = $self->{toolbar}->visible ? 0 : 1;
	$vis ? $self->{toolbar}->show : $self->{toolbar}->hide ;
	$self->{settings}{toolbar_vis} = $vis;
}

sub on_TStatusBar {
	my $self = pop;
	my $vis = $self->{statusbar}->visible ? 0 : 1;
	$vis ? $self->{statusbar}->show : $self->{statusbar}->hide ;
	$self->{settings}{statusbar_vis} = $vis;
}

sub on_TPane {
	my $self = pop;
	my $vis = $self->{l_vbox}->visible ? 0 : 1;
	$self->toggle_pane($vis)
}

sub on_TPathBar {
	my $self = pop;
	my $type = lc pop->get_name;
	$type =~ s/^pb//;
	$self->toggle_path_bar($type);
}

sub on_Back { pop->go_back(1) }

sub on_Forward { pop->go_forw(1) }

sub on_RBack {
	my $self = pop;
	my ($i, @recent) = $self->History->get_recent;
	return unless $i > 0;
	my $rec = $recent[$i-1];
	$self->load_page($rec);
}

sub on_RForward {
	my $self = pop;
	my ($i, @recent) = $self->History->get_recent;
	return unless $i < $#recent;
	my $rec = $recent[$i+1];
	$self->load_page($rec);
}

sub on_Parent {
	my $self = pop;
	my $namespace = $self->{page}->namespace;
	return if $namespace eq ':';
	$self->load_page($namespace);
}

sub on_Child {
	my $self = pop;
	my $namespace = $self->History->get_namespace;
	my $name = $self->{page}->name;
	return unless $namespace =~ /^(:*$name:+[^:]+)/;
	$self->load_page($1);
}

sub on_Home {
	my $self = pop;
	my $home = $self->{repository}->resolve_page($self->{settings}{home});
	$self->load_page($home);
}

sub on_JumpTo { pop->jump_page_dialog }

sub on_Help { pop->show_help }

sub on_About { pop->about_dialog }

=item C<widget()>

Returns the root window widget.
Use this widget for things like show_all() and hide_all().

=cut

sub widget { return $_[0]->{window} }

=item C<signal_connect(NAME, CODE, DATA)>

=cut

sub signal_connect {
	my ($self, $signal, $code, @data) = @_;
	$self->{_signals}{$signal} ||= [];
	push @{$self->{_signals}{$signal}}, [$code, @data];
}

=item C<signal_emit(NAME, ARGS)>

=cut

sub signal_emit {
	my ($self, $signal, @args) = @_;
	return unless exists $self->{_signals}{$signal};
	for (grep defined($_), @{$self->{_signals}{$signal}}) {
		my ($code, @data) = @$_;
		eval { $code->($self, @args, @data) };
		warn if $@;
	}
}

=item C<load_config>

Read config file.

=cut

sub load_config {
	my $self = shift;
	my $name = $self->{settings}{read_only} ? 'conf.ro' : 'conf.rw' ;
	my ($file) =
		grep { -f $_ && -r $_ }
		map  { File::Spec->catdir($_, 'zim', $name) }
		( File::BaseDir->xdg_config_home, File::BaseDir->xdg_config_dirs );

	return unless defined $file;
	$self->{settings}{conf_file} = $file;
	
	open CONF, $file or $self->exit_error("Could not read\n$file");
	while (<CONF>) {
		/^(.+?)=(.*)$/ or next;
		$self->{settings}{$1} = $2 if length $2;
	}
	close CONF;
}

=item C<save_config>

Save config file.

=cut

sub save_config {
	my $self = shift;
	$self->{settings}{pane_pos} = $self->{hpaned}->get_position;
	
	my $dir  = File::Spec->catdir(
		File::BaseDir->xdg_config_home, 'zim');
	_mkdir($dir) unless -e $dir;
	my $name = $self->{settings}{read_only} ? 'conf.ro' : 'conf.rw' ;
	my $file = File::Spec->catfile($dir, $name);
	
	open CONF, ">$file" or $self->exit_error("Could not write config\n$file");
	print CONF "# zim version:$Zim::VERSION\n";
	print CONF "$_=$$self{settings}{$_}\n"
		for sort keys %CONFIG; # we do not save all settings
	close CONF;
	#print "saved config to $file\n";
}

sub _mkdir {
	my ($vol, $dirs) = (@_ == 1)
		 ? (File::Spec->splitpath(shift(@_), 'NO_FILE')) : (@_) ;
	my @dirs = File::Spec->splitdir($dirs);
	my $path = File::Spec->catpath($vol, shift @dirs);
	mkdir $path or die "Could not create dir $path\n"
		if length $path and ! -d $path;
	while (@dirs) {
		$path = File::Spec->catdir($path, shift @dirs);
		mkdir $path or die "Could not create dir $path\n"
			unless -d $path;
	}
}


=item C<main_loop()>

This method runs the GUI application.

=cut

sub main_loop {
	my $self = shift;
	
	## Find a page to load
	unless ($self->{page}) {
		my $page = $self->{objects}{History}->get_current
			|| $self->{settings}{home};
		$page = $self->{repository}->resolve_page($page) unless ref $page;
		$self->load_page($page);
		# FIXME what if we are read_only and page does not exist ?
		# 	load_apge returns undef
		# 	first try home, but might also not exist
		# 	need page that always exists ... like about or version page
	}
	
	## Toggle some widgets
	$self->{message_lock} = 1;
	$self->{objects}{TreeView}->load_index();
	my ($actions, $settings) = ($self->{actions}, $self->{settings});
	$actions->get_action('TToolBar')->set_active($settings->{toolbar_vis});
	$actions->get_action('TStatusBar')->set_active($settings->{statusbar_vis});
	$actions->get_action('TPane')->set_active($settings->{pane_vis});
	$self->toggle_path_bar($self->{settings}{pathbar_type});
	$self->{message_lock} = 0;

	$self->{objects}{PageView}->grab_focus;

	## set save interval
	$self->{_save_timeout} = 
		Glib::Timeout->add(5000, sub {$self->save_page; 1})
		unless $self->{settings}{read_only};
	
	$self->signal_emit('main');
	Gtk2->main;
}

=item C<quit()>

Exit the main loop.

=cut

sub quit {
	my $self = shift;
	$self->on_delete_event(@_) && return;
	Gtk2->main_quit;
}

sub on_delete_event {
	# Called when the window is deleted
	# return true to avoid destruction
	my $self = shift;
	my $force = uc(shift) eq 'FORCE';

	eval {
		my ($settings, $window) = @{$self}{'settings', 'window'};
		@{$settings}{'width', 'height'} = $window->get_size;
		@{$settings}{'x', 'y'} = $window->get_position if $window->visible;
			# position is 0,0 when window is hidden
	};
	
	if ($self->{settings}{hide_on_delete} and !$force) {
		$self->{window}->hide;
		return 1;
	}
	
	$self->{save_lock} = 0; # just to be sure
	$self->save_page() || return 1;
	$self->save_config;

	unless ($self->{settings}{read_only}) {
		my ($vol, $dirs, undef) =
			File::Spec->splitpath($self->{settings}{hist_file});
		my $dir = File::Spec->catpath($vol, $dirs);
		$self->{objects}{History}->write if -d $dir;
			# Only safe history if directory exists ..
	}

	if ($self->{lock_file}) {
		unlink $self->{lock_file}
			or warn "Could not remove $self->{lock_file}\n";
	}

	return 0;
}

=item C<link_clicked(LINK)>

Loads a page in zim or open an external uri in a browser.
LINK is considered to be either an url or a page name.
Page names are resolved as relative links first.

=cut

sub link_clicked {
	my ($self, $link) = @_;
	$link = $$link[1] if ref $link; # link_data = [bit, link]
	#warn "link clicked: >>$link<<\n";
	return warn "Warning: You tried to folow an empty link.\n"
		unless length $link;

	if ($link =~ /^(?:(\w+):\/\/|mailto:)/) { # link is an url
		my $type = ($1 eq 'file') ? 'file_browser' : 'browser';
		my $browser = $self->{settings}{$type};
		$browser ||= $ENV{BROWSER} if $type eq 'browser';
		$link =~ s#^file://## if $type eq 'file_browser';
		unless ($browser) {
			$browser = $self->prompt_browser_dialog($type);
			return $self->error_dialog('You have no browser configured')
				unless $browser;
			$self->{settings}{$type} = $browser;
		}
		$browser =~ s/\%s/$link/ or $browser .= ' '.$link;
		#warn "Executing: $browser\n";
		unless (fork) { # child process
			exec $browser;
			exit 1; # just to be sure
		}
	}
	else {
		$self->load_page(
			$self->{page}->resolve_link($link) );
	}
}

=item C<go_back(INT)>

Go back one or more steps in the history stack.

=cut

sub go_back {
	my $self = shift;
	my $i = shift || 1;
	my $rec = $self->{objects}{History}->back($i) || return;
	$self->load_page($rec);
}

=item C<go_forw(INT)>

Go forward one or more steps in the history stack.

=cut

sub go_forw {
	my $self = shift;
	my $i = shift || 1;
	my $rec = $self->{objects}{History}->forw($i) || return;
	$self->load_page($rec);
}

=item C<reload()>

Save and reload the current page.

=cut

sub reload {
	my $self = shift;
	$self->load_page(
		$self->{objects}{History}->get_current );
}

=item C<load_page(PAGE)>

Loads a new page in the PageView, updates history etc. when
necessary. PAGE should be either an absolute page name, a history record
or a page object.

use C<link_clicked()> for relative page names, urls etc.

=cut

sub load_page {
	my ($self, $page) = @_;
	
	if ($self->{page}) {
		$self->{save_lock} = 0; # just to be sure
		$self->save_page || return;
	}

	my ($rec, $name);
	($name, $page, $rec) = 
		(ref($page) eq 'HASH') ? ($page->{name}, undef, $page) :
		ref($page)             ? ($page->name,   $page, undef) : ($page, undef, undef) ;
	
	unless ($page) { # get object
		return warn "Warning: You tried to load an empty name.\n" unless length $name;
		
		eval { $page = $self->{repository}->get_page($name) };
		return $self->error_dialog("Could not load page $name", $@) if $@;
		return $self->error_dialog("Page does not exist:\n$page")
			if ! defined $page
			or $self->{settings}{read_only} and ! $page->exists ;
	}
	unless ($rec) { # get history record
		$rec = $self->{objects}{History}->get_record($name);
	}
	
	$self->{objects}{History}->set_current($page);
	eval { $self->{objects}{PageView}->load_page($page) };
	if ($@) {
		$self->exit_error("Could not load page $name", $@);
		# TODO handle more gratiously
	}
	
	$self->{page} = $page;
	$self->{objects}{PageView}->set_state(%$rec) if defined $rec;
	$self->signal_emit('page_loaded', $name);
	
	$self->{window}->set_title("$name - Zim");
	$self->update_status();
}

=item C<save_page(FORCE)>

Check if the current page needs to be saved. If FORCE is true the page is saved
whether it was modified or not. Wrapper for C<Zim::Repository->save_page()>.

=cut

# TODO split out to save_page_if_modified() and save_page()
# save_page_if_modified(FORCE) overrules save_lock
# save_page() just saves it, no questions asked
# (maybe call the first one check_save_page() ?)

# TODO part of this logic belongs in the PageView component

sub save_page {
	my ($self, $force) = @_;
	my $modified = $self->{objects}{PageView}->modified;
	return 0 if $self->{app}{settings}{read_only};
	unless ($force) {
		return 1 if ! $modified;
		return 0 if $self->{save_lock};
	}
	
	$self->{save_lock} = 1; # Used to block autosave while "could not save" dialog
	
	my $page = $self->{page};
	my $tree = $self->{objects}{PageView}->get_parse_tree;
	
	eval { # FIXME maybe this logic belongs in PageView
		my $old_status = $page->status;
		if (@$tree > 2) { # content
			$page->set_parse_tree($tree);
			$self->signal_emit('page_created', $page->name)
				if $old_status eq 'new' or $old_status eq 'deleted';
		}
		else {
			$page->delete;
			$self->signal_emit('page_deleted', $page->name);
		}
	};
	unless ($@) {
		$self->{objects}{PageView}->modified(0);
		$self->signal_emit('page_saved', $page->name);
	}
	else { # FIXME this dialog could be a lot better
		my $page = $self->prompt_page_dialog($page->name,
			'Save As', 'Save Page As:', $@, 'gtk-dialog-error' );
		return 0 unless defined $page;
			# leave lock in position after returning
		# FIXME check logic here !
		$self->{page} = $self->{repository}->get_page($page);
		$self->save_page('FORCE'); # recurs
	}

	$self->update_status();
	$self->{save_lock} = 0;
	return 1;
}

=item C<rename_page(FROM, TO)>

Wrapper for C<Zim::Repository->move_page()>.

Move page from FROM to TO. If TO is undefined a dialog is shown to ask for a page name.

=cut

sub rename_page {
	my ($self, $from, $to) = @_;

	$from = $self->{page}->name unless defined $from;
	
	my $move_current = ($from eq $self->{page}->name);
	if ($move_current) { # FIXME clearer interface for this
		$self->{save_lock} = 0 if $move_current;
		$self->save_page or return;
	}
	else { $self->save_page }
	
	unless (defined $to) {
		$to = $self->prompt_rename_page_dialog($from);
		return unless defined $to;
		$to = $self->{repository}->resolve_page($to);
		# TODO check for things like ':'
	}

	#warn "Move '$from' to '$to'\n";
	my $page;
	eval { $page = $self->{repository}->move_page($from, $to) };
	return $self->error_dialog("Could not move $from to $to\n\n$@") if $@;

	# TODO trigger this with signals
	# problem is that we might have moved a whole namespace
	$self->{objects}{TreeView}->{_loaded} = 0;
	$self->{objects}{TreeView}->load_index;

	$self->load_page($page) if $move_current;
	
	$self->update_status;
	return 1;
}

=item C<delete_page(PAGE)>

Wrapper for C<Zim::Repository->delete_page>.
Asks the user for confirmation.

If PAGE is undefined the current page is deleted.

=cut

sub delete_page {
	# FIXME option to delete a complete sub-tree
	# FIXME integrate with the save_page() logic, now we have 
	# redundant code here
	my ($self, $page) = @_;
	$page = $self->{page}->name unless defined $page;
	
	my $name = ref($page) ? $page->name : $page;
	my $delete_current = ($name eq $self->{page}->name);
	$self->save_page; # Mind you, overriding save lock could lock us here
	
	return unless $self->prompt_confirm_delete_page($name);
	
	$page = $self->{page} if $delete_current; # make sure we have the right object
	eval { $self->{repository}->delete_page($page) };
	return $self->error_dialog("Could not delete $name\n\n$@") if $@;

	# TODO trigger this with signals
	# problem is that we might have deleted a whole namespace
	$self->{objects}{TreeView}->{_loaded} = 0;
	$self->{objects}{TreeView}->load_index;

	if ($delete_current) {
		eval { $self->{objects}{PageView}->load_page($page) };
		if ($@) {
			$self->exit_error("Could not load page $name\n\n$@");
			# TODO handle more gratiously
			# .. switching read_only would help here
		}
	}

	$self->update_status;
}

=item C<message(STRING)>

Flash a message in the statusbar for a short time.

=cut

sub message {
	my ($self, $str) = @_;
	return if $self->{message_lock};
	my $timeout = $self->{_message_timeout};
	$self->push_status($str, 'message');
	Glib::Source->remove($timeout) if $timeout >= 0;
	$timeout = Glib::Timeout->add(2500, sub {
		$self->pop_status('message');
		$self->{_message_timeout} = -1;
		return 0; # removes timeout
	} );
	$self->{_message_timeout} = $timeout;
}

=item C<update_status()>

Sets the statusbar to display the current page name and some other
information.

=cut

sub update_status {
	my $self = shift;
	my $stat = ' '.$self->{page}->name;
	if ($_ = $self->{page}->status()) { $stat .= ' ('.uc($_).')' }
	$stat .= $self->{objects}{PageView}->get_status;
	$self->push_status($stat, 'page');
}

=item C<push_status(STRING, CONTEXT)>

Put STRING in the status bar.

=cut

sub push_status {
	my ($self, $str, $id) = @_;
	my $statusbar = $self->{statusbar};
	$id = $statusbar->get_context_id($id);
	$statusbar->pop($id);
	$statusbar->push($id, $str);
}

=item pop_status(CONTEXT)

Removes a string from the status bar.

=cut

sub pop_status {
	my ($self, $id) = @_;
	my $statusbar = $self->{statusbar};
	$id = $statusbar->get_context_id($id);
	$statusbar->pop($id);
}

=item C<new_button(STOCK, TEXT)>

Creates a button with a stock image but different text.

=cut

sub new_button {
	my ($self, $stock, $text) = @_;
	my $hbox = Gtk2::HBox->new;
	$hbox->pack_start(
		Gtk2::Image->new_from_stock($stock, 'button'), 0,0,0);
	$hbox->pack_start(
		Gtk2::Label->new_with_mnemonic($text), 1,1,0);
	my $button = Gtk2::Button->new();
	$button->add(Gtk2::Alignment->new(0.5, 0.5, 0, 0));
	$button->child->add($hbox);
	return $button;
}

=item toggle_pane(BOOLEAN)

Show or hide the side pane with the index tree.
If BOOLEAN is "-1" the pane will only be toggled on untill a page is selected.

=cut

sub toggle_pane { # TODO does not update action state :(
	my ($self, $pane_visible) = @_;
	$self->{_pane_visible} = $pane_visible;
	my $widget = $self->{l_vbox};
	my $hpaned = $self->{hpaned};
	if ($pane_visible) {
		$hpaned->set_position($self->{settings}{pane_pos});
		$widget->show_all;
		$self->{notebook}->grab_focus;
	}
	else {
		$self->{settings}{pane_pos} = $hpaned->get_position();
		$widget->hide_all;
		$self->{objects}{PageView}->grab_focus;
	}
	$self->{settings}{pane_vis} = $pane_visible
		unless $pane_visible == -1;
}

=item C<add_to_pane(WIDGET, LABEL)>

Add a widget to the side pane. The label is added to the menu at the top of the side pane.

=cut

sub add_to_pane {
	my ($self, $widget, $label) = @_;
	$self->{notebook}->append_page($widget, $label);
#	$self->{pane_combo}->append_text($label);
	# Cross you fingers, lets hope combobox and notebook count the same way
	my $id = $self->{pane_items}{_id}++;
	$self->{pane_items}{lc($label)} = $id;
}

=item C<show_in_pane(NAME)>

Select which widget you want to see in the side pane.
Opens the side pane if needed.
NAME is the same string as the label used for C<add_to_pane()>.

=cut

sub show_in_pane {
	my ($self, $label) = @_;
	my $id = $self->{pane_items}{lc($label)};
	die "No such widget: $label" unless defined $id;
	$self->{notebook}->set_current_page($id);
	$self->{pane_combo}->set_active($id);
	#$self->{actions}->get_action('TPane')->set_active(1);
}

=item C<toggle_path_bar(TYPE)>

Set or cycle the type of the pathbar.

=cut

sub toggle_path_bar {
	my $self = shift;
	my $type = $self->{settings}{pathbar_type};
	$type = shift || (
		($type eq 'recent'   ) ? 'history'   :
		($type eq 'history'  ) ? 'namespace' :
		($type eq 'namespace') ? 'hidden'    : 'recent' );
	$self->{settings}{pathbar_type} = $type;
	
	my $path_bar = $self->{objects}{PathBar};
	if (grep {$type eq $_} qw/recent history namespace/) {
		$path_bar->widget->show_all;
		$path_bar->set_type($type);
	}
	elsif ($type eq 'hidden') {
		$path_bar->widget->hide_all;
		$path_bar->clear_items;
	}
	else { warn "unknown pathbar_type: $type\n" }

	$self->message("Pathbar type: $type");
}



# TODO the various dialogs could be out-sourced using Autoloader
# also a lot of code could be reduced by haveing a generic
# _prompt_dialog(title => ... text => ... fields => { })

=item C<exit_error(ERROR)>

Like C<error_dialog> but exits afterwards.

=cut

sub exit_error {
	my $self = shift;
	$self->error_dialog(@_);
	exit 1;
}

=item C<error_dialog(ERROR)>

This method is used to display errors.

=cut

sub error_dialog {
	my ($self, $text1, $text2) = @_;
	$text2 ||= $@ || $text1;
	my $dialog = Gtk2::MessageDialog->new(
		$self->{window}, 'modal', 'error', 'ok', $text1 );
		# parent, flags, type, buttons, format, ...
	print STDERR "zim: $text2\n";
	$dialog->run;
	$dialog->destroy;
	return undef;
}

=item C<repository_lock_dialog>

=cut

sub repository_lock_dialog {
	my $self = shift;
	my $dir = shift;
	my $dialog = Gtk2::MessageDialog->new(
		$self->{window}, 'modal', 'warning', 'yes-no', << "EOT");
The directory $dir 
seems to be already in use by
an other instance of zim.

Are you sure you want to continue ?
EOT
	$dialog->set_default_response('no');
	my $ok = ($dialog->run eq 'yes') ? 1 : 0;
	$dialog->destroy;
	return $ok;
}

=item C<jump_page_dialog()>

Prompts the user for a page name. This page is then opened.

=cut

sub jump_page_dialog {
	my $self = shift;
	my $page = $self->{page}->namespace;
	my $button = Gtk2::Button->new_from_stock('gtk-jump-to');
	$page = $self->prompt_page_dialog(
		$page, 'Jump to', '_Jump to Page:', undef, undef, $button);
	return unless defined $page;
	$page = $self->{repository}->resolve_page($page);
	$self->load_page($page);
}

=item C<prompt_page_dialog(PAGE, TITLE, PROMPT, TEXT, ICON)>

Method used to prompt the user for a page name. Returns PAGE.

The I<text> and I<icon> arguments are optional.

=cut

sub prompt_page_dialog {
	my ($self, $page, $title, $prompt, $text, $icon, $button) = @_;

	$page = '' if $page eq ':';
	
	my $dialog = Gtk2::Dialog->new(
		$title, $self->{window},
	       	[qw/modal destroy-with-parent no-separator/],
		'gtk-cancel' => 'cancel',
		( defined($button) ? () : ('gtk-open'   => 'ok') )
	);
	$dialog->set_resizable(0);
	$dialog->set_border_width(5);
	$dialog->set_icon($self->{window}->get_icon);
	$dialog->add_action_widget($button, 'ok') if defined $button;
	$dialog->set_default_response('ok') unless defined $button;
		# the unless ... surpresses a warning ... how to put button as default ?

	if (defined $text) {
		my $text_hbox = Gtk2::HBox->new(0, 5);
		$text_hbox->set_border_width(5);
		$dialog->vbox->add($text_hbox);
		
		if (defined $icon) {
			my $image = Gtk2::Image->new_from_stock($icon, 'dialog');
			$text_hbox->add($image);
		}
		
		my $text_label = Gtk2::Label->new($text);
		$text_hbox->add($text_label);
	}
	
	my $hbox  = Gtk2::HBox->new(0, 5);
	$hbox->set_border_width(5);
	$dialog->vbox->add($hbox);
	
	my $label = Gtk2::Label->new_with_mnemonic($prompt);
	$hbox->add($label);
	
	my $entry = Gtk2::Entry->new();
	$entry->signal_connect(activate => sub { $dialog->response('ok') });
	$entry->set_text($page);
	$hbox->add($entry);
	
	#my $completion = Gtk2::EntryCompletion->new;
	#$completion->set_model($self->TreeView->get_model());
	#$completion->set_text_column(1);
	#$entry->set_completion($completion);
	
	$dialog->show_all;
	if ($dialog->run eq 'ok') { $page = $entry->get_text }
	else                      { $page = undef            }
	$dialog->destroy;

	$page = undef unless $page =~ /\S/;
	return $page;
}

=item C<prompt_rename_page_dialog>

=cut

sub prompt_rename_page_dialog { # TODO base this dialog on prompt_page_dialog()
	my ($self, $page) = @_;
	
	my $title = "Rename page";
	my $dialog = Gtk2::Dialog->new(
		$title, $self->{window},
	       	[qw/modal destroy-with-parent no-separator/],
		'gtk-cancel' => 'cancel',
	);
	$dialog->set_resizable(0);
	$dialog->set_border_width(5);
	$dialog->set_icon($self->{window}->get_icon);
	$dialog->set_default_response('ok');

	my $move_button = $self->new_button('gtk-save', '_Rename');
	$dialog->add_action_widget($move_button, 'ok');
	# $dialog->set_default_response('ok'); FIXME

	my $entry = Gtk2::Entry->new();
	$entry->signal_connect(activate => sub { $dialog->response('ok') });
	$entry->set_text($page);
	$dialog->vbox->add($entry);

	$dialog->show_all;
	$entry->select_region(length($1), -1) if $page =~ /^(.+:)/;

	if ($dialog->run eq 'ok') { $page = $entry->get_text }
	else                      { $page = undef            }
	$dialog->destroy;

	$page = undef unless $page =~ /\S/;
	return $page;
}

=item C<prompt_link_dialog(TEXT, LINK, TITLE)>

Method used to prompt the user for a link. Returns TEXT and TITLE.

=cut

sub prompt_link_dialog {
	my ($self, $text, $link, $title) = @_;
	
	my $dialog = Gtk2::Dialog->new(
		$title, $self->{window},
	       	[qw/modal destroy-with-parent no-separator/],
		'gtk-cancel'  => 'cancel',
	);
	$dialog->set_resizable(0);
	$dialog->set_border_width(5);
	$dialog->set_icon($self->{window}->get_icon);
	
	my $link_button = $self->new_button($LINK_ICON, '_Link');
	$dialog->add_action_widget($link_button, 'ok');
	# $dialog->set_default_response('ok'); FIXME

	my $table = Gtk2::Table->new(2, 2);
	$table->set_border_width(5);
	$table->set_row_spacings(5);
	$table->set_col_spacings(5);
	$dialog->vbox->add($table);
	
	my $text_label = Gtk2::Label->new('Text:');
	my $link_label = Gtk2::Label->new('Links to:');
	my $text_entry = Gtk2::Entry->new();
	$text_entry->signal_connect(activate => sub { $dialog->response('ok') });
	my $link_entry = Gtk2::Entry->new();
	$link_entry->signal_connect(activate => sub { $dialog->response('ok') });
	$table->attach_defaults($text_label, 0,1, 0,1);
	$table->attach_defaults($text_entry, 1,2, 0,1);
	$table->attach_defaults($link_label, 0,1, 1,2);
	$table->attach_defaults($link_entry, 1,2, 1,2);
	
	$link_entry->set_text($link) if defined $link;
	if (defined $text) {
		$text_entry->set_text($text);
		$link_entry->grab_focus;
	}
	else { $text_entry->grab_focus }
	
	$dialog->show_all;
	if ($dialog->run eq 'ok') {
		$text = $text_entry->get_text;
		$link = $link_entry->get_text;
		$dialog->destroy;
		return undef
			unless $text =~ /\S/ or $link =~ /\S/;
		return $text, $link;
	}
	else {
		$dialog->destroy;
		return undef;
	}
}

=item C<prompt_confirm_delete_page(PAGE)>

Ask the user if they are sure about deleting a page.
Returns true if the user wants to proceed.

=cut

sub prompt_confirm_delete_page {
	my ($self, $page) = @_;
	
	my $title = "Delete page";
	my $dialog = Gtk2::Dialog->new(
		$title, $self->{window},
	       	[qw/modal destroy-with-parent no-separator/],
		'gtk-cancel'  => 'cancel',
		'gtk-ok'      => 'ok'
	);
	$dialog->set_resizable(0);
	$dialog->set_border_width(5);
	$dialog->set_icon($self->{window}->get_icon);
	$dialog->set_default_response('ok');

	$dialog->vbox->add( Gtk2::Label->new( << "EOT" ) );
Are you sure you want to
delete $page ?
EOT
	$dialog->show_all;
	my $ok = ($dialog->run eq 'ok') ? 1 : 0;
	$dialog->destroy;
	
	return $ok;
}

=item C<about_dialog()>

This dialog tells you about the version of zim you are using,
the copyright and the current config file. Most importantly 
it has a button to open the manual.

=cut

sub about_dialog { # TODO use Gtk2::AboutDialog for gtk > 2.6
	my $self = shift;
	
	my $dialog = Gtk2::Dialog->new(
		'About Zim', $self->{window},
	       	[qw/modal destroy-with-parent no-separator/],
		'gtk-close' => 'close',
	);
	$dialog->set_resizable(0);
	$dialog->set_border_width(5);
	$dialog->set_icon($self->{window}->get_icon);
	my $button = $self->new_button('gtk-help', '_More');
	$dialog->add_action_widget($button, 'help');
	$dialog->set_default_response('close');
	
	$dialog->vbox->add(
		Gtk2::Image->new_from_file($self->{settings}{icon_file}) );

	my $text = $LONG_VERSION;
	$text =~ s/^(.*)$/\n<b>$1<\/b>/m;
	my $label = Gtk2::Label->new();
	$label->set_markup($text);
	$label->set_justify('center');
	$dialog->vbox->add($label);
	
	$dialog->show_all;
	my $response = $dialog->run;
	$dialog->destroy;

	$self->show_help('zim:about') if $response eq 'help';
}

=item C<show_help(PAGE)>

Show a window showing the documentation. PAGE is optional.

=cut

sub show_help {
	my ($self, $page) = @_;
	my @args = ('--doc', $page ? $page : ());
	$self->exec_new_window(@args);
}

=item  C<exec_new_window(..)>

Executes a new process for $0, this gives a detached window.
Any arguments are passed on to the new process.

=cut

sub exec_new_window {
	my ($self, @args) = @_;
	
	@{$self->{settings}}{'width', 'height'} = $self->{window}->get_size;
	$self->save_config(); # maybe we are read_only too
	fork && return;
	#warn "Trying to execute: $0 @args\n";

	# child process
	no warnings;
	exec $0, @args;
	# backup procedure when $0 did not work
	exec $^X, $0, @args;
	exit 1; # just to be sure
}

=item C<filechooser_dialog(FILE, TITLE)>

Ask the user for a filename. FILE is the suggested filename.

=cut

sub filechooser_dialog {
	my ($self, $file, $dir, $title) = @_;
	
	my $dialog;
	$title ||= 'Select File';
	# if (Gtk2->CHECK_VERSION(2, 4, 0) and $Gtk2::VERSION >= 1.040) {
	$dialog = Gtk2::FileChooserDialog->new(
		$title, $self->{window}, 'open',
		'gtk-cancel' => 'cancel',
		'gtk-ok'     => 'ok'
	);
	# }
	#else { # old & ugly interface
	#	$dialog = Gtk2::FileSelection->new($title);
	#}
	$dialog->set_icon($self->{window}->get_icon) if $self->{window};
	$dialog->set_action('select-folder') if $dir;
	$dialog->set_filename(File::Spec->rel2abs($file)) if defined $file;
	$dialog->signal_connect('response', sub {
		$file = $_[1] eq 'ok' ? $dialog->get_filename : undef;
		$dialog->destroy;
	} );
	$dialog->run;

	return $file;
}

=item C<prompt_repository_dialog>

=cut

sub prompt_repository_dialog { # TODO improve visual layout
	my $self = shift;

	my $dialog = Gtk2::Dialog->new(
		"Open repository - Zim", $self->{window},
	       	[qw/destroy-with-parent no-separator/],
		'gtk-cancel' => 'cancel',
		'gtk-open'   => 'ok',
	);
	$dialog->set_default_size(100,250);
	$dialog->set_border_width(5);
	$dialog->set_icon(
		Gtk2::Gdk::Pixbuf->new_from_file($self->{settings}{icon_file}) );
	$dialog->set_default_response('ok');
	
	$dialog->vbox->set_spacing(5);
	$dialog->vbox->pack_start(
		Gtk2::Label->new('Please choose a repository'), 0,1,0 );
	
	my $list = Gtk2::SimpleList->new('Repository' => 'text');
	$list->set_headers_visible(0);
	$list->get_selection->set_mode('browse');
	$list->signal_connect(row_activated => sub { $dialog->response('ok') });
	$dialog->vbox->pack_start($list, 1,1,0);
	
	# Read list - FIXME also look in other xdg_config dirs
	my %source;
	my $file = File::Spec->catfile(
		File::BaseDir->xdg_config_home, 'zim', 'repositories.list' );
	if (-f $file) {
		open IN, $file or die "Could not read file: $file\n";
		while (<IN>) {
			chomp;
			my ($name, $source) = split '=', $_, 2;
			next unless $name =~ /\S/;
			$source{$name} = $source;
			push @{$list->{data}}, $name;
		}
		close IN;
	}
	
	my $hbox = Gtk2::HBox->new(1,10);
	my $add_button = Gtk2::Button->new_from_stock('gtk-add');
	my $rem_button = Gtk2::Button->new_from_stock('gtk-remove');
	$hbox->add($add_button);
	$hbox->add($rem_button);
	$dialog->vbox->pack_start($hbox, 0,1,0);
	
	my $modified = 0;
	$add_button->signal_connect(clicked => sub {
			my ($name, $source) = $self->prompt_new_repository_dialog($dialog);
			return unless defined $name;
			$source{$name} = $source;
			push @{$list->{data}}, $name;
			$modified = 1;
		} );
	$rem_button->signal_connect(clicked => sub {
			my ($i) = $list->get_selected_indices;
			splice @{$list->{data}}, $i, 1;
			$modified = 1;
		} );
	
	$dialog->show_all;
	my $repository;
	RUN_REP_DIALOG:
	if ($dialog->run eq 'ok') {
		my ($i) = $list->get_selected_indices;
		unless(defined $i) {
			$self->error_dialog("Please select a repository first.\n");
			goto RUN_REP_DIALOG;
		}
		
		my ($name) = @{$list->{data}[$i]};
		$repository = $source{$name};
		#warn "You selected \"$name\" ($i) => $repository\n";
	}
	else { $modified = 0 } # Don't save at cancel
	$dialog->destroy;

	# Save list
	if ($modified) {
		open OUT, ">$file" or die "Could not write file: $file\n";
		for (@{$list->{data}}) {
			my ($name) = @$_;
			print OUT "$name=$source{$name}\n";
		}
		close OUT or die "Could not write file: $file\n";
	}
	
	return $repository;
}

=item C<prompt_new_repository_dialog(WINDOW)>

=cut

sub prompt_new_repository_dialog { # TODO add "browse" button for dir
	my $self = shift;
	my $window = shift || $self->{window};
	
	my $dialog = Gtk2::Dialog->new(
		'New repository - Zim', $window,
	       	[qw/modal destroy-with-parent no-separator/],
		'gtk-cancel' => 'cancel',
		'gtk-ok'     => 'ok',
	);
	$dialog->set_resizable(0);
	$dialog->set_border_width(5);
	$dialog->set_icon(
		Gtk2::Gdk::Pixbuf->new_from_file($self->{settings}{icon_file}) );
	$dialog->set_default_response('ok');

	$dialog->vbox->set_spacing(5);
	$dialog->vbox->add( Gtk2::Label->new(
		"Please give a new directory\nto store your pages" ) );
	
	my $table = Gtk2::Table->new(2, 2);
	$table->set_border_width(5);
	$table->set_row_spacings(5);
	$table->set_col_spacings(5);
	$dialog->vbox->add($table);
	
	my $dir_label = Gtk2::Label->new('Directory:');
	my $name_label = Gtk2::Label->new('Name:');
	my $dir_entry = Gtk2::Entry->new();
	$dir_entry->signal_connect(activate => sub { $dialog->response('ok') });
	my $name_entry = Gtk2::Entry->new();
	$name_entry->signal_connect(activate => sub { $dialog->response('ok') });
	$table->attach_defaults($dir_label, 0,1, 0,1);
	$table->attach_defaults($dir_entry, 1,2, 0,1);
	$table->attach_defaults($name_label, 0,1, 1,2);
	$table->attach_defaults($name_entry, 1,2, 1,2);

	$dialog->show_all;
	my ($name, $dir);
	if ($dialog->run eq 'ok') {
		$name = $name_entry->get_text;
		$dir = $dir_entry->get_text;
	}
	$dialog->destroy;

	if ($dir !~ /\S/) { $name = undef }
	elsif ($name !~ /\S/) { $name = $dir }
	
	return ($name, $dir);
}

=item C<prompt_browser_dialog>

=cut

sub prompt_browser_dialog {
	my ($self, $type) = @_;
	
	my $title = ($type eq 'file_browser') ? "Choose file browser" : "Choose browser";
	my $dialog = Gtk2::Dialog->new(
		$title, $self->{window},
	       	[qw/modal destroy-with-parent no-separator/],
		'gtk-cancel' => 'cancel',
		'gtk-ok'     => 'ok',
	);
	$dialog->set_resizable(0);
	$dialog->set_border_width(5);
	$dialog->set_icon($self->{window}->get_icon);
	$dialog->set_default_response('ok');

	my $label = Gtk2::Label->new(
		($type eq 'file_browser')
			? "Please enter a file browser"
			: "Please enter a browser\nto open external links." );
	$dialog->vbox->add($label);
	
	my $entry = Gtk2::Entry->new();
	$entry->signal_connect(activate => sub { $dialog->response('ok') });
	$entry->set_text(($type eq 'file_browser') ? "rox '\%s'" : "firefox '\%s'");
	$dialog->vbox->add($entry);

	$dialog->show_all;
	my $browser = 
		($dialog->run eq 'ok') ? $entry->get_text : undef ;
	$dialog->destroy;
	$browser = undef unless $browser =~ /\S/;

	return $browser;
}

# "Search" dialog
#
# This dialog asks for a search query and then generates
# a page with search results.

#sub search_dialog {
#	my $dialog = Gtk2::Dialog->new(
#		# title, parent
#		'Search', $window,
#		# flags
#	       	[qw/modal destroy-with-parent no-separator/],
#		# buttons
#		'gtk-cancel' => 'cancel',
#		'gtk-find'   => 'ok',
#	);
#	$dialog->set_resizable(0);
#	$dialog->set_border_width(5);
#	$dialog->set_icon($window->get_icon);
#	#$dialog->set_default_response('ok');
#
#	my $table = Gtk2::Table->new(2, 2);
#	$table->set_border_width(5);
#	$table->set_row_spacings(5);
#	$table->set_col_spacings(5);
#	$dialog->vbox->add($table);
#	
#	my $label1 = Gtk2::Label->new('Find what:');
#	my $query_entry = Gtk2::Entry->new();
#	$query_entry->signal_connect(activate => sub { $dialog->response('ok') });
#	$table->attach_defaults($label1, 0,1, 0,1);
#	$table->attach_defaults($query_entry, 1,2, 0,1);
#	
#	$dialog->show_all;
#	if ($dialog->run eq 'ok') {
#		print "TODO: search algorithm that doesn't depend on grep(1)\n";
#		my $query = $query_entry->get_text;
#		return unless $query =~ /\S/;
#		my $page = Zim::Page->new($zim, 'Search Results');
#		$page->push_blocks(
#			['head1', 'Search Results'], "\n\n",
#			['B', '-- Search algorithm not yet stable --'],
#			"\n\nSearching for \"$query\"\n\n",
#		);
#		$query =~ s/\s/\\W+/g;
#		open GREP, "grep -ERi \"\\b$query\\b\" \"$ROOT\" |" or return;
#		while (<GREP>) {
#			/^(.+?):(.+)$/ or next;
#			my ($file, $text) = ($1, $2);
#			$file =~ s/^\Q$ROOT\E//;
#			next if $file =~ /^\./;
#			$page->push_blocks(
#				['link', $zim->pagename($file)],
#				"\n", ['pre', $text], "\n\n"
#			);
#		}
#		close GREP;
#		load_page($page);
#	}
#	$dialog->destroy;
#}

1;

__END__

=back

=head1 BUGS

Please mail the author if you find any bugs.

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

L<zim>(1),
L<Zim::Repository>

=cut

