package Zim::Repository;

use strict;
use POSIX qw(strftime);
use Encode;
use IO::File;
use File::Spec;
use File::Copy;
use File::MimeInfo;
use Zim::Page::Text;

our $VERSION = '0.10';

my $case_tolerant_fs = File::Spec->case_tolerant();

=head1 NAME

Zim::Repository - A wiki repository

=head1 SYNOPSIS

	use Zim::Repository;
	
	my $repo = Zim::Repository->new($root_dir);
	my $page = $repo->open_page('Home');

=head1 DESCRIPTION

This class defines the public interface to the document repository
as used by L<zim>(1). By default it represents filesystem based repositories,
but it can be overloaded for more abstract models. It uses L<Zim::Page>
objects as data containers.

=head1 METHODS

=head2 Public Methods

The following methods can be used by the GUI.

=over 4

=item new(NAMESPACE, DIR)

Simple constructor. DIR is the root directory of the repository.
NAMESPACE is the namespace that maps to that directory. For the 
toplevel repository leave NAMESPACE empty.

=cut

sub new {
	my ($class, $namespace, $dir, $format) = @_;
	$dir = File::Spec->file_name_is_absolute($dir)
		? File::Spec->canonpath($dir)
		: File::Spec->rel2abs($dir)   ;
	$namespace =~ s/:?$/:/; # '' => ':'
	my $self = bless {
		namespace => $namespace,
		root      => $dir,
		format    => $format || 'wiki',
	}, $class;
	
	$self->{ext} = ($self->{format} eq 'wiki') ? 'txt' : 'html'; # FIXME HACK FIXME
		# this belongs in a Formats.pm
	
	# load plugins (nested repositories)
	# TODO rename to "child" instead of "plugin"
	my $conf = File::Spec->catfile($dir, '.zim.config');
	if (-f $conf and -r _) {
		$self->{conffile} = $conf;
		open CONFIG, "<$conf" or die "Could not open: $conf\n";
		while (<CONFIG>) {
			/^(:\S+?)=(\S+)/ or next;
			my ($subspace, $mod) = ($1, $2);
			$subspace =~ s/:+$//;
			eval "use $mod;";
			die if $@;
			my $dir = $self->filename($subspace, 'DIR');
			$self->{plugins}{$subspace} = $mod->new($subspace, $dir);
			#print STDERR "plug: $mod for $subspace\n";
		}
		close CONFIG;
	}

	return $self;
}

=item list_pages(NAMESPACE)

Lists pages in NAMESPACE. Sub-namespaces have a trailing ':'
in the listing.

If a page is not present in this list that does not mean that it
doesn't exists. This list is used for hierarchical views of the
page structure; not all repositories need to support a hierarchical
ordening. Also pages may be autoloaded when asked for.

=cut

sub list_pages {
	my ($self, $namespace) = @_;
	die "Can't list empty namespace\n" unless length $namespace;
	
	$namespace =~ s/:?$/:/;
	if (my $plug = $self->_belongs_to_plugin($namespace)) {
		return $self->{plugins}{$plug}->list_pages($namespace);
	}

	my $dir = $self->filename($namespace, 'DIR');
	#print "list pages $namespace => $dir\n";
	return () unless -d $dir;
	opendir DIR, $dir or die "Could not list dir $dir\n";
	my @items = grep {defined $_} map {
		$_ = Encode::decode_utf8($_, 1); # use utf8 encoding
		my $item = File::Spec->catfile($dir, $_);
		if (-d $item) { s/(_files)?$/:/ }
		else {
			if (mimetype($item) =~ /^text/) { s/\.$$self{ext}$// }
			else { $_ = undef }
		}
		$_;
	} grep {! /^\./} readdir DIR;
	closedir DIR;
	return sort {lc($a) cmp lc($b)} @items;
}

sub _belongs_to_plugin {
	# FIXME can be done more efficiently with a length-ordered regex
        my ($self, $page) = @_;
	$page =~ s/^:*$self->{namespace}:*/:/;
        my $plug = '';
	for (grep {$page =~ /^$_:/} keys %{$self->{plugins}}) {
		$plug = $_ if length($_) > length($plug);
	}
	#print STDERR "plugin for $page => $plug\n";
        return $plug;
}


=item page_exists(PAGE_NAME)

Returns TRUE when there is an existing page under this name.

=cut

sub page_exists { # TODO should return extra data just like resolve_...
	my ($self, $name) = @_;
	
	if (my $plug = $self->_belongs_to_plugin($name)) {
		return $self->{plugins}{$plug}->page_exists($name);
	}
	
	return -f $self->filename($name);
}

=item open_page(PAGE_NAME)

Returns an object of the type L<Zim::Page::Text>.

If the page you requested didn't exist yet you get an object with it's status
set to "new".

Do not forget to call C<close_page()> when you are done with the page object.

=cut

sub open_page {
	my ($self, $name, $source) = @_;
	#print STDERR "open_page: @_\n";

	if (my $plug = $self->_belongs_to_plugin($name)) {
		return $self->{plugins}{$plug}->open_page($name);
	}
	
	my $page = Zim::Page::Text->new($self, $name);
	$source = [ $self->filename($name) ] unless defined $source;
	$page->set_source($source);
	$page->set_format($self->{format});
	$page->properties->{base} = 'file://'.$source->[1];
	
	unless (-f $$source[0]) {
		$page->{parse_tree} = $self->_template($page);
		$page->status('new');
	}

	return $page;
}

=item C<close_page(PAGE)>

When you are done using a page object let the repository know this by
calling this method. In some cases this might be essential for saving
the data in the page object.

=cut

sub close_page { }

=item move_page(OLD_NAME, NEW_NAME)

Moves a page from OLD_NAME to NEW_NAME.

You can use an object instead of OLD_NAME, in this case the object will also
be updated to reflect the new location.

=cut

sub move_page {
	my ($self, $old, $new) = @_;
	
	# TODO respect plugins here !
	# probably needs export interface

	# Move file
	my ($old_file, $old_dir) = $self->filename($old);
	my ($new_file, $new_dir) = $self->filename($new);
	_mkdir($new_dir);
	move($old_file, $new_file)
		or die "Could not move file $old_file to $new_file\n"
		if -f $old_file;

	# Update Page object
	if (ref $old) {
		$old->name($new);
		$old->set_source([$new_file, $new_dir]);
	}

	# Move tree below file
	my $old_tree = $self->filename($old, 'DIR');
	my $new_tree = $self->filename($new, 'DIR');
	move($old_tree, $new_tree)
		or die "Could not move dir $old_tree to $new_tree\n"
		if -d $old_tree;
	_rmdir($old_dir);
}

=item delete_page(PAGE)

Deletes a page.
PAGE can be either a page name or a page object.

=cut

sub delete_page {
	my ($self, $page) = @_;

	# TODO: respect plugins here !

	my ($file, $dir) = $self->filename($page);
	if (-f $file) {
		unlink $file or die "Could not remove file $file\n";
	}
	else {
		my $name = ref($page) ? $page->name : $page ;
		$dir = $self->filename($name, 'DIR'); # step one level down
	}
	_rmdir($dir);
	
	if (ref $page) {
		$page->{parse_tree} = undef;
		$page->status('deleted');
	}
}

sub _template {
	# FIXME make template configurable
	my ($self, $page) = @_;
	$page->name =~ /([^:]+):*$/;
	my $title = ucfirst($1);
	my $date = Encode::decode_utf8(
		strftime('%A %d/%m/%Y %H:%M', localtime), 1 );
	return ['Document', {},
		['head1', {}, $title],
		['Para',  {empty_lines => 1}, "Created $date\n"]  ];
}

=item C<resolve_link(PAGE, LINK)>

Returns a page name for a link. The link might be relative to the page.

PAGE can either be a page object or a page name.

This method calls C<resolve_page()> automaticly.

In list context the page name can be followed with some data
the may be given to the repository's C<open_page()>. This is
used for repository specific optimizations.

=cut

sub resolve_link {
	# TODO more intelligent upward lookup in path
	my ($self, $page, $link) = @_;
	my ($name, $namespace) = (ref $page)
		? ($page->name, $page->namespace      )
		: ($page, Zim::Page->namespace($page) ) ;

	if (my $plug = $self->_belongs_to_plugin($name)) {
		return $self->{plugins}{$plug}->resolve_link($page, $link);
	}
	
	#print STDERR "resolved link $link to ";
	$link =~ s/[^\w\.\:\-]/_/g;
	
	$link = ($link =~ s/^\.//) ? $name.':'.$link  : # sub namespace
		($link !~ /:/)     ? $namespace.$link : # relative link
		$link ;
		
	#print STDERR "to $link\n";
	return $self->resolve_page($link); # uses wantarray
}


=item C<resolve_page(NAME)>

This method finds a case-sensitive name corresponding to the
user supplied page name NAME. The page does not need to exist.

More generally this method is used to find a canonical page name
corresponding to a potential non-canonical name that is entered by
the user. This method may do all kinds of cleanups.

In list context the page name can be followed with some data
the may be given to the repository's C<open_page()>. This is
used for repository specific optimizations.

=cut

sub resolve_page {
	my ($self, $name) = @_;
	$name =~ s/^:?/:/;
	# TODO first resolve the part that maps to a plugin
	if (my $plug = $self->_belongs_to_plugin($name)) {
		return $self->{plugins}{$plug}->resolve_page($name);
	}
	#print STDERR "resolve_page $name => ";
	my ($file, $dir) = $self->filename($name, 0, 1); # case tolerant lookup
	$name = $self->pagename($file);
	#print STDERR "$name\n";
	return wantarray ? ($name, [$file, $dir]) : $name ;
}

=item C<export(PAGE, %OPTS)>

TODO stable api for this

=cut

# TODO when api is stable add export() to page which redirects here
# TODO use verbose option
# TODO respect plugins !
# TODO respect page interfaces

sub export {
	my ($self, $page, %opts) = @_;
	die "Need a page to export\n" unless length $page;
	for (qw/format template root/) {
		die "for exporting you need at least to provide a format, a template and an output root dir.\n" unless length $opts{$_};
	}

	$page = $page->name if ref $page;
	my $namespace = ($page =~ /:$/) ? $page : '';
	#print STDERR "export root: $opts{root}\n";
	my $exporter = ref($opts{root}) ? $opts{root} :
		Zim::Repository->new('', $opts{root}, $opts{format});
	#print STDERR "export template: $opts{template}\n";
	$exporter->{_template} = $opts{template}; # FIXME better interface for this - do not confuse with the template for new pages
	
	my $r = defined($opts{recurs}) ? $opts{recurs} : 1 ;
	if (length $namespace) {
		$self->_export_namespace($exporter, $page, $r);
	}
	else {
		$self->_export_page($exporter, $page);
	}

}


sub _export_namespace {
	my ($self, $exporter, $namespace, $recurs) = @_;
	
	my $index = 1;
	my @pages;
	for my $page ($self->list_pages($namespace)) {
		$index = 0 if $page =~ /index/;
		$self->_export_namespace($exporter, $namespace.$page, $recurs)
			if $recurs and $page =~ /:$/; # recurs
		push @pages, $page
			if $self->_export_page($exporter, $namespace.$page);
	}

	if ($index && @pages) { # FIXME option for this
		print STDERR "Writing index for $namespace\n";
		#print STDERR join(', ', @pages), "\n";
		$index = $exporter->open_page($namespace.'index');
		$index->set_parse_tree( ['Document', {},
			['head1', {}, "Document index for $namespace"],
			['Para', {},
				map {("* ", ['link', {to => $_}, $_], "\n")} @pages
			]
		] );
		$index->close;
	}
	
}

sub _export_page {
	my ($self, $exporter, $page) = @_;
	
	my $orig = $self->open_page($page);
	return 0 if $orig->status eq 'new';
	print STDERR "Exporting: $page\n";
	
	my $copy = $exporter->open_page($page);
	$copy->set_parse_tree( $orig->get_parse_tree );
	$copy->close;
	
	return 1;
}

=back

=head2 Private Methods

These methods are specific to filesystem based repositories
and should therefor never be called directly from the GUI.

=over 4

=item filename(PAGE, DIR)

Gives the filename corresponding to a page name.

DIR is a boolean that can be set to tell that PAGE is expected to
be a directory, which is useful when looking up a namespace.
This behaviour is also triggered if the page name ends with ':'.

In list context the filename and directory for PAGE are returned.

=cut

sub filename {
	my ($self, $page, $is_dir, $case_tolerant) = @_;

	if (ref $page) { # page object
		return wantarray ? (@{$page->{source}}) : $page->{source}[0]
			if defined $page->{source};
		$page = $page->name;
	}
	
	$page =~ s/^:*$self->{namespace}:*//i;
	$is_dir++ if $page =~ s/:+$//;
	
	my @dirs = split /:+/, $page;
	my $basename = pop @dirs unless $is_dir;

	# Find the right dir
	my $dir = File::Spec->catdir($self->{root}, @dirs);
	unless (!$case_tolerant_fs and  -d $dir) {
		$dir = $self->{root};
		for (0 .. $#dirs) {
			my $new_dir = $self->_grep_dir(
				$dir, $dirs[$_], 1, $case_tolerant );
			if (defined $new_dir) { $dir = $new_dir }
			else {
				$dir = File::Spec->catdir($dir, @dirs[$_ .. $#dirs]);
				last;
			}
		}
	}
	
	if ($is_dir) {
		return $dir unless wantarray;
		@dirs = File::Spec->splitdir($dir);
		pop @dirs;
		my $parent = File::Spec->catdir(@dirs);
		return ($dir, $parent);
	}

	# Check the filename
	my $file = $self->_grep_dir($dir, $basename, 0, $case_tolerant);
	$file = File::Spec->catfile($dir, $basename .'.'.$$self{ext})
		unless defined $file;

	return wantarray ? ($file, $dir) : $file;
}

sub _grep_dir { # find a file or subdir
	my ($self, $dir, $basename, $is_dir, $case_tolerant) = @_;
	return undef unless -d $dir;
	my $extended = $basename . ($is_dir ? '_files' : '.'.$$self{ext});
	my $wrong_type = 0;
	
	unless ($case_tolerant and $case_tolerant_fs) {
		if ($is_dir) {
			my $path = File::Spec->catdir($dir, $basename);
			return $path if -d $path;
			$wrong_type = 1 if -e _;
		}
		else {
			my $path = File::Spec->catfile($dir, $basename);
			return $path if -f $path;
			$wrong_type = 1 if -e _;
		}
	
		my $path = $is_dir
			? File::Spec->catdir($dir, $extended)
			: File::Spec->catfile($dir, $extended) ;
		return $path if $wrong_type or -e $path;
		return undef unless $case_tolerant;
	}
	
	my @check = (lc($basename), lc($extended));
	my $suggestion;
	my $other_type = lc $basename . ($is_dir ? '.'.$$self{ext} : '_files');
	opendir DIR, $dir or die "Could not list dir $dir\n";
	while (my $item = readdir DIR) { 
		# FIXME for case_tolerant_fs it would be nicer to check
		# case sensitive version first in the listing
		$item = Encode::decode_utf8($item, 1);
		if (grep {$_ eq lc($item)} @check) {
			if ($is_dir) {
				$item = File::Spec->catdir($dir, $item);
				unless (-d $item) {
					$suggestion = $item;
					next;
				}
			}
			else {
				$item = File::Spec->catfile($dir, $item);
				unless (-f $item) {
					$suggestion = $item;
					next;
				}
			}
			closedir DIR;
			return $item;
		}
		elsif (lc($item) eq $other_type) {
			$suggestion = $is_dir
				? File::Spec->catdir($dir, $item)
		       		: File::Spec->catfile($dir, $item) ;
		}
	}
	closedir DIR;

	if ($suggestion) {
		if ($is_dir) { $suggestion =~ s/\.$$self{ext}$//          }
		else         { $suggestion =~ s/(_files)?$/\.$$self{ext}/ }
		return $suggestion;
	}
	
	return undef;
}

=item pagename(FILE)

Returns the page name corresponding to FILE. FILE does not actually
need to exist and can be a directory as well as a file.

=cut

sub pagename {
	my ($self, $file) = @_;
	$file = File::Spec->abs2rel($file, $self->{root})
		if File::Spec->file_name_is_absolute($file);
	my @parts = map  {s/_files$//; $_} File::Spec->splitdir($file);
	$parts[-1] =~ s/\.$$self{ext}$//;
	return $self->{namespace} . join ':', @parts;
}

=item C<get_source(SOURCE, MODE)>

Returns an IO::File object.

=cut

sub get_source {
	my ($self, $source, $mode) = @_;
	my $io;
	if ($mode eq 'r') { $io = IO::File->new($$source[0], 'r') }
	elsif ($mode eq 'w') {
		#print "get IO object for $$source[0]\n";
		_mkdir($$source[1]) unless -e $$source[1];
		$io = IO::File->new($$source[0], 'w');
	}
	else { die "unknown mode: $mode" }
	
	binmode $io, ':utf8' unless $] < 5.008;
		# IO::File OO version broken in perl 5.8.7
	return $io;
}

# Methods below for filesystem interaction

sub _mkdir {
	my ($dir, $file) = @_; # if $file is true $dir is a file
	my ($vol, $dirs) = File::Spec->splitpath($dir, $file ? 0 : 1);
	my @dirs = File::Spec->splitdir($dirs);
	my $path = File::Spec->catpath($vol, shift @dirs);
	mkdir $path or die "Could not create dir $path\n"
		if length $path and ! -d $path;
	while (@dirs) {
		$path = File::Spec->catdir($path, shift @dirs);
		mkdir $path or die "Could not create dir $path\n"
			unless -d $path;
	}
}

sub _rmdir {
	my $dir = shift;
	#print STDERR "rmdir $dir\n";
	rmdir $dir or return; # fails when non-empty
	my ($vol, $dirs) = File::Spec->splitpath($dir, 1);
	my @dirs = File::Spec->splitdir($dirs);
	while (@dirs) {
		pop @dirs;
		$dir = File::Spec->catdir($vol, @dirs);
		#print STDERR "rmdir $dir\n";
		rmdir $dir or last; # fails when non-empty
	}
}

1;

__END__

=back

=head1 BUGS

Please mail the author if you find any bugs.

=head1 AUTHOR

Jaap Karssenberg || Pardus [Larus] E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

L<Zim>, L<Zim::Page>

=cut
