package Zim::Page::Text;

use strict;
use Zim::Page;

our $VERSION = '0.10';

our @ISA = qw/Zim::Page/;

=head1 NAME

Zim::Page::Text - Page object for Zim

=head1 DESCRIPTION

This class defines a page object that represents a text buffer.
This can either be a plain text or a formatted text (rich text).

The interfaces supported by this class are "source" and "formatted".
An individual object can have one of these or both.

The source interface uses C<IO::*> objects and requires the
repository to support an interface for opening and closing these objects.

The formatted interface uses parse trees to represent the logical
structure of the text. If a formatted text also has a source, a 
formatter object (C<Zim::Formats::*>) is used to parse the source.

=head1 METHODS

=over 4

=item C<interfaces()>

Returns a list of interfaces supported by this page.

=cut

sub interfaces {
	my $self = shift;
	return ( ($self->{format} ? 'formatted' : ()),
	         ($self->{source} ? 'source'    : ())  );
}

=back

=head2 Source Interface

=over 4

=item C<set_source(SOURCE)>

SOURCE is a scalar used to fetch a C<IO::*> object from the
repository later. This can for example be a filename but also
a complex structure containing all the parameters needed to
open an IO object.

This method should only be used by the repository. The GUI is not
allowed to know anything about the real source of pages, it can only
use the IO objects resulting from C<get_source(MODE)>.

=cut

sub set_source {
	$_[0]->{source} = $_[1];
	$_[0]->{parse_tree} = undef;
}

=item C<get_source(MODE)>

Returns an IO object or undef if there is none.
In general pages that have status "new" will not yet have a source.

MODE is optional and can either be 'r' or 'w' depending on whether you
would like the source to be opened for reading or for writing.

Do not forget to close the IO object when you are done with it.

( When using source the repository should also support the
C<get_source(SOURCE, MODE)> method. )

=cut

sub get_source {
	my ($self, $mode) = @_;
	return unless defined $self->{source};
	$self->{status} = '' if $mode eq 'w'; # remove "new" or "deleted"
	return $self->{repository}->get_source($self->{source}, $mode);
}

=back

=head2 Formatted Interface

=over 4

=item C<set_format(FORMAT)>

Sets a source format for this page. This can either be an object of the class
L<Zim::Formats> (or similar), or a name in which case this will be looked up
in the C<Zim::Formats::*> namespace.

Formats are only used for pages that also have a source object.

=cut

sub set_format {
	my ($self, $format) = @_;
	$self->{format} = ref($format) ? $format : _load_format($format);
}

our %_formats;

sub _load_format {
	my $name = shift;
	$_formats{$name} = _load_class($name) unless defined $_formats{$name};
	return $_formats{$name};
}

sub _load_class { # TODO this actually belongs in a Zim/Formats.pm
	my $name = shift;
	my $wanted = lc($name).'.pm';
	my $class;
	for (@INC) {
		# FIXME check object refs in @INC
		my $dir = File::Spec->catdir($_, qw/Zim Formats/);
		next unless -d $dir and -r _;
		opendir DIR, $dir or next;
		($class) = grep {lc($_) eq $wanted} readdir DIR;
		closedir DIR;
		last if defined $class;
	}
	$class =~ s/.pm$//;
	die "Could not find a class for format: $name\n" unless length $class;
	$class = "Zim::Formats::$class";
	eval "use $class;";
	die if $@;
	return $class;
}

=item C<get_parse_tree()>

Get the parse tree for this page.

When using source this method will return the tree resulting from running
the given source through the given formatter.

=cut

sub get_parse_tree {
	my $self = shift;
	return $self->{parse_tree} if defined $self->{parse_tree};
	return unless defined $self->{source} and defined $self->{format};
	
	my $io = $self->get_source('r');
	if ($io) {
		my $tree = $self->{format}->load_tree($io, $self);
		$io->close;
		$tree->[1] = { %{$self->{properties}}, %{$tree->[1]} };
		return $tree;
	}
	else { return ['Document', $self->{properties}] }
}

=item C<set_parse_tree(TREE)>

Set the parse tree for this page.

When using source this method will use the formatter to save the parse tree
to the IO object.

=cut

sub set_parse_tree {
	my ($self, $tree) = @_;
	$self->{status} = ''; # remove "new" or "deleted"
	if (defined $self->{source}) {
		my $io = $self->get_source('w')
			|| die "Could not save parse tree, did not get an IO object.\n";
		$self->{format}->save_tree($io, $tree, $self);
		$io->close;
	}
	else {
		$self->{parse_tree} = $tree;
	}
}

1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

L<Zim>,
L<Zim::Page>,
L<Zim::Formats>,
L<Zim::Repository>

=cut
