package Zim::Page;

use strict;

our $VERSION = '0.10';

=head1 NAME

Zim::Page - Page object for Zim

=head1 DESCRIPTION

This class defines a page object. This is a data container used by
L<Zim::Repository> to represent for example a file.

Since data can be represented in a variety of ways there are multiple
interfaces to access it. These interfaces can be thought as the parent
classes a page inherits from (given multiple-iheritance), but the
interfaces supported by an individual object can change more dynamicly.

This is a general base class that does not support any interfaces
itself and cannot be used on it's own.

=head1 METHODS

=over 4

=item C<new(PARENT, NAME)>

Simple constructor. PARENT should be a repository object of class Zim.
NAME is the page name for this object.

=cut

sub new {
	my ($class, $parent, $name) = @_;
	$name =~ s/:+$//;
	die "Can't create $class object without a page name!\n"
		unless length $name;
	$name =~ s/^:?/:/;
	my $self = bless {
		name => $name,
		repository => $parent,
		status => '',
		properties => {},
	}, $class ;
	return $self;
}

=item C<interfaces()>

Returns a list of interfaces supported by this page.

=cut

sub interfaces { } # empty list

=item C<properties()>

Returns a hash with properties. See L</PROPERTIES>.

=cut

sub properties { $_[0]->{properties} }

=item C<name()>

Get or set the full name of the page.

=cut

sub name {
	$_[0]->{name} = $_[1] if @_ > 1;
	return $_[0]->{name};
}

=item C<basename()>

Returns the last part of the page name.

=cut

sub basename {
	my $name = $_[0]->{name};
	$name =~ s/^(.*:+)//;
	return $name;
}

=item C<namespace()>

Returns the namespace to which this page belongs.

=cut

sub namespace {
	my $name = $_[1] || $_[0]->{name};
	#print STDERR "namespace for $name ";
	$name =~ /^(.*:)/;
	$name = $1 || ':';
	$name =~ s/::+/:/g;
	#print STDERR "is $name\n";
	return $name;
}

=item C<status(STRING)>

Set or get a status string for this page.
Typical status strings are 'new' and 'deleted'.

=cut

sub status {
	$_[0]->{status} = $_[1] if @_ > 1;
	return $_[0]->{status};
}

=item C<close()>

=item C<delete()>

=item C<move(NEW_NAME)>

=item C<resolve_link(LINK)>

The methods C<close()>, C<delete()>, C<move()> and C<resolve_link()>
are aliases for the methods C<close_page()>, C<delete_page()>, C<move_page()>
and C<resolve_link()> respectively in L<Zim::Repository>.

=cut

sub close { $_[0]->{repository}->close_page(@_) }

sub delete { $_[0]->{repository}->delete_page(@_) }

sub move { $_[0]->{repository}->move_page(@_) }

sub resolve_link { $_[0]->{repository}->resolve_link(@_) }

1;

__END__

=back

=head1 PROPERTIES

The page object contains a hash with properties. These can be any kind of data
from the backend that needs to be shared with the interface. Typically it are
config options that can be specified per-page.

For the "formatted" page interface the properties hash is used for the Document
meta attributes in the parse-tree.

Common properties are:

=over 4

=item base (url)

Base directory for files that belong to this page in some way or another.
This is for example used by the interface to resolve the location of image
files that are included in the page.

This value can be undefined when the repository does not allow direct access
to the source files.

TODO: At the moment this is the directory which contains the pae file, this
is open for change in future versions.

Currently only the file:// url is really supported.

=item read_only (boolean)

Tells the interface that this page should not be edited.

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

L<Zim>,
L<Zim::Repository>

=cut
