package Zim::Components::PageView;

use strict;
use vars '$AUTOLOAD';
use Gtk2;
use Gtk2::Pango;               # pango constants
use Gtk2::Ex::HyperTextView;   # custom widget
use Gtk2::Ex::HyperTextBuffer; #    "     "
use POSIX qw(strftime);

our $VERSION = '0.10';

# TODO move more logic into Gtk2::Ex::HyperTextBuffer !

=head1 NAME

Zim::Components::PageView - Page TextView widgets

=head1 DESCRIPTION

This module contains the widgets to display an editable
text buffer containing the current page. It includes a search entry
at the bottom of the TextView, formatting codes for the TextBuffer and 
an undo stack.

=head1 METHODS

Undefined methods are AUTOLOADED to the Gtk2::Ex::HyperTextView object.

=over 4

=cut

our $LINK_ICON = Gtk2->CHECK_VERSION(2, 6, 0) ? 'gtk-connect' : 'gtk-convert';
	# gtk-connect stock item was introduced in 2.6.0

my @ui_actions = (
	# name,          stock id,         label,            accelerator,       tooltip
	[ 'Head1',       undef,            'Head _1',        '<ctrl>1',         'Heading 1',     \&on_tag         ],
	[ 'Head2',       undef,            'Head _2',        '<ctrl>2',         'Heading 2',     \&on_tag         ],
	[ 'Head3',       undef,            'Head _3',        '<ctrl>3',         'Heading 3',     \&on_tag         ],
	[ 'Head4',       undef,            'Head _4',        '<ctrl>4',         'Heading 4',     \&on_tag         ],
	[ 'Head5',       undef,            'Head _5',        '<ctrl>5',         'Heading 5',     \&on_tag         ],
	[ 'Normal',      undef,            '_Normal',        '<ctrl>6',         'Normal',        \&on_tag         ],
	[ 'Bold',        'gtk-bold',       '_Bold',          '<ctrl>B',         'Bold',          \&on_tag         ],
	[ 'Italic',      'gtk-italic',     '_Italic',        '<ctrl>I',         'Italic',        \&on_tag         ],
	[ 'Underline',   'gtk-underline',  '_Underline',     '<ctrl>U',         'Underline',     \&on_tag         ],
	[ 'Verbatim',    undef,            '_Verbatim',      '<ctrl>T',         'Verbatim',      \&on_tag         ],
	[ 'Undo',        'gtk-undo',       '_Undo',          '<ctrl>Z',         'Undo',          \&on_Undo        ],
	[ 'Redo',        'gtk-redo',       '_Redo',          '<ctrl><shift>Z',  'Redo',          \&on_Redo        ],
	[ 'Cut',         'gtk-cut',        'Cu_t',           '<ctrl>X',         'Cut',           \&on_clipboard   ],
	[ 'Copy',        'gtk-copy',       '_Copy',          '<ctrl>C',         'Copy',          \&on_clipboard   ],
	[ 'Paste',       'gtk-paste',      '_Paste',         '<ctrl>V',         'Paste',         \&on_clipboard   ],
	[ 'Delete',      'gtk-delete',     'Delete',         'Delete',          'Delete',        \&on_Delete      ],
	[ 'EditLink',    'gtk-properties', '_Link',          '<ctrl>E',         'Edit link',     \&on_EditLink    ],
	[ 'InsertDate',  undef,            '_Date and Time', '<ctrl>D',         'Insert date',   \&on_InsertDate  ],
	[ 'InsertLink',  $LINK_ICON,       '_Link...',       undef,             'Insert link',   \&on_EditLink    ],
	[ 'InsertImage', undef,            '_Image...',      undef,             'Insert image',  \&on_InsertImage ],
);

my @ui_actions_ro = (
	[ 'Link',        $LINK_ICON,       '_Link',          '<ctrl>L',         'Link',          \&on_Link        ],
	[ 'Find',        'gtk-find',       '_Find',          '<ctrl>F',         'Find',          \&on_Find        ],
	[ 'FindNext',    undef,            'Find Ne_xt',     '<ctrl>G',         'Find next',     \&on_FindNext    ],
	[ 'FindPrev',    undef,            'Find Pre_vious', '<ctrl><shift>G',  'Find previous', \&on_FindPrev    ],
);

my $ui_layout = q{<ui>
	<menubar name='MenuBar'>
		<menu action='EditMenu'>
			<menuitem action='Undo'/>
			<menuitem action='Redo'/>
			<separator/>
			<menuitem action='Cut'/>
			<menuitem action='Copy'/>
			<menuitem action='Paste'/>
			<menuitem action='Delete'/>
			<separator/>
			<menuitem action='EditLink'/>
			<separator/>
		</menu>
		<menu action='InsertMenu'>
			<menuitem action='InsertDate'/>
			<separator/>
			<menuitem action='InsertImage'/>
			<menuitem action='InsertLink'/>
		</menu>
		<menu action='FormatMenu'>
			<menuitem action='Head1'/>
			<menuitem action='Head2'/>
			<menuitem action='Head3'/>
			<menuitem action='Head4'/>
			<menuitem action='Head5'/>
			<menuitem action='Normal'/>
			<separator/>
			<menuitem action='Bold'/>
			<menuitem action='Italic'/>
			<menuitem action='Underline'/>
			<menuitem action='Verbatim'/>
			<separator/>
			<menuitem action='Link'/>
		</menu>
	</menubar>
	<toolbar name='ToolBar'>
		<placeholder name='Format'>
			<toolitem action='Link'/>
			<toolitem action='EditLink'/>
			<separator/>
			<toolitem action='Bold'/>
			<toolitem action='Italic'/>
			<toolitem action='Underline'/>
		</placeholder>
		<separator/>
	</toolbar>
</ui>};

my $ui_layout_ro = q{<ui>
	<menubar name='MenuBar'>
		<menu action='EditMenu'>
			<menuitem action='Find'/>
			<menuitem action='FindNext'/>
			<menuitem action='FindPrev'/>
		</menu>
	</menubar>
	<accelerator action='Link'/>
</ui>};

=item C<new(app => PARENT)>

Simple constructor.

=item C<init()>

Method called by the constructor.

=cut

my ($k_tab, $k_return, $k_kp_enter, $k_backspace, $k_escape, $k_multiply, $k_home) =
	@Gtk2::Gdk::Keysyms{qw/Tab Return KP_Enter BackSpace Escape KP_Multiply Home/};

our %UNDO_STEPS = (
	delete     => 'insert',
	insert     => 'delete',
	apply_tag  => 'remove_tag',
	remove_tag => 'apply_tag'
);

$Gtk2::Ex::HyperTextBuffer::TAGS{link} = [foreground => 'blue'];
$Gtk2::Ex::HyperTextBuffer::TAGS{underline} = [background => 'yellow'];

sub new {
	my $class = shift;
	my $self = bless {@_}, $class;
	$self->init();
	return $self;
}

sub init { # called by new()
	my $self = shift;
	$self->{overwrite_mode} = 0;

	my $vbox = Gtk2::VBox->new(0, 0);
	$self->{vbox} = $vbox;
	
	my $scrolled_window = Gtk2::ScrolledWindow->new();
	$scrolled_window->set_policy('automatic', 'automatic');
	$scrolled_window->set_shadow_type('in');
	$vbox->add($scrolled_window);
	$self->{scrolled_window} = $scrolled_window;
	
	# init TextView
	my $htext = Gtk2::Ex::HyperTextView->new();
	$htext->set_left_margin(10);
	$htext->set_right_margin(5);
	$htext->set_editable(0) if $self->{app}{settings}{read_only};
	$htext->set_tabs( Gtk2::Pango::TabArray->new_with_positions(
		# initial_size, pos_in_pixels, ... allign => position
		1, 0, 'left' => 40 * PANGO_SCALE ) );
	$htext->{link_properties} = [foreground => 'blue']; # TextTag properties
	$htext->signal_connect(link_clicked =>
		sub { $self->{app}->link_clicked($_[1][1])                }  );
	$htext->signal_connect(link_enter =>
		sub { $self->{app}->push_status("Go to $_[1][1]", 'link') }  );
	$htext->signal_connect(link_leave =>
		sub { $self->{app}->pop_status('link')                    }  );
	$htext->signal_connect(toggle_overwrite => \&on_toggle_overwrite, $self);
	$htext->signal_connect(populate_popup => \&on_populate_popup);
	$htext->signal_connect(key_press_event => \&on_key_press_event, $self);
	my $reset_em = sub {
		$self->{buffer}->reset_edit_mode if $self->{buffer};
		return 0;
	};
	$htext->signal_connect(move_cursor => $reset_em);
	$htext->signal_connect(button_press_event => $reset_em);
	$scrolled_window->add($htext);
	$self->{htext} = $htext;

	# init search box
	my $hbox = Gtk2::HBox->new(0, 5);
	$hbox->set_no_show_all(1);
	$hbox->signal_connect(key_press_event => \&on_key_press_event_hbox);
	$vbox->pack_start($hbox, 0, 1, 3);
	$self->{hbox} = $hbox;

	$hbox->pack_start( Gtk2::Label->new(' Find: '), 0,1,0);
	
	my $entry = Gtk2::Entry->new();
	$entry->signal_connect_swapped(changed  => \&on_changed_entry, $self);
	$entry->signal_connect_swapped(activate => \&on_activate_entry,  $self);
	$hbox->pack_start($entry, 0, 1, 0);
	$self->{entry} = $entry;

	my $prev_button = $self->{app}->new_button('gtk-go-back', '_Previous');
	$prev_button->signal_connect(clicked =>
		sub { $self->find($entry->get_text, -1) } );
	$hbox->pack_start($prev_button, 0, 1, 0);
	
	my $next_button = $self->{app}->new_button('gtk-go-forward', '_Next');
	$next_button->signal_connect(clicked =>
		sub { $self->find($entry->get_text, 1) } );
	$hbox->pack_start($next_button, 0, 1, 0);

	# add toolbar buttons and key bindings
	my $accels = $self->{app}{ui}->get_accel_group;
	my $read_only = $self->{app}{settings}{read_only};
	my $actions = Gtk2::ActionGroup->new("PageView");
	$self->{app}{ui}->insert_action_group($actions, 0);
	$actions->add_actions(\@ui_actions_ro, $self);
	unless ($read_only) {
		$actions->add_actions(\@ui_actions, $self);
		$self->{app}{ui}->add_ui_from_string($ui_layout);
		$accels->connect( # ^Y (identical with the shift-^Z defined above)
			ord('Y'), ['control-mask'], ['visible'], sub {$self->redo} );
	}
	$self->{app}{ui}->add_ui_from_string($ui_layout_ro);
	$accels->connect( # alt-/ (identical with the ^F defined above)
		ord('/'), ['mod1-mask'], ['visible'], sub {$self->show_find} );

	$self->{htext}->set_cursor_visible( $read_only ? 0 : 1 );
}

sub AUTOLOAD {
	my $self = shift;
	$AUTOLOAD =~ s/^.*:://;
	return if $AUTOLOAD eq 'DESTROY';
	return $self->{htext}->$AUTOLOAD(@_);
}

sub on_tag {
	my ($action, $self) = @_;
	my $tag = lc $action->get_name;
	$self->apply_tag($tag);
}

sub on_Link { pop->apply_link }

sub on_Undo { pop->undo }

sub on_Redo { pop->redo }

sub on_clipboard {
	my ($action, $self) = @_;
	my $signal = lc($action->get_name).'_clipboard';
	$self->{htext}->signal_emit($signal);
}

sub on_Delete { pop->{htext}->signal_emit('delete_from_cursor', 'chars', 1) }

sub on_EditLink { pop->edit_link_dialog }

sub on_Find { pop->show_find }

sub on_FindNext { pop->find(undef, 1) }

sub on_FindPrev { pop->find(undef, -1) }

sub on_InsertDate {
	my $self = pop;
	$self->get_buffer->insert_at_cursor(
		strftime( $self->{app}{settings}{date_string}, localtime) );
}

sub on_InsertImage { pop->insert_image }

=item C<widget()>

Returns the root widget. This should be used to add the object to a container widget.
Also use this widget for things like show_all() and hide_all().

=cut

sub widget { return $_[0]->{vbox} }

sub on_key_press_event_hbox {
	my ($hbox, $event) = @_;
	return 0 unless $event->keyval == $k_escape;
	$hbox->hide;
	return 1;
}

sub on_changed_entry {
	my ($self, $entry) = @_;
	$self->find($entry->get_text, 0);
}

sub on_activate_entry {
	my ($self, $entry) = @_;
	$self->{hbox}->hide;
	$self->find($entry->get_text);
	$self->{htext}->grab_focus;
}

=item C<find(STRING, DIRECTION)>

Finds next occurance of STRING in the buffer, scrolls the buffer
and highlights the string. DIRECTION can be either 1 for forward or -1
for backward. If no direction is given a forward search is done including
the current position.

=cut

sub find {
	my ($self, $string, $direction) = @_;
	my $buffer = $self->{buffer};
	my $iter = $buffer->get_iter_at_mark( $buffer->get_insert );
	$iter->forward_char if $direction == 1;
	
	unless (defined $string) {
		# TODO also use selection here - think ^G while entry is hidden - also set entry when selection
		$string = $self->{entry}->get_text;
	}
	
	my ($start, $end);
	if ($direction == -1) { # backward
		($start, $end) = $iter->backward_search($string, 'visible-only');
		unless (defined $start) { # wrap around
			$iter = $buffer->get_end_iter;
			($start, $end) = $iter->backward_search($string, 'visible-only');
			return unless defined $start;
		}
	}
	else { # forward (direction 1 or 0)
		($start, $end) = $iter->forward_search($string, []);
		unless (defined $start) { # wrap around
			$iter = $buffer->get_start_iter;
			($start, $end) = $iter->forward_search($string, []);
			return unless defined $start;
		}
	}
	
	#print "found $string at offset ".$iter->get_offset."\n";
	$self->set_selection($start, $end);
	$self->{htext}->scroll_mark_onscreen( $buffer->get_insert );
}

=item C<set_selection(START, END)>

Select a region of the buffer.

=cut

sub set_selection {
	my ($self, $start, $end) = @_;
	my $buffer = $self->{htext}->get_buffer;
	$buffer->place_cursor($start);
	$buffer->move_mark( $buffer->get_selection_bound, $end );
#	$buffer->select_range($start, $end); # Since 2.4 FIXME
}

=item C<show_find()>

Show the search bar.

=cut

sub show_find {
	my $self = shift;
	$self->{hbox}->set_no_show_all(0);
	$self->{hbox}->show_all;
	$self->{entry}->grab_focus;
}

=item C<hide_find()>

Hide the search bar.

=cut

sub hide_find {
	my $self = shift;
	$self->{hbox}->hide_all;
	$self->{hbox}->set_no_show_all(1);
	$self->{htext}->grab_focus;
}

sub on_toggle_overwrite {
	my $self = pop;
	$self->{overwrite_mode} = $self->{overwrite_mode} ? 0 : 1;
	$self->{app}->update_status();
}

sub on_populate_popup { # FIXME add this code to HyperTextView
       	# add items to context menu
	my ($htext, $menu) = @_;
	my $iter = $htext->get_iter_at_pointer;
	return unless $iter;
	
	my $string;
	my $link = $htext->get_link_at_iter($iter);
	if (defined $link) {
		$link = $$link[1] if ref($link) eq 'ARRAY';
		$string = ($link =~ s/^mailto:\/*//)
			? 'Copy _Email Address'
			: 'Copy _Link' ;
	}
	else {
		my $pixbuf = $iter->get_pixbuf;
		return unless $pixbuf and defined $pixbuf->{image_src};
		$link = $pixbuf->{image_src};
		$string = 'Copy Image _Location';
	}
	
	my $seperator = Gtk2::MenuItem->new();
	$seperator->show;
	$menu->prepend($seperator);
	
	my $item = Gtk2::MenuItem->new($string);
	$item->signal_connect(activate => sub {
		my $clipboard = Gtk2::Clipboard->get(
			Gtk2::Gdk::Atom->new('PRIMARY') );
		$clipboard->set_text($link);
	} );
	$item->show;
	$menu->prepend($item);
}

sub on_key_press_event { # some extra keybindings
	# FIXME for more consistent behaviour test for selections
	my ($htext, $event, $self) = @_;
	my $val = $event->keyval;

	if ($self->{app}{settings}{read_only}) {
		return 0 unless $val == ord('/');
		$self->show_find;
		return 1;
	}
	
	if ($val == $k_return or $val == $k_kp_enter) { # Enter
		my $buffer = $htext->get_buffer;
		my $iter = $buffer->get_iter_at_mark($buffer->get_insert());
		return 1 if defined $htext->click_if_link_at_iter($iter);
		$self->parse_line($iter) or return 0;
		$htext->scroll_mark_onscreen( $buffer->get_insert );
		return 1;
	}
	elsif (
		$self->{app}{settings}{backsp_unindent} and $val == $k_backspace or
		$val == $k_tab and $event->state eq 'shift-mask'
	) { # BackSpace or Shift-Tab
		my $buffer = $htext->get_buffer;
		my ($start, $end) = $buffer->get_selection_bounds;
		if ($end and $end != $start) {
			my $cont = $self->selection_backspace($start, $end);
			return $val == $k_tab ? 1 : $cont;
		}
		my $iter = $buffer->get_iter_at_mark($buffer->get_insert());
		if ($self->parse_backspace($iter)) {
			$htext->scroll_mark_onscreen( $buffer->get_insert );
			return 1;
		}
	}
	elsif ($val == $k_tab or $val == ord(' ')) { # WhiteSpace
		my $buffer = $htext->get_buffer;
		if ($val == $k_tab) {
			my ($start, $end) = $buffer->get_selection_bounds;
			if ($end and $end != $start) {
				$self->selection_tab($start, $end);
				return 1;
			}
		}
		my $iter = $buffer->get_iter_at_mark($buffer->get_insert());
		my $string = ($val == $k_tab) ? "\t" : ' ';
		if ($self->parse_word($iter, $string)) {
			$htext->scroll_mark_onscreen( $buffer->get_insert );
			return 1;
		}
	}
	elsif ($val == ord('*') or $val == $k_multiply) { # Bullet
		my $buffer = $htext->get_buffer;
		my ($start, $end) = $buffer->get_selection_bounds;
		return 0 if !$end or $end == $start;
		$self->toggle_bullets($start, $end);
		return 1;
	}
	elsif ($val == $k_home and not $event->state >= 'control-mask') { # Home toggle
		my $buffer = $htext->get_buffer;
		my $insert = $buffer->get_iter_at_mark($buffer->get_insert());
		my $start  = $insert->copy;
		$htext->backward_display_line_start($start)
			unless $htext->starts_display_line($start);
		my $begin  = $start->copy;
		my $indent = '';
		while ($indent =~ /^\s*([^\s\w]\s*)?$/) {
			last if $begin->ends_line or ! $begin->forward_char;
			$indent = $start->get_text($begin);
		}
		$indent =~ /^(\s*([^\s\w]\s+)?)/;
		my $back = length($indent) - length($1);
		$begin->backward_chars($back) if $back > 0;
		$insert = ($begin->ends_line || $insert->equal($begin)) ? $start : $begin;
		if ($event->state >= 'shift-mask') {
			$buffer->move_mark_by_name('insert', $insert);
			# leaving the "selection_bound" mark behind
		}
		else { $buffer->place_cursor($insert) }
		return 1;
	}
		
	#else { printf "key %x pressed\n", $val } # perldoc -m Gtk2::Gdk::Keysyms

	return 0;
}

=item C<get_state()>

Returns a number of properties that need to be saved in the history.

=cut

sub get_state {
	my $self = shift;
	my $buffer = $self->{buffer} || return;
	my $cursor = $buffer->get_iter_at_mark($buffer->get_insert)->get_offset;
	#my $vscroll = $self->{scrolled_window}->get_vadjustment->get_value;
	#my $hscroll = $self->{scrolled_window}->get_hadjustment->get_value;
	return cursor => $cursor, undo => $self->{undo}, redo => $self->{redo};
		#, vscroll => $vscroll, hscroll => $hscroll;
}

=item C<set_state(property => value, ..)>

Set a number of properties that could be saved in the history.

=cut

sub set_state {
	my $self = shift;
	my %rec = @_;
	
	if (defined $rec{cursor}) {
		$self->{buffer}->place_cursor(
			$self->{buffer}->get_iter_at_offset($rec{cursor}) );
	
		$self->{htext}->scroll_mark_onscreen( $self->{buffer}->get_insert );
	}
	
	#$self->{scrolled_window}->get_vadjustment->set_value($rec{vscroll});
	#$self->{scrolled_window}->get_hadjustment->set_value($rec{hscroll});
	
	$self->{undo} = $rec{undo} || [];
	$self->{redo} = $rec{redo} || [];
}

=item C<get_status()>

Returns an info string for the current buffer.

=cut

sub get_status {
	my $self = shift;
	return '' . ( $self->{buffer}->get_modified ? '+' : '' ) .
	            ( $self->{overwrite_mode} ? ' --OVERWRITE--' : '' ) .
	            ( $self->{app}{settings}{read_only} ? ' [readonly]' : '' ) ;
}

=item C<load_page(PAGE)>

Load a new page object into the buffer.

=cut

sub load_page {
	my ($self, $page) = @_;
	my $use_spell = defined $self->{app}{objects}{Spell}; # FIXME ugly internals
	
	# clear the old buffer
	$self->{undo_lock} = 1;
	$self->{buffer}->clear if $self->{buffer};
	$self->{app}->Spell->detach($self->{htext}) if $use_spell;
	$self->{_prev_buffer} = $self->{buffer}; # FIXME hack to prevent segfaults
	
	# create a new HyperTextBuffer
	my $buffer = Gtk2::Ex::HyperTextBuffer->new();
	$buffer->create_default_tags;
	$self->{buffer} = $buffer;
	$self->{htext}->set_buffer($buffer);
	$buffer->set_parse_tree( $page->get_parse_tree );
	unless ($self->{app}{settings}{read_only}) {
		# connect signals _after_ load_parsetree()
		$buffer->signal_connect(delete_range => \&on_delete_range, $self);
		$buffer->signal_connect_after(insert_text => \&on_insert_text, $self);
		$buffer->signal_connect(modified_changed =>
			sub {$self->{app}->update_status} );
		$buffer->signal_connect(apply_tag => \&on_apply_tag, $self);
		$buffer->signal_connect(remove_tag => \&on_remove_tag, $self);
	}
	$self->{app}->Spell->attach($self->{htext}) if $use_spell;
	$self->{undo_lock} = 0;
	$buffer->set_modified(0);

	$buffer->place_cursor(
		$buffer->get_iter_at_offset(0) )
			unless $page->status eq 'new';
	
	$self->{undo} = [];
	$self->{redo} = [];
	
	$self->{htext}->scroll_mark_onscreen( $buffer->get_insert );
}

=item C<modified()>

=item C<modified(BOOLEAN)>

Get or set the modified bit. This bit is set when the buffer
is modified by the user.
It should be reset after succesfully saving the buffer content.

=cut

sub modified {
	return 0 unless defined $_[0]->{buffer};
	$_[0]->{buffer}->set_modified($_[1]) if defined $_[1];
	$_[0]->{buffer}->get_modified;
}

=item C<get_parse_tree(PAGE)>

Returns the parse tree for the current buffer contents.

=cut

sub get_parse_tree { $_[0]->{buffer}->get_parse_tree }

=item C<parse_backspace(ITER)>

This method is called when the user types a backspace.
It tries to update the formatting of the current line.
 
When TRUE is returned the widget does not recieve the backspace.

=cut

sub parse_backspace {
	my ($self, $iter) = @_;
	my $buffer = $self->{buffer};
	my $lf = $buffer->get_iter_at_line( $iter->get_line );
	my $line = $buffer->get_text($lf, $iter, 0);
	if ($line =~ s/\t([\*\x{2022}]\s)$/$1/) {
		$buffer->delete($lf, $iter);
		$buffer->insert($lf, $line);
		return 1;
	}
	return 0;
}

=item C<parse_line(ITER)>

This method is called when the user is about to insert a linebreak.
It checks the line left of the cursor of any markup that needs 
updating. It also takes care of autoindenting.

When TRUE is returned the widget does not recieve the linebreak.

=cut

sub parse_line {
	my ($self, $iter) = @_;
	my $buffer = $self->{buffer};
	my $lf = $buffer->get_iter_at_line( $iter->get_line );
	my $line = $buffer->get_text($lf, $iter, 0);
	#print ">>$line<<\n";
	if ($line =~ s/^(=+)\s+(\S)/$2/) { # heading
		my $h = length($1); # no (7 - x) monkey bussiness here
		$h = 5 if $h > 5;
		$line =~ s/\s+=+\s*$//;
		$buffer->delete($lf, $iter);
		$buffer->insert_with_tags_by_name($lf, $line, "head$h");
	}
	elsif ($line =~ /^\s*[\*\x{2022}]\s+$/) { # empty bullet
		$buffer->delete($lf, $iter);
	}
	elsif ($line =~ /^(\s*\W+\s+|\s+)/) { # auto indenting
		$buffer->insert($iter, "\n$1");
		$self->{htext}->scroll_mark_onscreen( $buffer->get_insert() );
		return 1;
	}
	return 0;
}

=item C<parse_word(ITER, CHAR)>

This method is called after the user ended typing a word.
It checks the word left of the cursor for any markup that
needs updating.

CHAR can be the key that caused a word to end, returning TRUE
makes it never reaching the widget.

=cut

sub parse_word {
	my ($self, $iter, $char) = @_;
	return 0 unless $char eq ' ' or $char eq "\t";
	return 0 if $self->_is_verbatim;
	my $buffer = $self->{buffer};
	my $lf = $iter->copy;
	$self->{htext}->backward_display_line_start($lf)
		unless $self->{htext}->starts_display_line($lf);
	my $line = $buffer->get_text($lf, $iter, 0) . $char;
	#print ">>$line<<\n";
	if ($line =~ /^(\s*)[\*\x{2022}](\s+)$/) { # bullet
		# FIXME \s* => \t
		return unless $lf->starts_line;
		my ($pre, $post) = ($1, $2);
		$pre .= $1 if $post =~ s/(\t)+$//; # switch tabs
		$line = $pre."\x{2022}".$post;
		$buffer->delete($lf, $iter);
		$buffer->insert($lf, $line);
		return 1;
	}
#	elsif ($line =~ /^(\t|  )/) { # pre
#		# FIXME \s* => \t
#		$iter->forward_char unless $iter->is_end; # FIXME somthing at end
#		$buffer->apply_tag_by_name('pre', $lf, $iter);
#	}
#	elsif ($line =~ /^(.*)\b(\w+:\/\/\S+[^\s\,\.\;])\s+$/) {
#		# FIXME get the right iters to highlight link
#	} # no wiki link markup supported here
	
	return 0;
}

sub _is_verbatim {
	my $self = shift;
	my $buffer = $self->{buffer};
	my $iter = $buffer->get_iter_at_mark( $buffer->get_insert );
	for ($iter->get_tags) {
		return 1 if lc($_->get_property('name')) eq 'verbatim';
	}
	return 0;
}

=item C<insert_image(IMAGE)>

Inserts an image into the buffer. Without argument prompts the user for a file.

=cut

sub insert_image {
	my ($self, $file) = @_;
	$file = $self->{app}->filechooser_dialog unless $file;
	return unless length $file;
	# TODO check relativeness of filename etc.
	$self->{buffer}->insert_image_from_file($file);
}

=item C<apply_tag(TAG)>

Applies the tag with the name TAG to any selected text.

=cut

sub apply_tag {
	my ($self, $tag) = @_;
	my ($start, $end) = $self->get_selection;
	return unless defined $start;

	if ($start->get_line != $end->get_line) {
		if ($tag eq 'verbatim') { $tag = 'Verbatim' }
		# TODO what if selection contains linebreaks ??
	}
	
	my $buffer = $self->{buffer};
	$buffer->remove_all_tags($start, $end);
	$buffer->apply_tag_by_name($tag, $start, $end)
		unless $tag eq 'normal';
	$buffer->set_modified(1);

	if ($tag =~ /^head/) { # give headers their own line
		$end = $end->ends_line ? undef : $end->get_offset ;
		$buffer->insert($start, "\n") unless $start->starts_line;
		$buffer->insert($buffer->get_iter_at_offset($end+1), "\n")
			unless ! defined $end;
	}
}

=item C<get_selection(TAG)>

=cut

sub get_selection { # selects current word if no selections
	my ($self, $tag) = @_;
	my $buffer = $self->{buffer};
	my ($start, $end) = $buffer->get_selection_bounds;
#	print "selection: $start, $end\n";
	if (!$end or $start == $end) {
		# TODO autoselect word if defined tag
		# ( or line if tag =~ /^head/ );
		return undef;
	}
	return ($start, $end);
}

=item C<apply_link(LINK)>

This method is called by the "Link" button or by the ^L keybinding.
It makes any selected text a link. This link is followed immediatly
if the 'follow_new_link' config option is set.

If LINK is undefined the link target is the same as the selected text.

If no text is selected it calls the "New Link" dialog.

In readonly modus the selected text is regarded as a link and
followed immediatly, but no actual link is made

=cut

sub apply_link {
	my ($self, $link) = @_;
	my ($start, $end) = $self->get_selection('link');
	
	unless (defined $start) {
		$self->{app}{settings}{read_only}
			? $self->{app}->goto_page_dialog
			: $self->edit_link_dialog ;
	}

	my $buffer = $self->{buffer};
	my $text = $buffer->get_text($start, $end, 0);
	$link = $text unless defined $link;
	return undef if $link =~ /\n/;
	
	unless ($self->{app}{settings}{read_only}) {
		my $bit = $link eq $text;
		$buffer->remove_all_tags($start, $end);
		$self->{htext}->apply_link([$bit, $link], $start, $end);
		$buffer->set_modified(1);
	}

	$self->{app}->link_clicked($link)
		if $self->{app}{settings}{read_only}
		or $self->{app}{settings}{follow_new_link};
}

=item C<toggle_bullets()>

If selected text is a bullet list this removes the bullets, else it adds
bullets.

=cut

sub toggle_bullets {
	my ($self, $start, $end) = @_;
	($start, $end) = $self->get_selection unless defined $start;
	return unless defined $start;
	
	my $text = $self->{buffer}->get_text($start, $end, 1);
	if ($text =~ /^\s*[\*\x{2022}]\s+/m) { # remove bullets
		$text =~ s/^(\s*)[\*\x{2022}]\s+/$1/mg
	}
	else { # set bullets
		$text =~ s/^(\s*)(\S)/$1\x{2022} $2/mg;
	}

	$self->replace_selection($text, $start, $end);
}

=item C<replace_selection(TEXT, START, END)>

=cut

sub replace_selection {
	my ($self, $text, $start, $end) = @_;
	($start, $end) = $self->get_selection unless defined $start;
	
	my $buffer = $self->{buffer};
	my $_start = $start->get_offset;
	my $_end   = $start->get_offset + length $text;
	
	$buffer->delete($start, $end);
	$buffer->insert($start, $text);

	($start, $end) = map $buffer->get_iter_at_offset($_), ($_start, $_end);
	$self->set_selection($start, $end);

	return $start, $end;
}

=item C<selection_tab()>

Puts a tab before every line of a selection.

=cut

sub selection_tab {
	my ($self, $start, $end) = @_;
	($start, $end) = $self->get_selection unless defined $start;
	return unless defined $start;
	
	my $buffer = $self->{buffer};
	my $text = $buffer->get_text($start, $end, 1);
	$text =~ s/^/\t/mg;

	my $verbatim = $self->_is_verbatim;
	($start, $end) = $self->replace_selection($text, $start, $end);
	$self->{buffer}->apply_tag_by_name('verbatim', $start, $end)
		if $verbatim;
}

=item C<selection_backspace()>

Removes a tab for every line of a selection.

=cut

sub selection_backspace {
	my ($self, $start, $end) = @_;
	($start, $end) = $self->get_selection unless defined $start;
	return unless defined $start;

	my $text = $self->{buffer}->get_text($start, $end, 1);
	my $verbatim = $self->_is_verbatim;
	if ($text =~ s/^\t//mg) {
		($start, $end) = $self->replace_selection($text, $start, $end);
		$self->{buffer}->apply_tag_by_name('Verbatim', $start, $end)
			if $verbatim;
		return 1;
	}

	return 0;
}

sub on_insert_text { # buffer, iter, string, length, self
#	(length($string) == 1)
#		? push(@undo_chars, $string)
#		: 
	$_[4]->add_undo('insert', $_[1]->get_offset - length($_[2]), $_[2]);
}

sub on_delete_range { # buffer, begin, end, self
	#print "delete range\n";
	my $string = $_[0]->get_text($_[1], $_[2], 0);
	$_[3]->add_undo('delete', $_[1]->get_offset, $string);
}

sub on_apply_tag { pop->_on_change_tag('apply_tag', @_) }

sub on_remove_tag { pop->_on_change_tag('remove_tag', @_) }

sub _on_change_tag {
	my ($self, $action, undef, $tag, $start, $end) = @_;
	my @off = ($start->get_offset, $end->get_offset);
	if ($tag->{is_link}) {
		$self->add_undo($action, @off, 'L', $tag->{link_data});
	}
	else {
		$self->add_undo($action, @off, $tag->get_property('name'));
	}
}

=item C<add_undo(ACTION, OFFSET, DATA)>

=cut

sub add_undo {
	my $self = shift;
	return if $self->{undo_lock}; # prohibit unwanted recursion
#	flush_undo_chars() if @undo_chars;
	my ($action, $offset, @data) = @_;
#	print "do: $action \@$offset: >>@data<<\n";
	push @{$self->{undo}}, [$action, $offset, @data];
	shift @{$self->{undo}} if @{$self->{undo}} > $self->{app}{settings}{undo_max};
	@{$self->{redo}} = ();
}

#sub flush_undo_chars {
#	return unless @undo_chars;
#	add_undo('insert', 
#}

=item C<undo()>

Undo one editing step in the buffer.

=cut

sub undo {
	my $self = shift;
	my ($undo, $redo) = @{$self}{'undo', 'redo'};
	return unless @$undo;
	my $step = pop @$undo;
	unshift @$redo, [@$step]; # force copy;
	$$step[0] = $UNDO_STEPS{$$step[0]};
	$self->_do_step(@$step);
}

=item C<redo()>

Redo one editing step in the buffer.

=cut

sub redo {
	my $self = shift;
	my ($undo, $redo) = @{$self}{'undo', 'redo'};
	return unless @$redo;
	my $step = shift @$redo;
	push @$undo, $step;
	$self->_do_step(@$step);
}

sub _do_step {
	my ($self, $action, $offset, @data) = @_;
	my $buffer = $self->{buffer};
	my $start = $buffer->get_iter_at_offset($offset);
	$self->{undo_lock} = 1;
	if ($action eq 'insert') {
		$buffer->insert($start, $data[0]);
		$buffer->place_cursor(
			$buffer->get_iter_at_offset($offset + length($data[0])));
	}
	elsif ($action eq 'delete') {
		my $end = $buffer->get_iter_at_offset($offset + length($data[0]));
		$buffer->delete($start, $end);
		$buffer->place_cursor($start);
	}
	elsif ($action eq 'apply_tag') {
		my $end = $buffer->get_iter_at_offset( $data[0] );
		$buffer->remove_all_tags($start, $end);
		if ($data[1] eq 'L') {
			$self->{htext}->apply_link($data[2], $start, $end);
		}
		else { $buffer->apply_tag_by_name($data[1], $start, $end) }
	}
	elsif ($action eq 'remove_tag') {
		my $end = $buffer->get_iter_at_offset( $data[0] );
		$buffer->remove_all_tags($start, $end);
	}
	$buffer->set_modified(1);
	$self->{htext}->scroll_mark_onscreen( $buffer->get_insert );
	$self->{undo_lock} = 0;
}

#sub flush_undo_mini {
#	return unless @undo_mini;
#	my (prev,
#	for (@undo_mini) {
#		my ($action, $offset, $char) = @$_;
#		
#}

=item C<edit_link_dialog()>

This dialog allows the user to create a link for which
the link target and the link text differ.

=cut

sub edit_link_dialog {
	my $self = shift;
	my ($start, $end) = $self->get_selection('L');

	my ($text, $link) = ('', '');
	if (defined $start) {
		$link = $self->{htext}->get_link_at_iter($start);
		$link = $$link[1] if $link;
		$text = $self->{buffer}->get_text($start, $end, 0);
		$text = undef if $text =~ /\n/;
	}

	my $title = defined($start) ? 'Edit Link' : 'Insert Link';
	
	($text, $link) = $self->{app}->prompt_link_dialog($text, $link, $title);

	return unless defined $text and ($text =~ /\S/ or $link =~ /\S/);
	
	# both entries default to the other
	$link = $text unless $link =~ /\S/;
	$text = $link unless $text =~ /\S/;

	# use delete + insert instead of apply because the text can be different
	my $buffer = $self->{buffer};
	if (defined $start) {
		$buffer->delete($start, $end);
	}
	else {
		$start = $buffer->get_iter_at_mark( $buffer->get_insert());
	}
	my $bit = $link eq $text;
	$self->{htext}->insert_link_at_iter($start, $text, [$bit, $link]);
	$self->{buffer}->set_modified(1);
}


1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

=cut

