package Zim;

use strict;
use vars qw/$AUTOLOAD %Config/;
use Gtk2;
use Gtk2::Gdk::Keysyms;
use POSIX qw(strftime);
use Zim::History;
use Zim::Components::TreeView;
use Zim::Components::PathBar;
use Zim::Components::PageView;

# TODO make components configable/autoloading like plugins

our $VERSION = 0.08;
our $LONG_VERSION = << "EOT";
zim $VERSION

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

Please report bugs to pardus\@cpan.org
EOT

=head1 NAME

Zim - The application object for zim

=head1 SYNOPSIS

	use Zim;
	use Zim::Repository;
	
	my $zim = Zim->new(\%SETTINGS);
	
	my $repository = Zim::Repository->new($DIR);
	$zim->set_repository($repository);
	
	$zim->main_loop;
	
	exit;

=head1 DESCRIPTION

This is developer documentation, for the user manual try
executing C<zim --doc>. For commandline options see L<zim>(1).

This module provides the application object for the Gtk2 application B<zim>.
The application has been split into several components for which the 
modules can be found in the ZIM::Components:: namespace. This object brings
together these components and manages the settings and data objects.

=head1 EXPORT

None by default.

=head1 METHODS

Undefined methods are AUTOLOADED either to an component or to the main
L<Gtk2::Window> object.

=over 4

=cut

our $LINK_ICON = Gtk2->CHECK_VERSION(2, 6, 0) ? 'gtk-connect' : 'gtk-convert';
	# gtk-connect stock item was introduced in 2.6.0

our %CONFIG = (  # Default config values
	pane_pos	=> 120,
	pane_vis	=> 0,
	pathbar_type	=> 'trace',
	icon_size	=> 'large-toolbar',
	width		=> 500,
	height		=> 350,
	default_root	=> undef,
	default_home	=> 'Home',
	browser		=> undef,
	date_string	=> '%A %d/%m/%Y',
	date_page	=> ':Date:%Y_%m_%d',
	hist_max	=> 20,
	undo_max	=> 50,
	follow_new_link => 1,
	use_xdg_cache	=> 0,
	use_tray_icon	=> 0,
	backsp_unindent => 1,
);

=item C<new(SETTINGS)>

Simple constructor.

=cut

sub new {
	my ($class, $settings) = @_;
	
	%$settings = (%CONFIG, %$settings); # set defaults
	$$settings{root} ||= $$settings{default_root};
	$$settings{home} ||= $$settings{default_home};
	my $self = bless {settings => $settings}, $class;
	$self->load_config;

	$self->{_message_timeout} = -1;
	$self->{_save_timeout} = -1;
	
	return $self;
}

sub DESTROY {
	my $self = shift;
	for (qw/_save_timeout _message_timeout/) {
		my $timeout = $self->{$_};
		Glib::Source->remove($timeout) if $timeout >= 0;
	}
}

sub AUTOLOAD {
	my $self = shift;
	$AUTOLOAD =~ s/^.*:://;
	return if $AUTOLOAD eq 'DESTROY';
	#warn join ' ', "Zim::AUTOLOAD called for $AUTOLOAD by: ", caller, "\n";
	return $self->{objects}{$AUTOLOAD} if exists $self->{objects}{$AUTOLOAD};
	return $self->{window}->$AUTOLOAD(@_);
}

=item C<set_repository(REPOSITORY)>

Set the repository object.

=cut

sub set_repository { $_[0]->{repository} = $_[1] }

=item C<gui_init()>

This method initializes all GUI objects that make up the application.

=cut

sub gui_init {
	my $self = shift;

	my $window = Gtk2::Window->new('toplevel');
	$window->set_default_size(@{$self->{settings}}{'width', 'height'});
	$window->signal_connect_swapped(delete_event => \&on_delete_event, $self);
	$window->signal_connect(destroy => sub { Gtk2->main_quit });
	$window->set_icon(
		Gtk2::Gdk::Pixbuf->new_from_file($self->{settings}{icon_file}) );
	$window->set_title('Zim');
	$self->{window} = $window;

	my $accels = Gtk2::AccelGroup->new;
	$window->add_accel_group($accels);
	$self->{accels} = $accels;

	my $vbox = Gtk2::VBox->new(0, 3);
	$window->add($vbox);
	$self->{vbox} = $vbox;

	my $toolbar = Gtk2::Toolbar->new();
	$toolbar->set_style('icons');
	$vbox->pack_start($toolbar, 0,1,0);
	$self->{toolbar} = $toolbar;

	my $hpaned = Gtk2::HPaned->new();
	$hpaned->set_position($self->{settings}{pane_pos});
	$vbox->add($hpaned);
	$self->{hpaned} = $hpaned;

	my $r_vbox = Gtk2::VBox->new(0, 5);
	$hpaned->add2($r_vbox);
	$self->{r_vbox} = $r_vbox;

	my $statusbar = Gtk2::Statusbar->new;
	$vbox->pack_start($statusbar, 0,1,0);
	$self->{statusbar} = $statusbar;

	$self->add_button(
		'Toggle Index', 'Alt I', 'gtk-index',
		sub {$self->toggle_pane($self->{settings}{pane_vis} ? 0 : 1)} );
	
	my $tree_view = Zim::Components::TreeView->new(app => $self);
	$tree_view->signal_connect(row_activated =>
		sub { $self->toggle_pane(0) if $self->{_pane_visible} == -1 } );
	$hpaned->add1($tree_view->widget);
	$self->{objects}{TreeView} = $tree_view;

	$toolbar->append_space;
	
	$self->add_button('Home', 'Alt Home', 'gtk-home',
		sub {$self->load_page($self->{settings}{home})} );
	my $back_button = $self->add_button(
		'Go Back', 'Alt Left', 'gtk-go-back', sub {$self->go_back(1)} );
	my $forw_button = $self->add_button(
		'Go Forward', 'Alt Right', 'gtk-go-forward', sub {$self->go_forw(1)} );
	$back_button->set_sensitive(0);
	$forw_button->set_sensitive(0);

	$toolbar->append_space;

	my $path_bar = Zim::Components::PathBar->new(app => $self);
	$r_vbox->pack_start($path_bar->widget, 0,1,0);
	$self->{objects}{PathBar} = $path_bar;

	my $page_view = Zim::Components::PageView->new(app => $self);
	$r_vbox->add($page_view->widget);
	$self->{objects}{PageView} = $page_view;

	if ($self->{settings}{use_xdg_cache}) {
		my $root = File::Spec->rel2abs($self->{settings}{root_dir});
		my ($vol, $dirs) = File::Spec->splitpath($root, 'NO_FILE');
		my @dirs = File::Spec->splitdir($dirs);
		$self->{settings}{hist_file} ||= File::Spec->catfile( 
			File::BaseDir->xdg_cache_home, 'zim', join ':', $vol, @dirs );
	}
	else {
		my $root = File::Spec->rel2abs($self->{settings}{root_dir});
		$self->{settings}{hist_file} ||= File::Spec->catfile($root, '.zim.history'); 
	}
	my $history = Zim::History->new(
		$self->{settings}{hist_file}, $page_view, $self->{settings}{hist_max});
	$self->{objects}{History} = $history;

	$toolbar->append_space;

	$self->add_button('Actions', '', 'gtk-execute',
		sub {$self->filechooser_dialog} );
	$self->add_button('About', 'F1', 'gtk-help',
		sub {$self->about_dialog} );


	$self->add_key('Ctrl S', sub {$self->save_page('FORCE')})
		unless $self->{settings}{read_only};

	$self->add_key('Ctrl G', sub {$self->goto_page_dialog});
	$self->add_key('Ctrl R', sub {$self->reload});
	$self->add_key('Ctrl Q', sub {$self->quit('FORCE')});
	$self->add_key('Ctrl W', sub {$self->quit});
	$self->add_key('Alt Up', sub {
		my $namespace = $self->{page}->namespace;
		return if $namespace eq ':';
		$self->load_page($namespace);
	} );
	$self->add_key('Alt Down', sub {
		my $namespace = $self->History->get_namespace;
		my $name = $self->{page}->name;
		return unless $namespace =~ /^(:*$name:+[^:]+)/;
		$self->load_page($1);
	} );
	$self->add_key('Alt D', sub {
		my $page = strftime($self->{settings}{date_page}, localtime);
		$self->load_page($page);
	} );
	$self->add_key('Ctrl  ', sub { # Ctrl-Space
		if ($self->{settings}{pane_vis}) {
			if ($tree_view->has_focus) { $page_view->grab_focus }
			else                       { $tree_view->grab_focus }
		}
		else { $self->toggle_pane($self->{_pane_visible} ? 0 : -1) }
	} );
	$self->add_key('Alt J', sub {$self->toggle_path_bar} );

	$self->signal_connect('page_loaded', sub {
		my ($back, $forw) = $history->state;
		$back_button->set_sensitive($back?1:0);
		$forw_button->set_sensitive($forw?1:0);
	} );

	# set save interval
	$self->{_save_timeout} = 
		Glib::Timeout->add(5000, sub {$self->save_page; 1})
		unless $self->{settings}{read_only};
	
	$window->show_all;

	if ($self->{settings}{use_tray_icon}) {
		eval {
			eval 'use Zim::Components::TrayIcon'; die $@ if $@;
			my $trayicon = Zim::Components::TrayIcon->new(app => $self);
		} ;
		warn $@ if $@;
	}

#	unless ($self->{settings}{read_only}) {
		# FIXME this is a ugly and quick hack
		# FIXME should this be in the repository api ??
#		my $root = $self->{repository}{root};
#		my $lock_file = File::Spec->catfile($root, '.zim.pid');
#		exit 1 if -e $lock_file and
#			! $self->repository_lock_dialog($root);
#		open LOCK, ">$lock_file" or $self->exit_error(
#			"Could not write lock file:\n$lock_file" );
#		print LOCK "$$\n";
#		close LOCK;
#		$self->{lock_file} = $lock_file;
#	}
	
	# Try saving when the user exits shell or desktop
	$SIG{$_} = sub {
		$self->quit('FORCE');
		exit 1;
	} for qw/INT TERM HUP/;
}

=item C<widget()>

Returns the root window widget.
Use this widget for things like show_all() and hide_all().

=cut

sub widget { return $_[0]->{window} }

=item C<signal_connect(NAME, CODE, DATA)>

=cut

sub signal_connect {
	my ($self, $signal, $code, @data) = @_;
	$self->{_signals}{$signal} ||= [];
	push @{$self->{_signals}{$signal}}, [$code, @data];
}

=item C<signal_emit(NAME, ARGS)>

=cut

sub signal_emit {
	my ($self, $signal, @args) = @_;
	return unless exists $self->{_signals}{$signal};
	for (grep defined($_), @{$self->{_signals}{$signal}}) {
		my ($code, @data) = @$_;
		eval { $code->($self, @args, @data) };
		warn if $@;
	}
}

=item C<load_config>

Read config file.

=cut

sub load_config {
	my $self = shift;
	my $name = $self->{settings}{read_only} ? 'conf.ro' : 'conf.rw' ;
	my ($file) =
		grep { -f $_ && -r $_ }
		map  { File::Spec->catdir($_, 'zim', $name) }
		( File::BaseDir->xdg_config_home, File::BaseDir->xdg_config_dirs );

	return unless defined $file;
	$self->{settings}{conf_file} = $file;
	
	open CONF, $file or $self->exit_error("Could not read\n$file");
	while (<CONF>) {
		/^(.+?)=(.*)$/ or next;
		$self->{settings}{$1} = $2 if length $2;
	}
	close CONF;
}

=item C<save_config>

Save config file.

=cut

sub save_config {
	my $self = shift;
	$self->{settings}{pane_pos} = $self->{hpaned}->get_position;
	
	my $dir  = File::Spec->catdir(
		File::BaseDir->xdg_config_home, 'zim');
	_mkdir($dir) unless -e $dir;
	my $name = $self->{settings}{read_only} ? 'conf.ro' : 'conf.rw' ;
	my $file = File::Spec->catfile($dir, $name);
	
	open CONF, ">$file" or $self->exit_error("Could not write config\n$file");
	print CONF "# zim version:$Zim::VERSION\n";
	print CONF "$_=$$self{settings}{$_}\n"
		for sort keys %CONFIG; # we do not save all settings
	close CONF;
	#print "saved config to $file\n";
}

sub _mkdir {
	my ($vol, $dirs) = (@_ == 1)
		 ? (File::Spec->splitpath(shift(@_), 'NO_FILE')) : (@_) ;
	my @dirs = File::Spec->splitdir($dirs);
	my $path = File::Spec->catpath($vol, shift @dirs);
	mkdir $path or die "Could not create dir $path\n"
		if length $path and ! -d $path;
	while (@dirs) {
		$path = File::Spec->catdir($path, shift @dirs);
		mkdir $path or die "Could not create dir $path\n"
			unless -d $path;
	}
}


=item C<main_loop()>

This method runs the GUI application.

=cut

sub main_loop {
	my $self = shift;
	
	unless ($self->{page}) {
		my $page = $self->{objects}{History}->get_current
			|| $self->{settings}{home};
		$self->load_page($page);
		# FIXME what if we are read_only and page does not exist ?
		# 	load_apge returns undef
		# 	first try home, but might also not exist
		# 	need page that always exists ... like about or version page
	}
	
	$self->{message_lock} = 1;
	$self->toggle_pane($self->{settings}{pane_vis});
	$self->toggle_path_bar($self->{settings}{pathbar_type});
	$self->{message_lock} = 0;

	$self->{objects}{PageView}->grab_focus;

	Gtk2->main;
}

=item C<quit()>

Exit the main loop.

=cut

sub quit {
	my $self = shift;
	return if $$self{_saved_state};
	$$self{_saved_state} = 1;
	$self->on_delete_event(@_) && return;
	Gtk2->main_quit;
}

sub on_delete_event {
	# Called when the window is deleted
	# return true to avoid destruction
	my $self = shift;
	my $force = uc(shift) eq 'FORCE';

	if ($self->{settings}{hide_on_delete} and !$force) {
		$self->{window}->hide;
		return 1;
	}
	
	#print STDERR "Zim saving state\n";
	
	eval {	@{$self->{settings}}{'width', 'height'} =
			$self->{window}->get_size          };
	
	$self->{save_lock} = 0; # just to be sure
	$self->save_page() || return 1;
	$self->save_config;

	unless ($self->{settings}{read_only}) {
		my ($vol, $dirs) = File::Spec->splitpath($self->{settings}{hist_file});
		_mkdir($vol, $dirs);
		$self->{objects}{History}->write;
	}

	if ($self->{lock_file}) {
		unlink $self->{lock_file}
			or warn "Could not remove $self->{lock_file}\n";
	}

	return 0;
}

=item C<link_clicked(LINK)>

Loads a page in zim or open an external uri in a browser.
LINK is considered to be either an url or a page name.
Page names are resolved as relative links first.

=cut

sub link_clicked {
	my ($self, $link) = @_;
	$link = $$link[1] if ref $link; # link_data = [bit, link]
	return warn "Warning: You tried to folow an empty link.\n"
		unless length $link;

	if ($link =~ /^(\w+:\/\/|mailto:)/) { # link is an url
		my $browser = $self->{settings}{browser} || $ENV{BROWSER};
		unless ($browser) {
			$browser = $self->prompt_browser_dialog;
			return $self->error_dialog('You have no browser configured')
				unless $browser;
			$self->{settings}{browser} = $browser;
		}
		$browser =~ s/\%s/$link/ or $browser .= ' '.$link;
		unless (fork) { # child process
			exec $browser;
			exit 1; # just to be sure
		}
	}
	else {
		$self->load_page(
			$self->{page}->resolve_link($link) );
	}
}

=item C<go_back(INT)>

Go back one or more steps in the history stack.

=cut

sub go_back {
	my $self = shift;
	my $i = shift || 1;
	my $hist = $self->{objects}{History};
	$hist->back($i) or return;
	$self->load_page($hist->get_current);
}

=item C<go_forw(INT)>

Go forward one or more steps in the history stack.

=cut

sub go_forw {
	my $self = shift;
	my $i = shift || 1;
	my $hist = $self->{objects}{History};
	$hist->forw($i) or return;
	$self->load_page($hist->get_current);
}

=item C<reload()>

Save and reload the current page.

=cut

sub reload {
	my $self = shift;
	my $hist = $self->{objects}{History};
	$self->load_page($hist->get_current);
}

=item C<load_page(PAGE)>

Loads a new page in the PageView, updates history etc. when
necessary. PAGE should be either an absolute page name, a history record
or a page object.

=cut

sub load_page {
	my ($self, $page, @args) = @_;
	
	if ($self->{page}) {
		$self->{save_lock} = 0; # just to be sure
		$self->save_page || return;
	}

	my $rec;
	if (ref($page) eq 'HASH') { # hist record
		($rec, $page) = ($page, $page->{name});
	}
	
	if (ref $page) { # page object
		$self->{page} = $page;
		$page = $self->{page}->name;
	}
	else {
		return warn "Warning: You tried to load an empty name.\n"
			unless length $page;
		
		my $p;
		eval { $p = $self->{repository}->load_page($page, @args) };
		return $self->error_dialog("Could not load page $page\n\n$@")
			if $@;
		
		return $self->error_dialog("Page does not exist:\n$page")
			if $self->{settings}{read_only}
			and $p->status eq 'new';

		$self->{page} = $p;
	}

	$self->{objects}{History}->set_current($self->{page});
	eval { $self->{objects}{PageView}->load_page($self->{page}) };
	if ($@) {
		$self->exit_error("Could not load page $page\n\n$@");
		# TODO handle more gratiously
		# .. switching read_only would help here
	}
	$rec = $self->{objects}{History}->record($page) unless defined $rec;
	$self->{objects}{PageView}->set_state(%$rec) if defined $rec;
	$self->signal_emit('page_loaded', $self->{page}->name);
	
	$self->{window}->set_title($self->{page}->name.' - Zim');
	$self->update_status();

	$self->{objects}{PageView}->grab_focus;
}

=item C<save_page(FORCE)>

Check if the current page needs to be saved. If FORCE is true the page is saved
whether it was modified or not. Wrapper for C<Zim::Repository->save_page()>.

=cut

# TODO split out to save_page_if_modified() and save_page()
# save_page_if_modified(FORCE) overrules save_lock
# save_page() just saves it, no questions asked
# (maybe call the first one check_save_page() ?)

sub save_page {
	my ($self, $force) = @_;
	my $modified = $self->{objects}{PageView}->modified;
	return 0 if $self->{app}{settings}{read_only};
	unless ($force) {
		return 1 if ! $modified;
		return 0 if $self->{save_lock};
	}
	
	$self->{save_lock} = 1; # Used to block autosave while "could not save" dialog
	
	my $page = $self->{page};
	$self->{objects}{PageView}->save_page($page);
	my $tree = $page->parse_tree;
	
	eval {
		my $old_status = $page->status;
		if (@$tree > 2) { # content
			$self->{repository}->save_page($page);
			$self->signal_emit('page_created', $page->name)
				if $old_status eq 'new' or $old_status eq 'deleted';
		}
		else {
			$self->{repository}->delete_page($page);
			$self->signal_emit('page_deleted', $page->name);
		}
	};
	unless ($@) {
		$self->{objects}{PageView}->modified(0);
		$self->signal_emit('page_saved', $page->name);
		$self->update_status();
	}
	else { # FIXME this dialog could be a lot better
		my $page = $self->prompt_page_dialog($page->name,
			'Save As', 'Save Page As:', $@, 'gtk-dialog-error' );
		return 0 unless defined $page;
			# leave lock in position after returning
		# FIXME check page exists
		$self->{page} = $self->{repository}->load_page($page);
		$self->save_page('FORCE'); # recurs
	}

	$self->{save_lock} = 0;
	return 1;
}

=item C<move_page(FROM, TO)>

Wrapper for C<Zim::Repository->move_page()>.

Move page from FROM to TO. If TO is undefined a popup is 
shown to ask for a page name.

=cut

sub move_page {
	my ($self, $from, $to) = @_;

	$from = $self->{page}->name unless defined $from;
	
	my $move_current = ($from eq $self->{page}->name);
	if ($move_current) { # FIXME clearer interface for this
		$self->{save_lock} = 0 if $move_current;
		$self->save_page or return;
	}
	else { $self->save_page }
	
	$to = $self->prompt_move_page_dialog($from)
		unless defined $to;
	
	return unless defined $to;

	# TODO check existence of target file
	#print "Move '$from' to '$to'\n";
	eval { $self->{repository}->move_page($from, $to) };
	return $self->error_dialog("Could not move $from to $to\n\n$@") if $@;

	# TODO trigger this with signals
	# problem is that we might have moved a whole namespace
	$self->{objects}{TreeView}->{_loaded} = 0;
	$self->{objects}{TreeView}->load_index;

	$self->load_page($to) if $move_current;
}

=item C<delete_page(PAGE)>

Wrapper for C<Zim::Repository->delete_page>.
Asks the user for confirmation.

If PAGE is undefined the current page is deleted.

=cut

sub delete_page {
	# FIXME option to delete a complete sub-tree
	# FIXME integrate with the save_page() logic, now we have 
	# redundant code here
	my ($self, $page) = @_;
	$page = $self->{page}->name unless defined $page;
	
	my $name = ref($page) ? $page->name : $page;
	my $delete_current = ($name eq $self->{page}->name);
	$self->save_page; # Mind you, overriding save lock could lock us here
	
	return unless $self->prompt_confirm_delete_page($name);
	
	$page = $self->{page} if $delete_current; # make sure we have the right object
	eval { $self->{repository}->delete_page($page) };
	return $self->error_dialog("Could not delete $name\n\n$@") if $@;

	# TODO trigger this with signals
	# problem is that we might have deleted a whole namespace
	$self->{objects}{TreeView}->{_loaded} = 0;
	$self->{objects}{TreeView}->load_index;

	if ($delete_current) {
		eval { $self->{objects}{PageView}->load_page($page) };
		if ($@) {
			$self->exit_error("Could not load page $name\n\n$@");
			# TODO handle more gratiously
			# .. switching read_only would help here
		}
	}
}

=item C<message(STRING)>

Flash a message in the statusbar for a short time.

=cut

sub message {
	my ($self, $str) = @_;
	return if $self->{message_lock};
	my $timeout = $self->{_message_timeout};
	$self->push_status($str, 'message');
	Glib::Source->remove($timeout) if $timeout >= 0;
	$timeout = Glib::Timeout->add(2500, sub {
		$self->pop_status('message');
		$self->{_message_timeout} = -1;
		return 0; # removes timeout
	} );
	$self->{_message_timeout} = $timeout;
}

=item C<update_status()>

Sets the statusbar to display the current page name and some other
information.

=cut

sub update_status {
	my $self = shift;
	my $stat = ' '.$self->{page}->name;
	if ($_ = $self->{page}->status()) { $stat .= ' ('.uc($_).')' }
	$stat .= $self->{objects}{PageView}->get_status;
	$self->push_status($stat, 'page');
}

=item C<push_status(STRING, CONTEXT)>

Put STRING in the status bar.

=cut

sub push_status {
	my ($self, $str, $id) = @_;
	my $statusbar = $self->{statusbar};
	$id = $statusbar->get_context_id($id);
	$statusbar->pop($id);
	$statusbar->push($id, $str);
}

=item pop_status(CONTEXT)

Removes a string from the status bar.

=cut

sub pop_status {
	my ($self, $id) = @_;
	my $statusbar = $self->{statusbar};
	$id = $statusbar->get_context_id($id);
	$statusbar->pop($id);
}

=item C<new_button(STOCK, TEXT)>

Creates a button with a stock image but different text.

=cut

sub new_button {
	my ($self, $stock, $text) = @_;
	my $hbox = Gtk2::HBox->new;
	$hbox->pack_start(
		Gtk2::Image->new_from_stock($stock, 'button'), 0,0,0);
	$hbox->pack_start(
		Gtk2::Label->new_with_mnemonic($text), 1,1,0);
	my $button = Gtk2::Button->new();
	$button->add(Gtk2::Alignment->new(0.5, 0.5, 0, 0));
	$button->child->add($hbox);
	return $button;
}

=item C<add_button(TEXT, KEY, STOCK, CODE)>

Add a button to the toolbar.

=cut

sub add_button {
	my ($self, $text, $key, $stock, $code) = @_;
	my $icon = Gtk2::Image->new_from_stock($stock, $self->{settings}{icon_size});
	my $button = $self->{toolbar}->append_item(
		#  text, tooltip_text, tooltip_private_text, icon, callback
		$text, $text.($key?" ($key)":''), 'Foo Bar', $icon, $code);
	if ($key) {
		my $mod = [];
		($mod, $key) = split ' ', $key, 2 if $key =~ / /;
		if    ($mod eq 'Alt')  { $mod = 'mod1-mask'    }
		elsif ($mod eq 'Ctrl') { $mod = 'control-mask' }
		$key = $Gtk2::Gdk::Keysyms{$key} || ord($key);
		$button->add_accelerator(
			'clicked', $self->{accels}, $key, $mod, 'visible');
	}
	return $button;
}

=item C<add_key(KEY, CODE)>

Add a key accelerator.

=cut

sub add_key {
	my ($self, $key, $code) = @_;
	
	my $mod = [];
	push @$mod, 'control-mask' if $key =~ s/^ctrl //i;
	push @$mod, 'mod1-mask'    if $key =~ s/^alt //i;
	push @$mod, 'shift-mask'   if $key =~ s/^shift //i;
	$key = $Gtk2::Gdk::Keysyms{$key} || ord($key);

	$self->{accels}->connect($key, $mod, 'visible', $code);
}

=item toggle_pane(BOOLEAN)

Show or hide the side pane with the index tree.
If BOOLEAN is "-1" the pane will only be toggled on untill a page is selected.

=cut

sub toggle_pane {
	my ($self, $pane_visible) = @_;
	$self->{_pane_visible} = $pane_visible;
	my $tree_view = $self->{objects}{TreeView};
	my $hpaned = $self->{hpaned};
	if ($pane_visible) {
		$hpaned->set_position($self->{settings}{pane_pos});
		$tree_view->show;
		$tree_view->load_index();
		$tree_view->grab_focus;
	}
	else {
		$self->{settings}{pane_pos} = $hpaned->get_position();
		$tree_view->hide;
		$self->{objects}{PageView}->grab_focus;
	}
	$self->{settings}{pane_vis} = $pane_visible
		unless $pane_visible == -1;
}

=item C<toggle_path_bar(TYPE)>

Set or cycle the type of the pathbar.

=cut

sub toggle_path_bar {
	my $self = shift;
	my $type = $_[0] ? shift :
		($self->{settings}{pathbar_type} eq 'trace')     ? 'namespace' :
		($self->{settings}{pathbar_type} eq 'namespace') ? 'hidden'    : 'trace';
	$self->{settings}{pathbar_type} = $type;
	
	my $path_bar = $self->{objects}{PathBar};
	if ($type eq 'trace' or $type eq 'namespace') {
		$path_bar->widget->show_all;
		$path_bar->set_type($type);
	}
	elsif ($type eq 'hidden') {
		$path_bar->widget->hide_all;
		$path_bar->clear_items;
	}
	else { warn "unknown pathbar_type: $type\n" }

	$self->message("Pathbar type: $type");
}



# TODO the various dialogs could be out-sourced using Autoloader
# also a lot of code could be reduced by haveing a generic
# _prompt_dialog(title => ... text => ... fields => { })

=item C<exit_error(ERROR)>

Like C<error_dialog> but exits afterwards.

=cut

sub exit_error {
	my $self = shift;
	$self->error_dialog(@_);
	exit 1;
}

=item C<error_dialog(ERROR)>

This method is used to display errors.

=cut

sub error_dialog {
	my ($self, $text1, $text2) = @_;
	$text2 ||= $@ || $text1;
	my $dialog = Gtk2::MessageDialog->new(
		$self->{window}, 'modal', 'error', 'ok', $text1 );
		# parent, flags, type, buttons, format, ...
	print STDERR "zim: $text2\n";
	$dialog->run;
	$dialog->destroy;
}

=item C<repository_lock_dialog>

=cut

sub repository_lock_dialog {
	my $self = shift;
	my $dir = shift;
	my $dialog = Gtk2::MessageDialog->new(
		$self->{window}, 'modal', 'warning', 'yes-no', << "EOT");
The directory $dir 
seems to be already in use by
an other instance of zim.

Are you sure you want to continue ?
EOT
	$dialog->set_default_response('no');
	my $ok = ($dialog->run eq 'yes') ? 1 : 0;
	$dialog->destroy;
	return $ok;
}

=item C<goto_page_dialog()>

Prompts the user for a page name. This page is then opened.

=cut

sub goto_page_dialog {
	my $self = shift;
	my $page = $self->{page}->namespace;
	$page = $self->prompt_page_dialog($page, 'Go to', '_Go to Page:');
	$self->link_clicked($page) if defined $page;
}

=item C<prompt_page_dialog(PAGE, TITLE, PROMPT, TEXT, ICON)>

Method used to prompt the user for a page name. Returns PAGE.

The I<text> and I<icon> arguments are optional.

=cut

sub prompt_page_dialog {
	my ($self, $page, $title, $prompt, $text, $icon) = @_;

	my $dialog = Gtk2::Dialog->new(
		$title, $self->{window},
	       	[qw/modal destroy-with-parent no-separator/],
		'gtk-cancel' => 'cancel',
		'gtk-open'   => 'ok',
	);
	$dialog->set_resizable(0);
	$dialog->set_border_width(5);
	$dialog->set_icon($self->{window}->get_icon);
	$dialog->set_default_response('ok');

	if (defined $text) {
		my $text_hbox = Gtk2::HBox->new(0, 5);
		$text_hbox->set_border_width(5);
		$dialog->vbox->add($text_hbox);
		
		if (defined $icon) {
			my $image = Gtk2::Image->new_from_stock($icon, 'dialog');
			$text_hbox->add($image);
		}
		
		my $text_label = Gtk2::Label->new($text);
		$text_hbox->add($text_label);
	}
	
	my $hbox  = Gtk2::HBox->new(0, 5);
	$hbox->set_border_width(5);
	$dialog->vbox->add($hbox);
	
	my $label = Gtk2::Label->new_with_mnemonic($prompt);
	$hbox->add($label);
	
	my $entry = Gtk2::Entry->new();
	$entry->signal_connect(activate => sub { $dialog->response('ok') });
	$entry->set_text($page);
	$hbox->add($entry);
	
	$dialog->show_all;
	if ($dialog->run eq 'ok') { $page = $entry->get_text }
	else                      { $page = undef            }
	$dialog->destroy;

	$page = undef unless $page =~ /\S/;
	return $page;
}

=item C<prompt_move_page_dialog>

=cut

sub prompt_move_page_dialog {
	my ($self, $page) = @_;
	
	my $title = "Move page";
	my $dialog = Gtk2::Dialog->new(
		$title, $self->{window},
	       	[qw/modal destroy-with-parent no-separator/],
		'gtk-cancel' => 'cancel',
	);
	$dialog->set_resizable(0);
	$dialog->set_border_width(5);
	$dialog->set_icon($self->{window}->get_icon);
	$dialog->set_default_response('ok');

	my $move_button = $self->new_button('gtk-save', '_Move');
	$dialog->add_action_widget($move_button, 'ok');
	# $dialog->set_default_response('ok'); FIXME

	my $entry = Gtk2::Entry->new();
	$entry->signal_connect(activate => sub { $dialog->response('ok') });
	$entry->set_text($page);
	$dialog->vbox->add($entry);

	$dialog->show_all;
	$entry->select_region(length($1), -1) if $page =~ /^(.+:)/;

	if ($dialog->run eq 'ok') { $page = $entry->get_text }
	else                      { $page = undef            }
	$dialog->destroy;

	$page = undef unless $page =~ /\S/;
	return $page;
}

=item C<prompt_link_dialog(TEXT, LINK, TITLE)>

Method used to prompt the user for a link. Returns TEXT and TITLE.

=cut

sub prompt_link_dialog {
	my ($self, $text, $link, $title) = @_;
	
	my $dialog = Gtk2::Dialog->new(
		$title, $self->{window},
	       	[qw/modal destroy-with-parent no-separator/],
		'gtk-cancel'  => 'cancel',
	);
	$dialog->set_resizable(0);
	$dialog->set_border_width(5);
	$dialog->set_icon($self->{window}->get_icon);
	
	my $link_button = $self->new_button($LINK_ICON, '_Link');
	$dialog->add_action_widget($link_button, 'ok');
	# $dialog->set_default_response('ok'); FIXME

	my $table = Gtk2::Table->new(2, 2);
	$table->set_border_width(5);
	$table->set_row_spacings(5);
	$table->set_col_spacings(5);
	$dialog->vbox->add($table);
	
	my $text_label = Gtk2::Label->new('Text:');
	my $link_label = Gtk2::Label->new('Links to:');
	my $text_entry = Gtk2::Entry->new();
	$text_entry->signal_connect(activate => sub { $dialog->response('ok') });
	my $link_entry = Gtk2::Entry->new();
	$link_entry->signal_connect(activate => sub { $dialog->response('ok') });
	$table->attach_defaults($text_label, 0,1, 0,1);
	$table->attach_defaults($text_entry, 1,2, 0,1);
	$table->attach_defaults($link_label, 0,1, 1,2);
	$table->attach_defaults($link_entry, 1,2, 1,2);
	
	$link_entry->set_text($link) if defined $link;
	if (defined $text) {
		$text_entry->set_text($text);
		$link_entry->grab_focus;
	}
	else { $text_entry->grab_focus }
	
	$dialog->show_all;
	if ($dialog->run eq 'ok') {
		$text = $text_entry->get_text;
		$link = $link_entry->get_text;
		$dialog->destroy;
		return undef
			unless $text =~ /\S/ or $link =~ /\S/;
		return $text, $link;
	}
	else {
		$dialog->destroy;
		return undef;
	}
}

=item C<prompt_confirm_delete_page(PAGE)>

Ask the user if they are sure about deleting a page.
Returns true if the user wants to proceed.

=cut

sub prompt_confirm_delete_page {
	my ($self, $page) = @_;
	
	my $title = "Delete page";
	my $dialog = Gtk2::Dialog->new(
		$title, $self->{window},
	       	[qw/modal destroy-with-parent no-separator/],
		'gtk-cancel'  => 'cancel',
		'gtk-ok'      => 'ok'
	);
	$dialog->set_resizable(0);
	$dialog->set_border_width(5);
	$dialog->set_icon($self->{window}->get_icon);
	$dialog->set_default_response('ok');

	$dialog->vbox->add( Gtk2::Label->new( << "EOT" ) );
Are you sure you want to
delete $page ?
EOT
	$dialog->show_all;
	my $ok = ($dialog->run eq 'ok') ? 1 : 0;
	$dialog->destroy;
	
	return $ok;
}

=item C<about_dialog()>

This dialog tells you about the version of zim you are using,
the copyright and the current config file. Most importantly 
it has a button to open the manual.

=cut

sub about_dialog {
	my $self = shift;
	
	my $dialog = Gtk2::Dialog->new(
		'About Zim', $self->{window},
	       	[qw/modal destroy-with-parent/],
		'gtk-help' => 'help',
		'gtk-ok' => 'ok',
	);
	$dialog->set_resizable(0);
	$dialog->set_border_width(5);
	$dialog->set_icon($self->{window}->get_icon);
	$dialog->set_default_response('ok');
	
	$dialog->vbox->add(
		Gtk2::Image->new_from_file($self->{settings}{icon_file}) );

	my $text = $LONG_VERSION;
	$text =~ s/^(.*)$/<b>$1<\/b>/m; # set first line bold
	$text .= "\n\nThe current config file is: $self->{settings}{conf_file}\n";
	$text .= "\nThe repository config is: $self->{repository}{conffile}\n"
		if exists $self->{repository}{conffile};
	my $label = Gtk2::Label->new();
	$label->set_markup($text);
	$label->set_justify('center');
	$dialog->vbox->add($label);
	
	$dialog->show_all;
	if ($dialog->run eq 'help') {
		unless (fork) { # child process
			no warnings;
			@{$self->{settings}}{'width', 'height'} = $self->{window}->get_size;
			$self->save_config(); # maybe we are read_only too
			exec $0, '--doc';
			# backup procedure to find perl
			# when $0 is not executable for some reason
			eval 'use Config';
			exit 1 if $@;
			my $perl = $Config{perl5} || 'perl';
			exec $perl, $0, '--doc';
			exit 1; # just to be sure
		}
	}
	$dialog->destroy;
}

=item C<filechooser_dialog(FILE, TITLE)>

Ask the user for a filename. FILE is the suggested filename.

=cut

sub filechooser_dialog {
	my ($self, $file, $title) = @_;
	
	my $dialog;
	$title ||= 'Select File';
	if (Gtk2->CHECK_VERSION(2, 4, 0) and $Gtk2::VERSION >= 1.040) {
		$dialog = Gtk2::FileChooserDialog->new(
			$title, $self->{window}, 'open',
			'gtk-cancel' => 'cancel',
			'gtk-ok'     => 'ok'
		);
	}
	else { # old & ugly interface
		$dialog = Gtk2::FileSelection->new($title);
	}
	$dialog->set_icon($self->{window}->get_icon) if $self->{window};
	$dialog->set_action('select-folder');
	$dialog->set_filename(File::Spec->rel2abs($file)) if defined $file;
	$dialog->signal_connect('response', sub {
		$file = $_[1] eq 'ok' ? $dialog->get_filename : undef;
		$dialog->destroy;
	} );
	$dialog->run;

	return $file;
}

=item C<prompt_root_dialog>

=cut

sub prompt_root_dialog {
	my ($self, $page, $title, $prompt, $text, $icon) = @_;

	my $dialog = Gtk2::Dialog->new(
		"Open directory - Zim", undef,
	       	[qw/modal destroy-with-parent no-separator/],
		'gtk-cancel' => 'cancel',
		'gtk-open'   => 'ok',
	);
	$dialog->set_resizable(0);
	$dialog->set_border_width(5);
	$dialog->set_icon(
		Gtk2::Gdk::Pixbuf->new_from_file($self->{settings}{icon_file}) );
	$dialog->set_default_response('ok');
	
	$dialog->vbox->add(
		Gtk2::Image->new_from_file($self->{settings}{icon_file}) );

	$dialog->vbox->add( Gtk2::Label->new( << 'EOT' ) );
Welcom to Zim.

Zim needs a directory to store its wiki page.
Please enter a default directory below.

You can open other directories by giving them as commandline
a argument to zim(1).

If you want to change the default directory later you need to
edit the "default_root" option in the config file.
(TODO: add a nice dialog for changing config options)
EOT
	
	my $hbox  = Gtk2::HBox->new(0, 5);
	$hbox->set_border_width(5);
	$dialog->vbox->add($hbox);
	
	my $label = Gtk2::Label->new("Open directory: ");
	$hbox->add($label);
	
	my $entry = Gtk2::Entry->new();
	$entry->signal_connect(activate => sub { $dialog->response('ok') });
	$entry->set_text("$ENV{HOME}/zim_pages/");
	$hbox->add($entry);
	
	my $root;
	$dialog->show_all;
	if ($dialog->run eq 'ok') { $root = $entry->get_text }
	else                      { $root = undef            }
	$dialog->destroy;

	$root = undef unless $root =~ /\S/;
	return $root;
}

=item C<prompt_browser_dialog>

=cut

sub prompt_browser_dialog {
	my $self = shift;
	
	my $title = "Choose browser";
	my $dialog = Gtk2::Dialog->new(
		$title, $self->{window},
	       	[qw/modal destroy-with-parent no-separator/],
		'gtk-ok'     => 'ok',
		'gtk-cancel' => 'cancel',
	);
	$dialog->set_resizable(0);
	$dialog->set_border_width(5);
	$dialog->set_icon($self->{window}->get_icon);
	$dialog->set_default_response('ok');

	my $label = Gtk2::Label->new(<< "EOT");
Please enter a browser
to open external links.
EOT
	$dialog->vbox->add($label);
	
	my $entry = Gtk2::Entry->new();
	$entry->signal_connect(activate => sub { $dialog->response('ok') });
	$entry->set_text("firefox '\%s'");
	$dialog->vbox->add($entry);

	$dialog->show_all;
	my $browser = 
		($dialog->run eq 'ok') ? $entry->get_text : undef ;
	$dialog->destroy;
	$browser = undef unless $browser =~ /\S/;

	return $browser;
}

# "Search" dialog
#
# This dialog asks for a search query and then generates
# a page with search results.

#sub search_dialog {
#	my $dialog = Gtk2::Dialog->new(
#		# title, parent
#		'Search', $window,
#		# flags
#	       	[qw/modal destroy-with-parent no-separator/],
#		# buttons
#		'gtk-cancel' => 'cancel',
#		'gtk-find'   => 'ok',
#	);
#	$dialog->set_resizable(0);
#	$dialog->set_border_width(5);
#	$dialog->set_icon($window->get_icon);
#	#$dialog->set_default_response('ok');
#
#	my $table = Gtk2::Table->new(2, 2);
#	$table->set_border_width(5);
#	$table->set_row_spacings(5);
#	$table->set_col_spacings(5);
#	$dialog->vbox->add($table);
#	
#	my $label1 = Gtk2::Label->new('Find what:');
#	my $query_entry = Gtk2::Entry->new();
#	$query_entry->signal_connect(activate => sub { $dialog->response('ok') });
#	$table->attach_defaults($label1, 0,1, 0,1);
#	$table->attach_defaults($query_entry, 1,2, 0,1);
#	
#	$dialog->show_all;
#	if ($dialog->run eq 'ok') {
#		print "TODO: search algorithm that doesn't depend on grep(1)\n";
#		my $query = $query_entry->get_text;
#		return unless $query =~ /\S/;
#		my $page = Zim::Page->new($zim, 'Search Results');
#		$page->push_blocks(
#			['head1', 'Search Results'], "\n\n",
#			['B', '-- Search algorithm not yet stable --'],
#			"\n\nSearching for \"$query\"\n\n",
#		);
#		$query =~ s/\s/\\W+/g;
#		open GREP, "grep -ERi \"\\b$query\\b\" \"$ROOT\" |" or return;
#		while (<GREP>) {
#			/^(.+?):(.+)$/ or next;
#			my ($file, $text) = ($1, $2);
#			$file =~ s/^\Q$ROOT\E//;
#			next if $file =~ /^\./;
#			$page->push_blocks(
#				['link', $zim->pagename($file)],
#				"\n", ['pre', $text], "\n\n"
#			);
#		}
#		close GREP;
#		load_page($page);
#	}
#	$dialog->destroy;
#}

1;

__END__

=back

=head1 BUGS

Please mail the author if you find any bugs.

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

L<zim>(1),
L<Zim::Repository>

=cut

