package Zim::Components::TreeView;

use strict;
use vars '$AUTOLOAD';
use Gtk2;
use Gtk2::Gdk::Keysyms;

our $VERSION = 0.01;

=head1 NAME

Zim::Components::TreeView - Page index widgets

=head1 DESCRIPTION

This module contains the widgets to display the index 
of a zim repository as a TreeView.

=head1 METHODS

Undefined methods AUTOLOAD to the L<Gtk2::TreeView> widget.

=over 4

=cut

my ($k_left, $k_right) = @Gtk2::Gdk::Keysyms{qw/Left Right/};

sub new {
	my $class = shift;
	my $self = bless {@_}, $class;
	$self->init();
	return $self;
}

sub init { # called by new()
	my $self = shift;
	$self->{tree_iter} = {};

	my $scroll_window = Gtk2::ScrolledWindow->new();
	$scroll_window->set_policy('automatic', 'automatic');
	$scroll_window->set_shadow_type('in');
	$self->{scroll_window} = $scroll_window;

	my $tree_store = Gtk2::TreeStore->new('Glib::String', 'Glib::String');
	#$tree_store->set_sort_column_id(0, 'ascending');
	$self->{tree_store} = $tree_store;

	my $tree_view = Gtk2::TreeView->new($tree_store);
	my $renderer = Gtk2::CellRendererText->new();
	$renderer->signal_connect_swapped(edited => \&on_cell_edited, $self);
	$self->{renderer} = $renderer;
	my $page_column = Gtk2::TreeViewColumn->new_with_attributes(
        	'Pages', $renderer, 'text', 0);
	$tree_view->append_column($page_column);
	$tree_view->set_headers_visible(0);
	#$tree_view->set_reorderable(1);
	$tree_view->get_selection->set_mode('browse'); # one item selected at all times
	$tree_view->signal_connect(key_press_event      => \&on_key_press_event);
	$tree_view->signal_connect(button_release_event => \&on_button_release_event, $self);
	$tree_view->signal_connect(row_activated        => \&on_row_activated, $self);
	$tree_view->signal_connect(popup_menu           => sub {
			my ($path) = $tree_view->get_cursor;
			$self->popup_context_menu($path);
		} );
	$scroll_window->add($tree_view);
	$self->{tree_view} = $tree_view;

	my @buttons;
	push @buttons, $self->{app}->add_button(
        	'Expand All', 'Ctrl +', 'gtk-zoom-in', sub {$tree_view->expand_all});
	$self->{app}->add_key('Ctrl KP_Add', sub {$tree_view->expand_all});
	$self->{app}->add_key('Ctrl =', sub {$tree_view->expand_all});
	push @buttons, $self->{app}->add_button(
		'Collapse All', 'Ctrl -', 'gtk-zoom-out', sub {$tree_view->collapse_all});
	$self->{app}->add_key('Ctrl KP_Subtract', sub {$tree_view->collapse_all});
	$self->{buttons} = \@buttons;
	
	$self->{app}->signal_connect(page_loaded => sub {$self->select_page(pop)});
	$self->{app}->signal_connect(page_deleted => sub {$self->unlist_page(pop)});
	$self->{app}->signal_connect(page_created => sub {
		my $page = pop;
		$self->list_page($page);
		$self->select_page($page);
	} );
}

sub AUTOLOAD {
	my $self = shift;
	$AUTOLOAD =~ s/^.*:://;
	return if $AUTOLOAD eq 'DESTROY';
	return $self->{tree_view}->$AUTOLOAD(@_);
}
	
=item C<widget()>

Returns the root widget. This should be used to add the object to a container widget.
Also use this widget for things like show_all() and hide_all().

=cut

sub widget { return $_[0]->{scroll_window} }

=item C<hide()>

Hides the root widget and all asociated toolbar buttons.

=cut

sub hide { # FIXME belongs in base class
	$_[0]->widget->hide;
	$_->hide for @{$_[0]->{buttons}};
	$_[0]->{app}{toolbar}->queue_draw;
}

=item C<show()>

Shows the root widget and all asociated toolbar buttons.

=cut

sub show { # FIXME belongs in base class
	$_[0]->widget->show;
	$_->show for @{$_[0]->{buttons}};
	$_[0]->{app}{toolbar}->queue_draw;
}

sub on_key_press_event { # some extra keybindings for the treeview
	my ($tree_view, $event) = @_;
	my $val = $event->keyval();
	my $key = chr($val);
	#print "got key $key ($val)\n";
	if ($val == $k_left) { # Left arrow
		my ($path) = $tree_view->get_selection->get_selected_rows;
		$tree_view->collapse_row($path) if defined $path;
	}
	elsif ($val == $k_right) { # Rigth arrow
		my ($path) = $tree_view->get_selection->get_selected_rows;
		$tree_view->expand_row($path, 0) if defined $path;
	}
	else { return 0 }
	return 1;
}

sub on_button_release_event {
	my ($tree_view, $event, $self) = @_;
	return 0 if $event->type ne 'button-release';
	
	my ($x, $y) = $event->get_coords;
	my ($path, $column) = $tree_view->get_path_at_pos($x, $y);
	return 0 unless $path;
	
	$self->activate_if_selected($path, $column)
		if $event->button == 1;
	$self->popup_context_menu($path, $event)
		if $event->button == 3;
	return 0;
}

sub on_row_activated { # load a page when clicked or entered
	my $self = pop;
	my ($tree_view, $path) = @_;
	my $page = $self->page_from_path($path);
	$self->{app}->load_page($page)
		unless $page eq $self->{app}{page}->realname;
}

=item C<page_from_path(PATH)>

Returns the page name corresponding to a certain TreePath.

=cut

sub page_from_path {
	my ($self, $path) = @_;
	my $model = $self->{tree_view}->get_model;
	my $iter = $model->get_iter($path);
	my ($base, $real) = $model->get($iter);
	return wantarray ? ($real, $base) : $real;
}

=item C<activate_if_selected(PATH, COLUMN)>

Emit the "row_activated" signal if PATH was already selected.
This method is used for the single-click navigation of the TreeView.

COLUMN is an optional argument.

=cut

sub activate_if_selected { # single-click navigation
	my ($self, $path, $column) = @_;
	my $tree_view = $self->{tree_view};

	my ($selected) = $tree_view->get_selection->get_selected_rows;
	return 0 unless $selected and 0 == ($path->compare($selected));

	my $page = $self->page_from_path($path);
	$column ||= $tree_view->get_column(0);
	$tree_view->row_activated($path, $column)
		unless $page eq $self->{app}{page}->realname;
}

=item C<popup_context_menu(PATH, EVENT)>

Show the context menu for TreePath PATH.

EVENT is an optional argument.

=cut

sub popup_context_menu {
	my ($self, $path, $event) = @_;
	my $tree_view = $self->{tree_view};
	my $page = $self->page_from_path($path);
	
	my $menu = Gtk2::Menu->new();
	my $item1 = Gtk2::MenuItem->new('Move page');
	$item1->signal_connect(
		activate => sub { $self->{app}->move_page($page, undef, 1) });
	$menu->append($item1);
	my $item2 = Gtk2::MenuItem->new('Delete page');
	$item2->signal_connect(
		activate => sub { $self->{app}->delete_page($page) });
	$menu->append($item2);
	$menu->show_all;

	my ($button, $time) = $event
		? ($event->button, $event->time)
		: (0, 0) ;
	#my $pos_func = $event ? undef : sub {
	#};
	$menu->popup(undef, undef, undef, undef, $button, $time);
	return 1; # needed for popup_menu signal (shift-F10), but why ?
}

# load_index()
#
# Called the first time the pane is showed to fill the index tree.
#
# TODO this should be caching algo

sub load_index {
	my $self = shift;
	return if $self->{_loaded};
	$self->{tree_store}->clear; # just to be sure
	$self->{tree_iter} = {};
	eval { $self->list_pages('') };
	$self->select_page($self->{app}{page}->realname);
	$self->{app}->message("Index loaded");
	$self->{_loaded} = 1;
}

# list_pages(namespace)
#
# Wrapper around Zim->list_pages that fills the index tree.

sub list_pages {
	my ($self, $namespace) = @_;
	#print "> $namespace\n";
	for ($self->{app}{repository}->list_pages($namespace)) {
		#print "\t$_\n";
		my $is_dir = ($_ =~ s/:$//);
		$self->list_page($namespace.':'.$_);
		$self->list_pages($namespace.':'.$_) if $is_dir; # recurs
	}
	Gtk2->main_iteration while (Gtk2->events_pending);
}

# list_page(realname)
#
# Adds a page to the index tree.

sub list_page {
	my ($self, $name) = @_;
	return unless defined $name;
	
	my ($tree_store, $tree_iter) = @$self{qw/tree_store tree_iter/};
	$name =~ s/^:+//;
	my ($page, @parts) = (undef, split /:+/, $name);
	my $iter;
	while (@parts) {
		$page = $page.':'.$parts[0];
		unless (exists $$tree_iter{$page}) {
			$$tree_iter{$page} = $tree_store->append($iter);
			$tree_store->set_value($$tree_iter{$page},
				0 => $parts[0],
				1 => $page      );
		}
		$iter = $$tree_iter{$page};
		shift @parts;
	}

	return 1;
}

# unlist_page(realname)
#
# Removes a page from the index unless it has children.

sub unlist_page {
	my ($self, $page) = @_;
	my ($tree_store, $tree_iter) = @$self{qw/tree_store tree_iter/};
	return unless defined $page;
	$page =~ s/^:*/:/;
	return unless exists $$tree_iter{$page};
	return if $tree_store->iter_children($$tree_iter{$page}); # has children
	$tree_store->remove($$tree_iter{$page});
	delete $$tree_iter{$page};
}

=item C<select_page(PAGE)>

Update the TreeView to display and select a certain page.

=cut

sub select_page {
	my ($self, $page) = @_;
	my ($tree_view, $tree_store, $tree_iter) = @$self{qw/tree_view tree_store tree_iter/};
	$page =~ s/^:*/:/;
	return unless exists $$tree_iter{$page};
	
	my $path = $tree_store->get_path($$tree_iter{$page});
	$tree_view->expand_to_path($path);
	$tree_view->get_selection->select_path($path);

	$tree_view->set_cursor($path);
	$tree_view->scroll_to_cell($path, undef, 0); # vert. scroll
}

1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

L<Zim>

=cut

