package Gtk2::Ex::HyperTextBuffer;

our $VERSION = 0.01;

use strict;
use Gtk2;
use Gtk2::Pango; # pango constants
use Glib::Object::Subclass Gtk2::TextBuffer:: ;
use File::Spec;  # for locating images

=head1 NAME

HyperTextBuffer - A TextBuffer for the HyperTextView widget

=head1 DESCRIPTION

This module is a subclass of L<Gtk2::TextBuffer> intended for use with 
L<Gtk2::Ex::HyperTextView>. It has a high level interface that uses parse trees
instead of plain text. It also keeps an undo-/redo-stack.

=head1 HIERARCHY

  Glib::Object
  +----Gtk2::TextBuffer
       +----Gtk2::Ex::HyperTextBuffer


=head1 TAGS

Tags are used in L<Gtk2::TextBuffer> to render certain parts of the text buffer with
a certain colours or certain decorations. In B<Gtk2::Ex::HyperTextBuffer> they also
represent a node in the document parse tree. See L<Gtk2::TextTag> for more details.

The following tags are pre-defined:

=over 4

=item head1

=item head2

=item head3

=item head4

=item head5

=item bold

=item italic

=item underline

=item verbatim

=item Verbatim

Like 'verbatim' but intended for multiline blocks.

=item link

For links anonymous TextTags are used, but the visual properties are defined the
same way as for other tags.

=cut

# It seems to me that 10 * PANGO_SCALE is the normal font size
# TODO make tag styles configable
our %TAGS = (
	bold      => [weight => PANGO_WEIGHT_BOLD],
	italic    => [style => 'italic'],
	underline => [underline => 'single'],
	verbatim  => [family => 'monospace', wrap_mode => 'none'],
	Verbatim  => [family => 'monospace', wrap_mode => 'none', indent => 40],
	link      => [foreground => 'blue', underline => 'single'],
	head1     => [weight => PANGO_WEIGHT_BOLD, size => 10*1.2**3 * PANGO_SCALE ],
	head2     => [weight => PANGO_WEIGHT_BOLD, size => 10*1.2**2 * PANGO_SCALE ],
	head3     => [weight => PANGO_WEIGHT_BOLD, size => 12 * PANGO_SCALE        ],
	head4     => [weight => PANGO_WEIGHT_BOLD, size => 10 * PANGO_SCALE        ],
	head5     => [style  => 'italic', weight => PANGO_WEIGHT_BOLD, size => 10 * PANGO_SCALE],
);

our @_tags = grep {$_ ne 'link'} keys %TAGS;

=back

=head1 METHODS

=over 4

=item C<< new(PROPERTY => VALUE, ..) >>

Simple constructor.

=cut

#sub new { # FIXME annoying warning about tagtable
#	my $class = shift;
#	if    (@_ == 1) { unshift @_, 'tag_table' }
#	elsif (@_ == 0) { unshift @_, 'tag_table', Gtk2::TextTagTable->new }
#	&Glib::Object::new($class, @_);
#}

sub INIT_INSTANCE {
	my $self = shift;
	$self->{edit_mode_tags} = [];
	$self->{image_marks} = [];
	$self->create_tag($_ => @{$TAGS{$_}}) for @_tags;
}

=item C<tag_aliases(\%aliases)>

Set or get the hash with tag aliases. This hash is used when
the parse tree is loaded or dumped.

=cut

sub tag_aliases {
	$_[0]->{aliases} = $_[1] if @_ > 1;
	return $_[0]->{aliases};
}

=item C<set_parse_tree(TREE)>

Load the buffer with contents from a parse tree. The tree format is based on
the format as used by L<Pod::Simple>. You can use the C<tag_aliases> hash to
tell the parser that "B" actually means 'bold' etc.

For example:

	['Document', {},
		['head1', {}, 'Test page'],
		"\n\n",
		"This is a test page showing some syntax:\n\n",
		['bold', {}, 'Bold text'], "\n",
		['italic', {}, 'Italic text'], "\n",
		['bold', {},
			['italic', {}, 'Bold and Italic']
		], "\n",
	]

=cut

sub set_parse_tree {
	my ($self, $tree) = @_;

	my ($start, $end) = $self->get_bounds;
	$self->{edit_mode_tags} = [];
	$self->{image_marks} = [];
	$self->delete($start, $end); # clear buffer
	
	#use Data::Dumper; print Dumper @$tree[0,1];
	$self->insert_blocks_at_cursor($tree);

	$self->set_modified(0);
}

=item C<insert_blocks(ITER, BLOCK, BLOCK, ..)>

Insert one or more partial parse trees in the buffer at ITER.

=cut

sub insert_blocks {
	my $self = shift;
	my $iter = shift;
	$self->place_cursor($iter);
	$self->insert_blocks_at_curesor(@_);
}

=item C<inset_blocks_at_cursor(BLOCK, BLOCK, ...)>

Like C<insert_blocks()> but inserts at current cursor position.

=cut

my $_Verbatim = 0;

sub insert_blocks_at_cursor {
	my ($self, @blocks) = @_;
	for (@blocks) {
		unless (ref $_) {
			$_ =~ s/^(\s*)\*(\s+)/$1\x{2022}$2/mg unless $_Verbatim; # FIXME doesn't belong in this module !
			if (@{$self->{edit_mode_tags}}) {
				my $iter = $self->get_iter_at_mark( $self->get_insert );
				$self->insert_with_tags(
					$iter, $_, @{$self->{edit_mode_tags}} );
			}
			else { $self->insert_at_cursor($_) }
		}
		else {
			my ($type, $meta) = splice @$_, 0, 2;
			$type = $self->{aliases}{$type} if exists $self->{aliases}{$type};
			my $tag;
			if ($type eq 'link') {
				my $bit = (@$_ == 1 and $$_[0] eq $$meta{to});
				$tag = $self->create_link_tag([$bit, $$meta{to}]);
			}
			else { $tag = $self->get_tag_table->lookup($type) }
			
			if ($tag) {
				$_Verbatim++ if $type eq 'Verbatim'; # FIXME
				push @{$self->{edit_mode_tags}}, $tag;
				$self->insert_blocks_at_cursor(@$_); # recurs
				pop @{$self->{edit_mode_tags}};
				$_Verbatim-- if $type eq 'Verbatim'; # FIXME
			}
			elsif ($type eq 'image') { # FIXME make this a seperate routine
				my $file = $$meta{src};
				$file = File::Spec->rel2abs($file, $self->{base})
					if defined $self->{base};
				#print "file: $$meta{src}, base: $self->{base} => $file\n";
				my $image = Gtk2::Gdk::Pixbuf->new_from_file($file)
					if -f $file and -r _;
				$image = Gtk2::Image->new->render_icon(
						'gtk-missing-image', 'button' )
					unless $image;
				die "BUG: could not insert image: $$meta{src}\n" unless $image;
				$image->{image_src} = $$meta{src};
				my $iter = $self->get_iter_at_mark( $self->get_insert );
				$self->insert_pixbuf($iter, $image);
				push @{$self->{image_marks}}, 
					$self->create_mark(undef, $iter, 1) ;
				#$self->{image_marks}[-1]->set_visible(1);
			}
			elsif (grep {$_ eq $type} qw/Document Para/) {
				$self->{base} = $$meta{base} if $$meta{base};
				$self->insert_blocks_at_cursor(@$_); # recurs
			}
			else { die "Unknown tag in parse tree: $type" }
		}
	}
}

=item C<get_parse_tree()>

Returns a parse tree based on the buffer contents.

=cut

sub get_parse_tree { # FIXME this can be optimized for memory usage
	my $self = shift;
	my $tree = ['Document', {},];
	my @stack = ($tree);
	my @positions = $self->_get_positions;
	my $text = $self->get_slice($self->get_bounds, 1);
	my $previous = 0;
	while (@positions) {
		my ($pos, $open, $head) = @{ shift @positions };
		my $slice = substr $text, $previous, ($pos - $previous);
		$previous = $pos;
		$slice =~ s/^(\s*)\x{2022}(\s+)/$1*$2/mg; # FIXME totally wrong level
		push @{$stack[-1]}, $slice if length $slice;
		if (! defined $open) { # inline objects
			push @{$stack[-1]}, $head;
			$previous++; # skip over \x{fffc} character
		}
		elsif ($open) { # open tag
			my $node = [$$head[0], { %{$$head[1]} } ]; # force copy
			push @{$stack[-1]}, $node;
			push @stack, $node;
		}
		else { # close tag
			# warning: no consistency check is being done here
			pop @stack;
		}
	}
	my $slice = substr $text, $previous;
	$slice =~ s/^(\s*)\x{2022}(\s+)/$1*$2/mg; # FIXME totally wrong level
	push @{$stack[-1]}, $slice if @stack and length $slice;
	#use Data::Dumper; print STDERR Dumper $tree;
	return $tree;
}

sub _get_positions { # builds an index of all tag toggles
	my $self = shift;
	my @tags;
	$self->get_tag_table->foreach(sub { push @tags, @_ });
	my @positions;

	# Find tag regions
	for my $tag (@tags) {
		my $start = $self->get_start_iter;
		my $type = $tag->{is_link} ? 'link' : $tag->get_property('name');
		my $meta = ($type eq 'link') ? {to => $tag->{link_data}[1]} : {};
		$type = $self->{aliases}{$type} if exists $self->{aliases}{$type};
		my $header = [$type, $meta];
		while ($start->begins_tag($tag) or $start->forward_to_tag_toggle($tag)) {
			push @positions, [$start->get_offset, 1, $header];
			$start->forward_to_tag_toggle($tag) or last; # find end
			push @positions, [$start->get_offset, 0, $header];
			last if $start->is_end; # prevent infinite loop
		}
	}
	
	# Find inline objects
	my @done;
	for (@{$self->{image_marks}}) {
		my $iter = $self->get_iter_at_mark($_);
		$iter->backward_char;
		my $off = $iter->get_offset;
		next if grep {$_ == $off} @done;
		push @done, $off;

		my $image = $iter->get_pixbuf;
		next unless $image;
		push @positions, [$off, undef, ['image', {src => $image->{image_src}}]];
	}
	
	return sort {$$a[0] <=>$$b[0]} @positions;
}

=item C<create_link_tag(LINK_DATA)>

Returns an anonymous L<Gtk2::TextIter> that holds LINK_DATA.

LINK_DATA should be an array reference. The first item in this array
should be a boolean descibing whether the link matches the text it is applied
to. If this boolean is set to tru the second element in the array will be update
to hold the text if the buffer is edited.

Every link tag should only be applied to one text region.

=cut

sub create_link_tag {
	my($self, $data) = @_;
	
	my $tag = $self->create_tag(undef, @{$TAGS{link}});
	$tag->{is_link}   = 1;
	$tag->{link_data} = $data;

	return $tag;
}

sub select_word {
}

sub select_line {
}

sub apply_tag_to_selection {
	my ($self, $tag) = @_;
	
}

sub replace_selection {
}

sub set_edit_mode_tags {
	my $self = shift;
	my $table = $self->get_tag_table;
	$self->{edit_mode_tags} =
		[ map { ref($_) ? $_ : $table->lookup($_) } @$_ ];
}

sub get_edit_mode_tags { @{$_[0]->{edit_mode_tags}} }

sub toggle_edit_mode_tag {
	my ($self, $tag) = @_;
	$tag = $self->get_tag_table->lookup($tag) unless ref $tag;
	if (grep {$_ eq $tag} @{$self->{edit_mode_tags}}) {
		@{$self->{edit_mode_tags}} = grep {$_ ne $tag} @{$self->{edit_mode_tags}};
	}
	else {
		push @{$self->{edit_mode_tags}}, $tag;
	}
}

sub undo {
}

sub redo {
}

1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

=cut

