package Zim::Repository;

use strict;
use POSIX qw(strftime);
use File::Spec;
use File::MimeInfo;
use Zim::Page;

our $VERSION = 0.06;

my $is_case_tolerant = File::Spec->case_tolerant();

=head1 NAME

Zim::Repository - A wiki repository

=head1 SYNOPSIS

	use Zim::Repository;
	
	my $repo = Zim::Repository->new($root_dir);
	my $page = $repo->load_page('Home');

=head1 DESCRIPTION

This class defines the public interface to the document repository
as used by L<zim>(1). By default it represents filesystem based repositories,
but it can be overloaded for more abstract models.

=head1 METHODS

=head2 Public Methods

The following methods can be used by the GUI.

=over 4

=item new(NAMESPACE, DIR)

Simple constructor. DIR is the root directory of the repository.
NAMESPACE is the namespace that maps to that directory. For the 
toplevel repository leave NAMESPACE empty.

=cut

sub new {
	my ($class, $namespace, $dir, $format) = @_;
	$dir = File::Spec->canonpath($dir);
	my $self = bless {
		namespace => $namespace || '',
		root      => $dir,
		format    => $format || 'wiki',
	}, $class;
	
	$self->{formatter} =  $self->load_format( $self->{format} );
	$self->{ext} = $self->{formatter}->extension;
	
	# load plugins (nested repositories
	my $conf = File::Spec->catfile($dir, '.zim.config');
	if (-f $conf and -r _) {
		$self->{conffile} = $conf;
		for (split /\n/, Zim::Page::File::_read_file($conf)) {
			/^:(\S+?)=(\S+)/ or next;
			my ($subspace, $mod) = ($1, $2);
			$subspace =~ s/:+$//;
			eval "use $mod;";
			die if $@;
			my $dir = $self->filename($subspace, 'DIR');
			$self->{plugins}{$subspace} = $mod->new($subspace, $dir);
			#print "plug: $mod for $namespace\n";
		}
	}

	return $self;
}

sub load_format {
	my $self = shift;
	my $wanted = lc( shift(@_) ).'.pm';
	my $class;
	for (@INC) {
		# FIXME object refs in @INC
		my $dir = File::Spec->catdir($_, qw/Zim Formats/);
		next unless -d $dir and -r _;
		opendir DIR, $dir or next;
		($class) = grep {lc($_) eq $wanted} readdir DIR;
		closedir DIR;
		last if $class;
	}
	$class =~ s/.pm$//;
	die "Could not find a module for format: $self->{format}\n" unless $class;
	$class = "Zim::Formats::$class";
	eval "use $class;";
	die if $@;
	return $class;
}

=item list_pages(NAMESPACE)

Lists pages in NAMESPACE. Sub-namespaces have a trailing ':'
in the listing.

=cut

sub list_pages {
	my ($self, $namespace) = @_;
	
	$namespace =~ s/:*$/:/;
	if (my $plug = $self->belongs_to_plugin($namespace)) {
		return $self->{plugins}{$plug}->list_pages($namespace);
	}

	my $dir = $self->filename($namespace, 'DIR');
	#print "list pages $namespace => $dir\n";
	return () unless -d $dir;
	opendir DIR, $dir or die "Could not list dir $dir\n";
	my @items = grep {defined $_} map {
		my $item = File::Spec->catfile($dir, $_);
		if (-d $item) { s/(_files)?$/:/ }
		else {
			if (mimetype($item) =~ /^text/) { s/\.$$self{ext}$// }
			else { $_ = undef }
		}
		$_;
	} grep {! /^\./} readdir DIR;
	closedir DIR;
	@items = map {lc $_} @items if $is_case_tolerant;
	return sort {lc($a) cmp lc($b)} @items;
}

sub belongs_to_plugin {
        my ($self, $page) = @_;
	$page =~ s/^:*$self->{namespace}:*//;
        my ($plug) = grep {$page =~ /^$_:/} keys %{$self->{plugins}};
        return $plug;
}


=item page_exists(PAGE_NAME)

Returns TRUE when there is an existing page under this name.

=cut

sub page_exists {
	my ($self, $name) = @_;
	
	if (my $plug = $self->belongs_to_plugin($name)) {
		return $self->{plugins}{$plug}->page_exists($name);
	}
	
	return -f $self->filename($name);
}

=item load_page(PAGE_NAME)

Returns an object of the type L<Zim::Page>.

=cut

sub load_page {
	my ($self, $name) = @_;

	if (my $plug = $self->belongs_to_plugin($name)) {
		return $self->{plugins}{$plug}->load_page($name);
	}
	
	my $page = Zim::Page::File->new($self, $name);
	#print "page $page => file ".$page->filename."\n";
	if (-f $page->filename) { $page->read_file }
	else {
		$page->parse_tree( $self->_template($page) );
		$page->{status} = 'new';
	}

	return $page;
}

=item save_page(PAGE)

Saves the data for the page object PAGE.

=cut

sub save_page {
	my ($self, $page) = @_;

	if ($page->{zim} ne $self) {
		# This one isn't ours !
		$page = Zim::Page::File->new_from_object($self, $page);
	}
	
	if ($page->isa('Zim::Page::File')) {
		$page->write_file;
		$page->{status} = '';
		return 1;
	}
	else {
		warn "Can't save page ".$page->name."\n";
		return 0;
	}
}

=item move_page(SOURCE, TARGET)

Moves a page from SOURCE to TARGET.
SOURCE and TARGET can be either page names or page objects.

=cut

sub move_page {
	my ($self, $source, $target) = @_;

	$source = $self->load_page($source)
		unless ref $source;
	$target = $self->load_page($target)
		unless ref $target;
	
	$source->rewind;
	$target->clear;
	while (defined ($_ = $source->read_block)) {
		$target->push_blocks($_);
	}

	$self->delete_page($source);
	$self->save_page($target);
}

=item delete_page(PAGE)

Deletes a page.
PAGE can be either a page name or a page object.

=cut

sub delete_page {
	my ($self, $page) = @_;

	$page = Zim::Page::File->new($self, $page)
		unless ref $page;

	if ($page->isa('Zim::Page::File')) {
		$page->remove_file
			if -f $page->filename;
		$page->{status} = 'deleted';
	}
	else {
		warn "Can't delete page ".$page->pagename."\n";
	}
}

sub _template {
	# FIXME make template configurable
	my ($self, $page) = @_;
	my $name = $page->name;
	$name =~ /([^:]+):*$/;
	my $title = $1;
	$title = ucfirst($title) unless $title =~ /[A-Z]/;
	return $self->{formatter}->load_tree(
		"====== $title ======\n".
		'Created '.strftime('%A %d/%m/%Y %H:%M', localtime)."\n\n",
		$page
	);
}

=back

=head2 Private Methods

These methods are specific to filesystem based repositories
and should therefor never be called directly from the GUI.

=over 4

=item filename(NAME, DIR)

Gives the filename corresponding to a page name.
NAME is a string. (This lookup is more difficult then it might seem.)

DIR is a boolean that can be set to tell that NAME is expected to
be a directory. This behaviour is also triggered if the page name
ends with a ':'.

In list context filename, directory and real page name are returned.

=cut

sub filename { # FIXME check for optimizations
	my ($self, $page, $is_dir) = @_;

	$page =~ s/^:*$self->{namespace}:*//i;
	$is_dir++ if $page =~ s/:+$//;
	
	$page = lc($page) if $is_case_tolerant;
	
	my @dirs = split /:+/, $page;
	my $basename = pop @dirs unless $is_dir;

	# check the dirs
	my @realname;
	my $dir = File::Spec->catdir($self->{root}, @dirs);
	if (-d $dir) { @realname = @dirs }
	else {
		my $path = $self->{root};
		for (0 .. $#dirs) {
			my $casepath = File::Spec->catdir($path, $dirs[$_]);
			if (-e $casepath) { $path = $casepath }
			else {
				$dirs[$_] = lc($dirs[$_]);
				$path = File::Spec->catdir($path, $dirs[$_]);
			}
			push @realname, $dirs[$_];
			$dirs[$_] .= '_files' if -f $path; # existing files
		}
		$dir = File::Spec->catdir($self->{root}, @dirs);
	}

	if ($is_dir) {
		return $dir unless wantarray;
		my $realname = join ':', $self->{namespace}, @realname;
		return $dir, $dir, $realname;
	}

	# check which filename to use
	my $file;
	for ($basename, lc($basename)) {
		$basename = $_;
		$file = File::Spec->catfile($dir, $_);
		last if -f $file;
		$file = "$file.$$self{ext}";
		last if -f $file;
	}
	push @realname, $basename;
	#print "$page => file: $file, realname: @realname\n";

	return $file unless wantarray;
	my $realname = join ':', $self->{namespace}, @realname;
	return $file, $dir, $realname;
}

=item pagename(FILE)

Returns the page name corresponding to FILE. FILE does not actually
need to exist and can be a directory as well as a file.

=cut

sub pagename {
	my ($self, $file) = @_;
	# $file =~ s/^\Q$self->{root}\E//;
	my @parts = map  {s/_files$//; $_} File::Spec->splitdir($file);
	$parts[-1] =~ s/\.$$self{ext}$//;
	return join ':', $self->{namespace}, @parts;
}

1;

__END__

=back

=head1 BUGS

Please mail the author if you find any bugs.

=head1 AUTHOR

Jaap Karssenberg || Pardus [Larus] E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

L<Zim>, L<Zim::Page>

=cut
