package Zim::Page;

use strict;

our $VERSION = 0.06;

=head1 NAME

Zim::Page - Page object for Zim

=head1 SYNOPSIS

FIXME simple code example

=head1 DESCRIPTION

This module contains the base class for page objects as
used by L<Zim::Repository>. 

Also included is B<Zim::Page::File> which can be used as a
base class for pages that corespond to files.

=head1 METHODS

=head2 Zim::Page

This class defines the public interface for page objects.
The GUI should only use methods from this class.

=over 4

=item new(PARENT, NAME)

Simple constructor. PARENT should be a repository object of class Zim.
NAME is the page name for this object.

=cut

sub new {
	my ($class, $parent, $name) = @_;
	$name =~ s/^:+|:+$//g;
	die "Can't create $class object without a page name!\n"
		unless length $name;
	my $self = bless {
		zim => $parent,
		pagename => $name,
		parse_tree => ['Document', {}],
		status => '',
	}, $class ;
	$self->init;
	return $self;
}

sub init { } # to be overloaded

=item new_from_object(PARENT, PAGE)

Constructor that clones another page object.
This other page object can belong to another repository.
This method is for example used to export data from one format to another.

Note that both page objects do not necessarily need to be of the same class,
the Zim::Page api should therefor be used to clone the object.

=cut

sub new_from_object {
	my ($class, $parent, $obj) = @_;
	my $self = bless {
		zim => $parent,
		pagename => $obj->name,
		realname => $obj->realname,
		parse_tree => $obj->parse_tree,
		status => $obj->status,
	}, $class ;
	$self->init;
	return $self;
}

=item name( )

Returns the name of the page.

=cut

sub name { $_[0]->{pagename} }

=item realname( )

Returns a name for the current page that can be used for comparing
page objects. By default the real name is the same as the page name,
but for example for file based pages where a case insensitive lookup
is done these could differ.

=cut

sub realname { $_[0]->{pagename} }

=item basename( )

Returns the last part of the page name.

=cut

sub basename {
	my $name = $_[0]->{pagename};
	$name =~ s/^(.*:+)//;
	return $name;
}

=item namespace( )

Returns the first part of the page name.
Including a trailing ':'.

=cut

sub namespace {
	my $name = $_[0]->{pagename};
	$name =~ /^:*(.*:+)/;
	my $namespace = $1 || '';
	return wantarray ? split(/:+/, $namespace) : ":$namespace";
}

=item status(STRING)

Set or get a status string for this page.
Typical status strings are 'new' and 'deleted'.

=cut

sub status {
	my ($self, $status) = @_;
	$self->{status} = $status if defined $status;
	return $self->{status};
}

=item resolv_link(LINK)

Returns a page name for a page linked from this page.
This method allows you to have relative links.

=cut

sub resolve_link {
	my ($self, $link) = @_;
	$link =~ s/[^\w\.\:\-]/_/g;
	$link = ($link =~ s/^\.//) ? $self->name.':'.$link : # sub namespace
		($link !~ /:/) ? $link = $self->namespace.$link : # relative link
		$link ;
	return $link;
}

=item parse_tree( )

Get or set the parse tree.

=cut

sub parse_tree {
	$_[0]->{parse_tree} = $_[1] if @_ > 1;
	return $_[0]->{parse_tree};
}

=item push_blocks(BLOCK, BLOCK, ...)

Append more data to this page.

=cut

sub push_blocks {
	my $self = shift;
	push @{$self->{parse_tree}}, @_;
}

=item unshift_blocks(BLOCK, BLOCK, ...)

Prepend a data to this page.

=cut

sub unshift_blocks {
	# splice between header and content
	my $self = shift;
	splice @{$self->{parse_tree}}, 2, 0, @_;
}

=item clean( )

Logical cleanup of the data tree.

=cut

sub clean {
#	my $self = shift;
#	for my $i (1 .. $#{$self->{data}}) {
#		if (ref $self->{data}[$i]) {
#			if ($self->{data}[$i][0] eq 'normal') {
#				my (undef, @text) = @{$self->{data}[$i]};
#				$self->{data}[$i] = join '', @text;
#			}
#			else { next }
#		}
#		
#		next if ref $self->{data}[$i-1];
#		$self->{data}[$i] = $self->{data}[$i-1] . $self->{data}[$i];
#		$self->{data}[$i-1] = undef;
#	}
#	@{$self->{data}} = grep defined($_), @{$self->{data}};
}

=item clear( )

Clears all data from this page and initialises an empty parse tree.

=cut

sub clear { $_[0]->{parse_tree} = ['Document', {}] }

=item save( )

=item delete( )

=item move( )

The methods C<save()>, C<delete()> and C<move()> are aliases
for the methods C<save_page()>, C<delete_page()> and C<move_page()>
in L<Zim::Repository>.

=cut

sub save { $_[0]->{zim}->save_page(@_) }

sub delete { $_[0]->{zim}->delete_page(@_) }

sub move { $_[0]->{zim}->move_page(@_) }

package Zim::Page::File;

=back

=head2 Zim::Page::File

This class defines a private interface for page objects that
are stored in plain text files. It is used by the class Zim.
These methods should never be called directly from the GUI.

This class derives from B<Zim::Page>.

=over 4

=cut

our @ISA = 'Zim::Page';

sub init {
	my $self = shift;
	if (defined $self->{realname}) { # new_from_object
		# FIXME filename lookup belongs in Repository object
		my $name = $self->{realname};
		$name =~ s/^:+|:+$//g;
		my @parts = split /:+/, "$name.$$self{zim}{ext}";
		$self->{filename} = File::Spec->catfile($self->{zim}{root}, @parts);
		pop @parts;
		$self->{dirname} = File::Spec->catdir($self->{zim}{root}, @parts);
	}
	else {
		@{$self}{qw/filename dirname realname/} =
	       		$self->{zim}->filename($self->{pagename});
	}
}

sub realname { return $_[0]->{realname} }

=item filename( )

Returns the filename for the current object.

=cut

sub filename {
	return wantarray 
		? ( @{$_[0]}{qw/filename dirname/} )
		: $_[0]->{filename} ;
}

=item read_file( )

Reads content from the file corresponding to this page and parses it.

=cut

sub read_file {
	my $self = shift;
	return unless -f $self->{filename};
	$self->{parse_tree} =
		$self->{zim}->{formatter}->load_tree(
			_read_file($self->{filename}), $self );
}

=item write_file( )

Calls Zim::Formatter to convert the data structure to plain text and
writes this to the corresponding text file.

=cut

sub write_file {
	my $self = shift;
	_mkdir( $self->{dirname} ) unless -d $self->{dirname};
	_write_file( $self->{filename},
		$self->{zim}->{formatter}->save_tree($self->{parse_tree}, $self)
	);
}

=item remove_file( )

Removes the current file (effectively deleting the page) and
also the current directory if it is empty.

=cut

sub remove_file {
	my $self = shift;
	unlink $self->{filename}
		or die "Could not remove file $self->{filename}\n";
	_rmdir($self->{dirname});
}

=back

The following functions are used by the methods. These are not
methods themselfs and should be called without object reference.

=over 4

=item _read_file(FILE)

Returns the contents of a file.

=cut

sub _read_file {
	my $file = shift;
	open TXT, $file or die "Could not read $file\n";
	binmode TXT, ':utf8' unless $] < 5.008;
	my $text = join '', <TXT>;
	close TXT;
	return $text;
}

=item _write_file(FILE, TEXT)

Writes file contents.

=cut

sub _write_file {
	my $file = shift;
	open TXT, ">$file" or die "Could not write $file\n";
	binmode TXT, ':utf8' unless $] < 5.008;
	print TXT @_;
	close TXT or die "Could not write $file\n";
}

=item _mkdir(DIR)

Creates the complete path for DIR.

=cut

sub _mkdir {
	my ($vol, $dirs) = File::Spec->splitpath(shift(@_), 'NO_FILE');
	my @dirs = File::Spec->splitdir($dirs);
	my $path = File::Spec->catpath($vol, shift @dirs);
	mkdir $path or die "Could not create dir $path\n"
		if length $path and ! -d $path;
	while (@dirs) {
		$path = File::Spec->catdir($path, shift @dirs);
		mkdir $path or die "Could not create dir $path\n"
			unless -d $path;
	}
}

=item _rmdir(DIR)

Removes DIR and parent directories if empty.

=cut

sub _rmdir {
	my ($vol, $dirs) = File::Spec->splitpath(shift(@_), 'NO_FILE');
	my @dirs = File::Spec->splitdir($dirs);
	my $path = File::Spec->catpath($vol, shift @dirs);
	rmdir $path; # fails when not empty
	while (@dirs) {
		$path = File::Spec->catdir($path, shift @dirs);
		last unless -d $path;
		rmdir $path; # fails when not empty
	}
}

1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

L<Zim>,
L<Zim::Repository>

=cut

