package Zim::History;

use strict;
use Storable qw/nstore retrieve/;

our $VERSION = 0.06;

=head1 NAME

Zim::History - History object for zim

=head1 SYNOPSIS

	use Zim::History;
	
	my $hist = Zim::History->new($HIST_FILE, $PAGEVIEW, 20);
	my $page_record = $hist->current;

=head1 DESCRIPTION

This object manages zim's page history.

=head1 METHODS

=over 4

=item C<new(FILE, PAGEVIEW, MAX)>

Constructor. Takes the name of the cache file as an argument
and reads this cache file. The second argument is the pageview
object. When a new record is made for the current page the C<get_state()> method is
called on this object to get state information that needs to be saved.
Third argument is the maximum number of items in the history.

=cut

sub new {
	my ($class, $file, $view, $max) = @_;
	my $self = bless {
		file     => $file,
		PageView => $view,
		max      => $max,
		back     => [],
		forw     => [],
	}, $class;
	$self->read;
	return $self;
}

=item C<read>

Read the cache file.

=cut

sub read {
	my $self = shift;
	return unless -f $self->{file} and -r _;
	my @cache = @{ retrieve($self->{file}) };
	unless (ref $cache[0]) { # early versions did not have the version number
		my $version = shift @cache;
		return unless $version == $VERSION;
	}
	@{$self}{qw/back current forw/} = @cache;
}

=item C<write>

Write the cache file.

=cut

sub write {
	my $self = shift;
	$self->current; # force conversion from object to record
	nstore([$VERSION, @{$self}{qw/back current forw/}], $self->{file});
}

=item C<current()>

Returns the history record for the current page.

=cut

sub current {
	my ($self) = @_;
	return undef unless $self->{current};
	$self->{current} = $self->_record;
	return $self->{current};
}

=item C<state()>

Returns two integers representing the number of items in the back stack
and the number of items in the forward stack.

=cut

sub state {
	my $self = shift;
	my $back = scalar @{$self->{back}};
	my $forw = scalar @{$self->{forw}};
	return $back, $forw;
}

=item C<push(PAGE)>

Add a page to the history stack. This pops all items currently in the
forward queue.

PAGE can either be a L<Zim::Page> object or a history record
created by C<record()>. If it is a page object the creation of the
history record is delayed till the next call to the C<current()> method.

=cut

sub push {
	my ($self, $page) = @_;
	my $realname = (ref($page) eq 'HASH') ? $page->{realname} : $page->realname;
	my $current = $self->_record if $self->{current};
	
	if ($current and $current->{realname} ne $realname) { # no redundant entries
		CORE::push @{$self->{back}}, $current;
		shift @{$self->{back}} if @{$self->{back}} > $self->{max};
		$self->{forw} = [];
	}
	
	$self->{current} = $page;
}

=item C<back(INT)>

Go back one or more steps in the history stack.

=cut

sub back {
	my ($self, $i) = @_;
	return 0 if $i < 1 or $i > @{$self->{back}};
	unshift @{$self->{forw}}, $self->current if $self->{current};
	unshift @{$self->{forw}}, splice @{$self->{back}}, -$i+1 if $i > 1;
	$self->{current} = pop @{$self->{back}};
	return 1;
}

=item C<forw(INT)>

Go forward one or more steps in the history stack.

=cut

sub forw {
	my ($self, $i) = @_;
	return 0 if $i < 1 or $i > @{$self->{forw}};
	CORE::push @{$self->{back}}, $self->current if $self->{current};
	CORE::push @{$self->{back}}, splice @{$self->{forw}}, 0, $i-1 if $i > 1;
	$self->{current} = shift @{$self->{forw}};
	return 1;
}

=item C<record(PAGE)>

Returns the history record for a given page object or undef.

=cut

sub record {
	my ($self, $page) = @_;
	my ($back, $current, $forw) = @{$self}{qw/back current forw/};
	
	my $realname = ref($page) ? $page->realname : $page;
	my ($rec) = grep {$_->{realname} eq $realname} (
		((ref($current) eq 'HASH') ? ($current) : ()),
		@$forw, reverse(@$back) );

	return $rec || undef;
}

sub _record { # creates the history record for the current page
	my $self = shift;
	my ($back, $current, $forw) = @{$self}{qw/back current forw/};
	return $current if ref($current) eq 'HASH';
		
	my $realname = $current->realname;
	my ($rec) = grep {$_->{realname} eq $realname} @$forw, reverse(@$back);
	$rec = {realname => $realname} unless $rec;

	%$rec = ( %$rec,
		name     => $current->name,
		basename => $current->basename,
		$self->{PageView}->get_state($current),
	);

	return $rec;
}

=item C<list()>

Returns a complete list of all records currently in the history.

=cut

sub list {
	my $self = shift;
	return @{$self->{back}}, $self->_record, @{$self->{forw}};
}

1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

=cut

