package Zim::Components::PathBar;

use strict;
use vars '$AUTOLOAD';
use Gtk2;
use Gtk2::Ex::ButtonPathBar; # custom widget

our $VERSION = 0.01;

=head1 NAME

Zim::Components::PathBar - Path bar widgets

=head1 DESCRIPTION

This module contains the widgets to display a path bar.
This bar can either show the browsing history or the namespace of
the current page.

=head1 METHODS

Undefined methods are AUTOLOADED to the Gtk2::Ex::ButtonPathBar object.

=over 4

=cut

sub new {
	my $class = shift;
	my $self = bless {@_}, $class;
	$self->init();
	return $self;
}

sub init { # called by new()
	my $self = shift;
	$self->{path_bar} = Gtk2::Ex::ButtonPathBar->new(spacing => 3);
	$self->{app}->signal_connect(
		'history_changed' => \&on_history_changed, $self);
}

sub AUTOLOAD {
	my $self = shift;
	$AUTOLOAD =~ s/^.*:://;
	return if $AUTOLOAD eq 'DESTROY';
	return $self->{path_bar}->$AUTOLOAD(@_);
}

=item C<widget()>

Returns the root widget. This should be used to add the object to a container widget.
Also use this widget for things like show_all() and hide_all().

=cut

sub widget { return $_[0]->{path_bar} }

=item C<set_type(TYPE)>

Set the type of the pathbar. Type can be 'namespace' or 'trace'.

=cut

sub set_type {
	$_[0]->{type} = $_[1];
	$_[0]->{path_loaded} = '';
}

sub on_history_changed {
	my $self = pop;
	if ($self->{type} eq 'namespace') {
		$self->load_namespace;
	}
	elsif ($self->{type} eq 'trace') {
		$self->load_trace;
	}
	# else hidden, ignoring
}

sub load_namespace {
	my $self = shift;
	my @path = grep {length $_} split /:+/, $self->{app}{page}->name();
	my $path = ':'.join(':', @path);
	
	if ($self->{path_loaded} =~ /^$path/i) {
		$self->{path_bar}->select_item($#path);
		return;
	}
	
	$self->{path_bar}->set_path( sub {
		shift; # button
		$self->{app}->go(':'.join(':', @_));
	}, @path);
	$self->{path_loaded} = $path;
	
	for ($self->{path_bar}->get_items()) {
		my $b_path = ':'.join(':', @{$_->{path_data}}) ;
		$_->signal_connect(enter => sub { $self->{app}->push_status("Go to $b_path", 'link') });
		$_->signal_connect(leave => sub { $self->{app}->pop_status('link')                   });
	}

}

sub load_trace {
	my $self = shift;

	my ($back, $forw) = $self->{app}->History->state;
	if ($self->{path_loaded} and $forw) {
		$self->{path_bar}->select_item($back); # == $#back + 1
		return;
	}

	my $i = 0;
	my @path = map [@$_{qw/basename realname/}, $i++],
		$self->{app}->History->list;

	$self->{path_bar}->set_path( sub {
		my $i = pop;
		my ($back, $forw) = $self->{app}->History->state;
		$i -= $back;
		if    ($i < 0) { $self->{app}->go_back(-$i) }
		elsif ($i > 0) { $self->{app}->go_forw($i)  }
		else { $self->{app}->reload() }
	}, @path);
	$self->{path_bar}->select_item($back); # == $#back + 1
	$self->{path_loaded} = 1;

	for ($self->{path_bar}->get_items()) {
		my $b_path = $_->{path_data}[0];
		$_->signal_connect(enter => sub { $self->{app}->push_status("Go to $b_path", 'link') });
		$_->signal_connect(leave => sub { $self->{app}->pop_status('link')                   });
	}
}

1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

=cut

