package Zim::Components::PageView;

use strict;
use vars '$AUTOLOAD';
use Gtk2;
use Gtk2::Pango;             # pango constants
use Gtk2::Ex::HyperTextView; # custom widget
use POSIX qw(strftime);

our $VERSION = 0.01;

=head1 NAME

Zim::Components::PageView - Page TextView widgets

=head1 DESCRIPTION

This module contains the widgets to display an editable
text buffer containing the current page. It includes a search entry
at the bottom of the TextView, formatting codes for the TextBuffer and 
an undo stack.

=head1 METHODS

Undefined methods are AUTOLOADED to the Gtk2::Ex::HyperTextView object.

=over 4

=cut

my ($k_tab, $k_return, $k_kp_enter, $k_backspace, $k_escape, $k_multiply) =
	@Gtk2::Gdk::Keysyms{qw/Tab Return KP_Enter BackSpace Escape KP_Multiply/};

# It seems to me that 10 * PANGO_SCALE is the normal font size
# TODO make tag styles configable
our %TAGS = (
	B     => [weight => PANGO_WEIGHT_BOLD],
	I     => [style => 'italic'],
	U     => [background => 'yellow'],
	C     => [family => 'monospace', wrap_mode => 'none'],
	head1 => [weight => PANGO_WEIGHT_BOLD, size => 10*1.2**3 * PANGO_SCALE ],
	head2 => [weight => PANGO_WEIGHT_BOLD, size => 10*1.2**2 * PANGO_SCALE ],
	head3 => [weight => PANGO_WEIGHT_BOLD, size => 12 * PANGO_SCALE        ],
	head4 => [weight => PANGO_WEIGHT_BOLD, size => 10 * PANGO_SCALE        ],
	head5 => [style  => 'italic', weight => PANGO_WEIGHT_BOLD, size => 10 * PANGO_SCALE],
);

our %UNDO_STEPS = (
	delete     => 'insert',
	insert     => 'delete',
	apply_tag  => 'remove_tag',
	remove_tag => 'apply_tag'
);

sub new {
	my $class = shift;
	my $self = bless {@_}, $class;
	$self->init();
	return $self;
}

sub init { # called by new()
	my $self = shift;
	$self->{overwrite_mode} = 0;

	my $vbox = Gtk2::VBox->new(0, 3);
	$self->{vbox} = $vbox;
	
	my $scroll_window = Gtk2::ScrolledWindow->new();
	$scroll_window->set_policy('automatic', 'automatic');
	$scroll_window->set_shadow_type('in');
	$vbox->add($scroll_window);
	$self->{scroll_window} = $scroll_window;
	
	# init TextView
	my $htext = Gtk2::Ex::HyperTextView->new();
	$htext->set_left_margin(10);
	$htext->set_right_margin(5);
	$htext->set_editable(0) if $self->{app}{settings}{read_only};
	$htext->set_tabs( Gtk2::Pango::TabArray->new_with_positions(
		# initial_size, pos_in_pixels, ... allign => position
		1, 0, 'left' => 40 * PANGO_SCALE ) );
	$htext->{link_properties} = [foreground => 'blue']; # TextTag properties
	$htext->signal_connect(link_clicked =>
		sub { $self->{app}->go($_[1][1])                          }  );
	$htext->signal_connect(link_enter =>
		sub { $self->{app}->push_status("Go to $_[1][1]", 'link') }  );
	$htext->signal_connect(link_leave =>
		sub { $self->{app}->pop_status('link')                    }  );
	$htext->signal_connect(toggle_overwrite => \&on_toggle_overwrite, $self);
	$htext->signal_connect(populate_popup => \&on_populate_popup);
	$htext->signal_connect(key_press_event => \&on_key_press_event, $self);
	$scroll_window->add($htext);
	$self->{htext} = $htext;

	# init search box
	my $hbox = Gtk2::HBox->new(0, 5);
	$vbox->pack_start($hbox, 0, 1, 0);
	$hbox->signal_connect(key_press_event => \&on_key_press_event_hbox);
	$self->{hbox} = $hbox;

	$hbox->pack_start( Gtk2::Label->new(' Find: '), 0,1,0);
	
	my $entry = Gtk2::Entry->new();
	$entry->signal_connect_swapped(changed  => \&on_changed_entry, $self);
	$entry->signal_connect_swapped(activate => \&on_activate_entry,  $self);
	$hbox->pack_start($entry, 0, 1, 0);
	$self->{entry} = $entry;

	my $prev_button = $self->{app}->new_button('gtk-go-back', 'Previous');
	$prev_button->signal_connect(clicked =>
		sub { $self->find($entry->get_text, -1) } );
	$hbox->pack_start($prev_button, 0, 1, 0);
	
	my $next_button = $self->{app}->new_button('gtk-go-forward', 'Next');
	$next_button->signal_connect(clicked =>
		sub { $self->find($entry->get_text, 1) } );
	$hbox->pack_start($next_button, 0, 1, 0);

	# add toolbar buttons and key bindings
	my $read_only = $self->{app}{settings}{read_only};
	unless ($read_only) { # Style menu  # FIXME better menu layout
		my $style_menu = Gtk2::Menu->new;
		$style_menu->append( Gtk2::TearoffMenuItem->new() );
		for my $tag (
			[ 'Head 1',   'Ctrl 1' => 'head1'  ],
			[ 'Head 2',   'Ctrl 2' => 'head2'  ],
			[ 'Head 3',   'Ctrl 3' => 'head3'  ],
			[ 'Head 4',   'Ctrl 4' => 'head4'  ],
			[ 'Head 5',   'Ctrl 5' => 'head5'  ],
			[ 'Normal',   'Ctrl 6' => 'Normal' ],
			[ 'Bold',     'Ctrl B' => 'B'      ],
			[ 'Italic',   'Ctrl I' => 'I'      ],
			[ 'Mark',     'Ctrl U' => 'U'      ],
			[ 'Verbatim', 'Ctrl T' => 'C'      ],
		) {
			my $code = sub { $self->apply_tag($$tag[2]) };
		
			my $item_hbox = Gtk2::HBox->new(0,5);
			# FIXME do something with alignments etc.
			$item_hbox->pack_start(Gtk2::Label->new($$tag[0]), 1,1,0);
			$item_hbox->pack_start(Gtk2::Label->new("($$tag[1])"), 0,1,0);
			my $item = Gtk2::MenuItem->new();
			$item->add($item_hbox);
			$item->signal_connect(activate => $code);
			$style_menu->append($item);

			$self->{app}->add_key($$tag[1] => $code);
		}
		my $item = Gtk2::MenuItem->new('Bullet list');
		$item->signal_connect(activate => sub {$self->toggle_bullets});
		$style_menu->append($item);
		
		$style_menu->show_all;

		my $style_button = $self->{app}->add_button(
			'Styles', 'Alt T', 'gtk-select-font', sub {
			$style_menu->popup(undef, undef, undef, undef, 1, 0);
		} );
		$style_button->signal_connect( button_press_event => sub {
			my (undef, $event) = @_;
			return 0 unless $event->button == 1;
			
			my ($x, $y) = $style_button->get_parent_window->get_position;
			my $allocation = $style_button->allocation;
			$x += $allocation->x;
			$y += $allocation->y + $allocation->height;

			$style_menu->popup(
				undef, undef, sub {$x, $y}, undef, $event->button, $event->time);
			return 1;
		});
		$style_menu->attach_to_widget($style_button, 0);
	}

	$self->{app}->add_button(
		'Link', 'Ctrl L', $Zim::LINK_ICON, sub {$self->apply_link});
	$self->{app}->add_button(
		'Edit Link', 'Ctrl E', 'gtk-properties',
		sub {$self->edit_link_dialog} ) unless $read_only;

	$self->{app}->add_key('Alt /', sub {$self->show_search} );

	unless ($read_only) { # Some more keybindings
		$self->{app}->add_key('Ctrl Z', sub {$self->undo});
		$self->{app}->add_key('Ctrl Y', sub {$self->redo});
		my $date_string = $self->{app}{settings}{date_string};
		$self->{app}->add_key('Ctrl D', sub {
			$self->get_buffer->insert_at_cursor(
				strftime($date_string, localtime) )
		} );
	}
}

sub AUTOLOAD {
	my $self = shift;
	$AUTOLOAD =~ s/^.*:://;
	return if $AUTOLOAD eq 'DESTROY';
	return $self->{htext}->$AUTOLOAD(@_);
}
	
=item C<widget()>

Returns the root widget. This should be used to add the object to a container widget.
Also use this widget for things like show_all() and hide_all().

=cut

sub widget { return $_[0]->{vbox} }

sub on_key_press_event_hbox {
	my ($hbox, $event) = @_;
	return 0 unless $event->keyval == $k_escape;
	$hbox->hide;
	return 1;
}

sub on_changed_entry {
	my ($self, $entry) = @_;
	$self->find($entry->get_text);
}

sub on_activate_entry {
	my ($self, $entry) = @_;
	$self->{hbox}->hide;
	$self->find($entry->get_text);
	$self->{htext}->grab_focus;
}

=item C<find(STRING, DIRECTION)>

Finds next occurance of STRING in the buffer, scrolls the buffer
and highlights the string. DIRECTION can be either 1 for forward or -1
for backward. If no direction is given a forward search is done including
the current position.

=cut

sub find {
	my ($self, $string, $direction) = @_;
	my $buffer = $self->{buffer};
	my $iter = $buffer->get_iter_at_mark( $buffer->get_insert );
	$iter->forward_char if $direction == 1;
	
	my ($start, $end);
	unless ($direction == -1) { # forward
		($start, $end) = $iter->forward_search($string, 'visible-only');
		unless (defined $start) { # wrap around
			$iter = $buffer->get_start_iter;
			($start, $end) = $iter->forward_search($string, 'visible-only');
			return unless defined $start;
		}
	}
	else { # backward
		($start, $end) = $iter->backward_search($string, 'visible-only');
		unless (defined $start) { # wrap around
			$iter = $buffer->get_end_iter;
			($start, $end) = $iter->backward_search($string, 'visible-only');
			return unless defined $start;
		}
	}
	
	#print "found $string at offset ".$iter->get_offset."\n";
	$buffer->place_cursor($start);
	$buffer->move_mark($buffer->get_selection_bound, $end);
	$self->{htext}->scroll_mark_onscreen( $buffer->get_insert );
}

=item C<show_search()>

=cut

sub show_search {
	my $self = shift;
	$self->{hbox}->show_all;
	$self->{entry}->grab_focus;
}

=item C<hide_search()>

=cut

sub hide_search {
	my $self = shift;
	$self->{hbox}->hide_all;
	$self->{htext}->grab_focus;
}

sub on_toggle_overwrite {
	my $self = pop;
	$self->{overwrite_mode} = $self->{overwrite_mode} ? 0 : 1;
	$self->{app}->update_status();
}

sub on_populate_popup { # add items to context menu
	my ($htext, $menu) = @_;
	my $buffer = $htext->get_buffer;
	my $link = $htext->get_link_at_pointer();
	return unless defined $link;
	
	my $seperator = Gtk2::MenuItem->new();
	$seperator->show;
	$menu->prepend($seperator);
	
	my $item = Gtk2::MenuItem->new(
		($link =~ s/^mailto:\/*//) ? 'Copy _Email Address' : 'Copy _Link' );
	$item->signal_connect(activate => sub {
		my $clipboard = Gtk2::Clipboard->get(
			Gtk2::Gdk::Atom->new('PRIMARY', 1) );
		$clipboard->set_text($link);
	} );
	$item->show;
	$menu->prepend($item);
}

sub on_key_press_event { # some extra keybindings
	my ($htext, $event, $self) = @_;
	my $val = $event->keyval;

	#if ($self->{app}{settings}{read_only}) {
		#return 0 unless $val == ord('/');
		#$self->{hbox}->show;
		#$self->{entry}->grab_focus;
		#return 1;
	#}
	
	if ($val == $k_return or $val == $k_kp_enter) { # Enter
		my $buffer = $htext->get_buffer;
		my $iter = $buffer->get_iter_at_mark($buffer->get_insert());
		return 1 if defined $htext->click_if_link_at_iter($iter);
		$self->parse_line($iter) or $buffer->insert_at_cursor("\n");
		$htext->scroll_mark_onscreen( $buffer->get_insert );
		return 1;
	}
	elsif ($val == $k_backspace) { # BackSpace
		my $buffer = $htext->get_buffer;
		my ($start, $end) = $buffer->get_selection_bounds;
		if ($end and $end != $start) {
			$self->selection_backspace($start, $end);
			return 1;
		}
		my $iter = $buffer->get_iter_at_mark($buffer->get_insert());
		if ($self->parse_backspace($iter)) {
			$htext->scroll_mark_onscreen( $buffer->get_insert );
			return 1;
		}
	}
	elsif ($val == $k_tab or $val == ord(' ')) { # WhiteSpace
		my $buffer = $htext->get_buffer;
		if ($val == $k_tab) {
			my ($start, $end) = $buffer->get_selection_bounds;
			if ($end and $end != $start) {
				$self->selection_tab($start, $end);
				return 1;
			}
		}
		my $iter = $buffer->get_iter_at_mark($buffer->get_insert());
		my $string = ($val == $k_tab) ? "\t" : ' ';
		if ($self->parse_word($iter, $string)) {
			$htext->scroll_mark_onscreen( $buffer->get_insert );
			return 1;
		}
	}
	elsif ($val == ord('*') or $val == $k_multiply) { # Bullet
		my $buffer = $htext->get_buffer;
		my ($start, $end) = $buffer->get_selection_bounds;
		return 0 if !$end or $end == $start;
		$self->toggle_bullets($start, $end);
		return 1;
	}
		
	#else { printf "key %x pressed\n", $val } # perldoc -m Gtk2::Gdk::Keysyms

	return 0;
}

=item C<get_state()>

Returns a number of properties that need to be saved in the history.

=cut

sub get_state {
	my $self = shift;
	my $buffer = $self->{buffer};
	my $cursor = $buffer->get_iter_at_mark($buffer->get_insert)->get_offset;
	return cursor => $cursor, undo => $self->{undo}, redo => $self->{redo};
}

=item C<get_status()>

Returns an info string for the current buffer.

=cut

sub get_status {
	my $self = shift;
	return '' . ( $self->{buffer}->get_modified ? '+' : '' ) .
	            ( $self->{overwrite_mode} ? ' --OVERWRITE--' : '' ) .
	            ( $self->{app}{settings}{read_only} ? ' [readonly]' : '' ) ;
}

=item C<load_page(PAGE)>

Load a new page object into the buffer.

=cut

sub load_page {
	my ($self, $page) = @_;
	
	# create a new TextBuffer
	my $buffer = Gtk2::TextBuffer->new();
	$buffer->create_tag($_ => @{$TAGS{$_}}) for keys %TAGS;
	$self->{buffer} = $buffer;
	$self->{htext}->set_buffer($buffer);
	$self->load_parsetree($page);
	unless ($self->{app}{settings}{read_only}) {
		# connect signals _after_ load_parsetree()
		$buffer->signal_connect(delete_range => \&on_delete_range, $self);
		$buffer->signal_connect_after(insert_text => \&on_insert_text, $self);
		$buffer->signal_connect(modified_changed =>
			sub {$self->{app}->update_status} );
		$buffer->signal_connect(apply_tag => \&on_apply_tag, $self);
		$buffer->signal_connect(remove_tag => \&on_remove_tag, $self);
	}
	$buffer->set_modified(0);

	# look for previous cursor position and undo stack
	my $rec = $self->{app}->History->record($page);
	if (defined $rec) {
		$buffer->place_cursor(
			$buffer->get_iter_at_offset($rec->{cursor}) );

		$self->{undo} = $rec->{undo};
		$self->{redo} = $rec->{redo};
	}
	else {
		$buffer->place_cursor(
			$buffer->get_iter_at_offset(0) )
				unless $page->status eq 'new';
		
		$self->{undo} = [];
		$self->{redo} = [];
	}

	$self->{htext}->scroll_mark_onscreen( $buffer->get_insert );
	$self->{htext}->grab_focus;
}

=item C<load_parsetree()>

This method parses the data from the current page object and
puts it into the text buffer

=cut

sub load_parsetree {
	my ($self, $page) = @_;
	#use Data::Dumper; print Dumper $page;
	my $tree = $page->parse_tree;
	$self->insert_block($_) for @{$tree}[2 .. $#$tree];
}

sub insert_block {
	my ($self, $block) = @_;
	my $buffer = $self->{buffer};
	unless (ref $block) {
		s/^(\s*)[\*\xB7](\s+)/$1\x{2022}$2/mg; # bullets
		# \xB7 is the latin1 "high dot" 
		# \x2022 is the utf8 bullet
		# FIXME lists should be identified by the backend
		$buffer->insert_at_cursor($_);
	}
	else {
		my ($tag, $meta, @node) = @$block;
		my $iter = $buffer->get_iter_at_mark(
				$buffer->get_insert() );
		if ($tag eq 'Para') {
			$self->insert_block($_) for @node; # recurs
		}
		elsif ($tag eq 'L') {
			my ($to, $text) = ($meta->{to}, $node[0]);
			my $link = [($to eq $text), $to];
			$self->{htext}->insert_link_at_iter($iter, $text, $link);
		}
		elsif ($tag eq 'image') { # experimental feature
			my $file = File::Spec->catfile(
				$self->{app}{settings}{data_dir}, $meta->{src});
			$file = Gtk2::Gdk::Pixbuf->new_from_file($file);
			$buffer->insert_pixbuf($iter, $file);
		}
		else {
			if ($tag eq 'Verbatim') {
				$node[0] =~ s/^/\t/mg;
				$tag = 'C';
			}
			$buffer->insert_with_tags_by_name(
				$iter, $node[0], $tag );
		}
	}
}

=item C<modified()>

=item C<modified(BOOLEAN)>

Get or set the modified bit. This bit is set when the buffer
is modified by the user.
It should be reset after succesfully saving the buffer content.

=cut

sub modified {
	return 0 unless defined $_[0]->{buffer};
	$_[0]->{buffer}->set_modified($_[1]) if defined $_[1];
	$_[0]->{buffer}->get_modified;
}

=item C<save_page(PAGE)>

Put the content of the buffer back in a page object.
Returns true if the buffer is not empty.

=cut

sub save_page {
	my ($self, $page) = @_;	
	return $self->save_parsetree($page);
}

=item C<save_parsetree(PAGE)>

This routine gets the text from the buffer and puts it back 
into the current page object.

=cut

sub save_parsetree {
	my ($self, $page) = @_;
	my $buffer = $self->{buffer};
	my ($start, $end) = $buffer->get_bounds;
	my $text = $buffer->get_text($start, $end, 0); # start, end, hidden_chars

	return 0 unless $text =~ /\S/;
	
	$page->clear;
	my $last = length $text;
	for (reverse $self->find_tags) {
		my ($start, $end, $name, $meta) = @$_;
		next unless $start < $last; # just to be sure
		my $rest = substr($text, $end, ($last-$end), '');
		my $block = substr($text, $start, ($end-$start), '');
		#print "$name from $start till $end : $block\n";
		$rest =~ s/\x{2022}/\*/g;
		if ($name eq 'C' and $block =~ /\n/) {
			$name = 'Verbatim';
			$block =~ s/^\t//mg;
		}
		$page->unshift_blocks([$name, $meta, $block], $rest);
		$last = $start;
	}
	$page->unshift_blocks($text) if length $text;
	#use Data::Dumper; print Dumper $page->parse_tree;
	$page->clean;
	
	return 1;
}

sub find_tags { # Method to find all tags in the buffer with their ranges
	my $self = shift;
	my $buffer = $self->{buffer};
	my $table  = $buffer->get_tag_table;
	my (@tags, @tag_objs);
#	print "table size: ", $table->get_size, "\n";
	$table->foreach(sub { push @tag_objs, @_ });
	for my $tag (@tag_objs) { # iterate over named and anonymous tags
		my $iter = $buffer->get_start_iter;
		while ( $iter->begins_tag($tag) or $iter->forward_to_tag_toggle($tag) ) {
			my @tag = ( $iter->get_offset );
			$iter->forward_to_tag_toggle($tag) or last;
			push @tag, $iter->get_offset;
			if ($tag->{is_link}) { push @tag, 'L', {to => $tag->{link_data}[1]} }
			else                 { push @tag, $tag->get_property('name'), {} }
			push @tags, \@tag;
			last if $iter->is_end; # prevent infinite loop
		}
	}
	return sort {$$a[0] <=> $$b[0]} @tags;
}

=item C<parse_backspace(ITER)>

This method is called when the user types a backspace.
It tries to update the formatting of the current line.
 
When TRUE is returned the widget does not recieve the backspace.

=cut

sub parse_backspace {
	my ($self, $iter) = @_;
	my $buffer = $self->{buffer};
	my $lf = $buffer->get_iter_at_line( $iter->get_line );
	my $line = $buffer->get_text($lf, $iter, 0);
	if ($line =~ s/\t([\*\x{2022}]\s)$/$1/) {
		$buffer->delete($lf, $iter);
		$buffer->insert($lf, $line);
		return 1;
	}
	return 0;
}

=item C<parse_line(ITER)>

This method is called when the user is about to insert a linebreak.
It checks the line left of the cursor of any markup that needs 
updating. It also takes care of autoindenting.

When TRUE is returned the widget does not recieve the linebreak.

=cut

sub parse_line {
	my ($self, $iter) = @_;
	my $buffer = $self->{buffer};
	my $lf = $buffer->get_iter_at_line( $iter->get_line );
	my $line = $buffer->get_text($lf, $iter, 0);
	#print ">>$line<<\n";
	if ($line =~ s/^(=+)\s+(\S)/$2/) { # heading
		my $h = length($1); # no (7 - x) monkey bussiness here
		$h = 5 if $h > 5;
		$line =~ s/\s+=+\s*$//;
		$buffer->delete($lf, $iter);
		$buffer->insert_with_tags_by_name($lf, $line, "head$h");
	}
	elsif ($line =~ /^\s*[\*\x{2022}]\s+$/) { # empty bullet
		$buffer->delete($lf, $iter);
	}
	elsif ($line =~ /^(\s*\W+\s+|\s+)/) { # auto indenting
		$buffer->insert($iter, "\n$1");
		$self->{htext}->scroll_mark_onscreen( $buffer->get_insert() );
		return 1;
	}
	return 0;
}

=item C<parse_word(ITER, CHAR)>

This method is called after the user ended typing a word.
It checks the word left of the cursor for any markup that
needs updating.

CHAR can be the key that caused a word to end, returning TRUE
makes it never reaching the widget.

=cut

sub parse_word {
	my ($self, $iter, $char) = @_;
	return unless $char eq ' ' or $char eq "\t";
	my $buffer = $self->{buffer};
	my $lf = $buffer->get_iter_at_line( $iter->get_line );
	my $line = $buffer->get_text($lf, $iter, 0) . $char;
	#print ">>$line<<\n";
	if ($line =~ /^(\s*)[\*\x{2022}](\s+)$/) { # bullet
		# FIXME \s* => \t
		my ($pre, $post) = ($1, $2);
		$pre .= $1 if $post =~ s/(\t)+$//; # switch tabs
		$line = $pre."\x{2022}".$post;
		$buffer->delete($lf, $iter);
		$buffer->insert($lf, $line);
		return 1;
	}
#	elsif ($line =~ /^(\t|  )/) { # pre
#		# FIXME \s* => \t
#		$iter->forward_char unless $iter->is_end; # FIXME somthing at end
#		$buffer->apply_tag_by_name('pre', $lf, $iter);
#	}
#	elsif ($line =~ /^(.*)\b(\w+:\/\/\S+[^\s\,\.\;])\s+$/) {
#		# FIXME get the right iters to highlight link
#	} # no wiki link markup supported here
	
	return 0;
}

=item C<apply_tag(TAG)>

Applies the tag with the name TAG to any selected text.

=cut

sub apply_tag {
	my ($self, $tag) = @_;
	my ($start, $end) = $self->get_selection($tag);
	return unless defined $start;

	# TODO what if selection contains linebreaks ??
	
	my $buffer = $self->{buffer};
	$buffer->remove_all_tags($start, $end);
	$buffer->apply_tag_by_name($tag, $start, $end)
		unless $tag eq 'Normal';
	$buffer->set_modified(1);

	if ($tag =~ /^head/) { # give headers their own line
		$end = $end->ends_line ? undef : $end->get_offset ;
		$buffer->insert($start, "\n") unless $start->starts_line;
		$buffer->insert($buffer->get_iter_at_offset($end+1), "\n")
			unless ! defined $end;
	}
	elsif ($tag eq 'C') {
		my $text = $buffer->get_text($start, $end, 1);
		($start, $end) = $self->selection_tab($start, $end)
			if $text =~ /\n/ and $text =~ /^[^\t]/m;
		$buffer->remove_all_tags($start, $end);
		$buffer->apply_tag_by_name('C', $start, $end);
	}
}

sub get_selection { # selects current word if no selections
	my ($self, $tag) = @_;
	my $buffer = $self->{buffer};
	my ($start, $end) = $buffer->get_selection_bounds;
	if (!$end or $start == $end) {
		# TODO autoselect word ( or line if tag =~ /^head/ );
		return undef;
	}
	return ($start, $end);
}

=item C<apply_link(LINK)>

This method is called by the "Link" button or by the ^L keybinding.
It makes any selected text a link. This link is followed immediatly
if the 'follow_new_link' config option is set.

If LINK is undefined the link target is the same as the selected text.

If no text is selected it calls the "New Link" dialog.

In readonly modus the selected text is regarded as a link and
followed immediatly, but no actual link is made

=cut

sub apply_link {
	my ($self, $link) = @_;
	my ($start, $end) = $self->get_selection('L');
	
	unless (defined $start) {
		$self->{app}{settings}{read_only}
			? $self->{app}->goto_page_dialog
			: $self->edit_link_dialog ;
	}

	my $buffer = $self->{buffer};
	my $text = $buffer->get_text($start, $end, 0);
	$link = $text unless defined $link;
	return undef if $link =~ /\n/;
	
	unless ($self->{app}{settings}{read_only}) {
		my $bit = $link eq $text;
		$buffer->remove_all_tags($start, $end);
		$self->{htext}->apply_link([$bit, $link], $start, $end);
		$buffer->set_modified(1);
	}

	$self->{app}->go($link)
		if $self->{app}{settings}{read_only}
		or $self->{app}{settings}{follow_new_link};
}

=item C<toggle_bullets()>

If selected text is a bullet list this removes the bullets, else it adds
bullets.

=cut

sub toggle_bullets {
	my ($self, $start, $end) = @_;
	($start, $end) = $self->get_selection unless defined $start;
	return unless defined $start;
	
	my $text = $self->{buffer}->get_text($start, $end, 1);
	if ($text =~ /^\s*[\*\x{2022}]\s+/m) { # remove bullets
		$text =~ s/^(\s*)[\*\x{2022}]\s+/$1/mg
	}
	else { # set bullets
		$text =~ s/^(\s*)(?:\-\s+)?(\S)/$1\x{2022} $2/mg;
	}

	$self->replace_selection($text, $start, $end);
}

sub replace_selection {
	my ($self, $text, $start, $end) = @_;
	($start, $end) = $self->get_selection unless defined $start;
	
	my $buffer = $self->{buffer};
	my $_start = $start->get_offset;
	my $_end   = $start->get_offset + length $text;
	
	$buffer->delete($start, $end);
	$buffer->insert($start, $text);

	($start, $end) = map $buffer->get_iter_at_offset($_), ($_start, $_end);
	$buffer->place_cursor($start);
	$buffer->move_mark($buffer->get_selection_bound, $end);

	return $start, $end;
}

=item C<selection_tab()>

Puts a tab before every line of a selection.

=cut

sub selection_tab {
	my ($self, $start, $end) = @_;
	($start, $end) = $self->get_selection unless defined $start;
	return unless defined $start;
	
	my $buffer = $self->{buffer};
	my $text = $buffer->get_text($start, $end, 1);
	$text =~ s/^/\t/mg;

	return $self->replace_selection($text, $start, $end);
}

=item C<selection_backspace()>

Removes a tab for every line of a selection.

=cut

sub selection_backspace {
	my ($self, $start, $end) = @_;
	($start, $end) = $self->get_selection unless defined $start;
	return unless defined $start;
	
	my $text = $self->{buffer}->get_text($start, $end, 1);
	if ($text =~ s/^\t//mg) {
		return $self->replace_selection($text, $start, $end);
	}
	else {
		$self->{buffer}->delete($start, $end);
		return undef;
	}
}

sub on_insert_text { # buffer, iter, string, length, self
#	(length($string) == 1)
#		? push(@undo_chars, $string)
#		: 
	$_[4]->add_undo('insert', $_[1]->get_offset - length($_[2]), $_[2]);
}

sub on_delete_range { # buffer, begin, end, self
	#print "delete range\n";
	my $string = $_[0]->get_text($_[1], $_[2], 0);
	$_[3]->add_undo('delete', $_[1]->get_offset, $string);
}

sub on_apply_tag { pop->_on_change_tag('apply_tag', @_) }

sub on_remove_tag { pop->_on_change_tag('remove_tag', @_) }

sub _on_change_tag {
	my ($self, $action, undef, $tag, $start, $end) = @_;
	my @off = ($start->get_offset, $end->get_offset);
	if ($tag->{is_link}) {
		$self->add_undo($action, @off, 'L', $tag->{link_data});
	}
	else {
		$self->add_undo($action, @off, $tag->get_property('name'));
	}
}

sub add_undo {
	my $self = shift;
	return if $self->{undo_lock}; # prohibit unwanted recursion
#	flush_undo_chars() if @undo_chars;
	my ($action, $offset, @data) = @_;
#	print "do: $action \@$offset: >>@data<<\n";
	push @{$self->{undo}}, [$action, $offset, @data];
	shift @{$self->{undo}} if @{$self->{undo}} > $self->{app}{settings}{undo_max};
	@{$self->{redo}} = ();
}

#sub flush_undo_chars {
#	return unless @undo_chars;
#	add_undo('insert', 
#}

=item C<undo()>

Undo one editing step in the buffer.

=cut

sub undo {
	my $self = shift;
	my ($undo, $redo) = @{$self}{'undo', 'redo'};
	return unless @$undo;
	my $step = pop @$undo;
	unshift @$redo, [@$step]; # force copy;
	$$step[0] = $UNDO_STEPS{$$step[0]};
	$self->_do_step(@$step);
}

=item C<redo()>

Redo one editing step in the buffer.

=cut

sub redo {
	my $self = shift;
	my ($undo, $redo) = @{$self}{'undo', 'redo'};
	return unless @$redo;
	my $step = shift @$redo;
	push @$undo, $step;
	$self->_do_step(@$step);
}

sub _do_step {
	my ($self, $action, $offset, @data) = @_;
	my $buffer = $self->{buffer};
	my $start = $buffer->get_iter_at_offset($offset);
	$self->{undo_lock} = 1;
	if ($action eq 'insert') {
		$buffer->insert($start, $data[0]);
		$buffer->place_cursor(
			$buffer->get_iter_at_offset($offset + length($data[0])));
	}
	elsif ($action eq 'delete') {
		my $end = $buffer->get_iter_at_offset($offset + length($data[0]));
		$buffer->delete($start, $end);
		$buffer->place_cursor($start);
	}
	elsif ($action eq 'apply_tag') {
		my $end = $buffer->get_iter_at_offset( $data[0] );
		$buffer->remove_all_tags($start, $end);
		if ($data[1] eq 'L') {
			$self->{htext}->apply_link($data[2], $start, $end);
		}
		else { $buffer->apply_tag_by_name($data[1], $start, $end) }
	}
	elsif ($action eq 'remove_tag') {
		my $end = $buffer->get_iter_at_offset( $data[0] );
		$buffer->remove_all_tags($start, $end);
	}
	$buffer->set_modified(1);
	$self->{htext}->scroll_mark_onscreen( $buffer->get_insert );
	$self->{undo_lock} = 0;
}

#sub flush_undo_mini {
#	return unless @undo_mini;
#	my (prev,
#	for (@undo_mini) {
#		my ($action, $offset, $char) = @$_;
#		
#}

=item C<edit_link_dialog()>

This dialog allows the user to create a link for which
the link target and the link text differ.

=cut

sub edit_link_dialog {
	my $self = shift;
	my ($start, $end) = $self->get_selection('L');

	my ($text, $link) = ('', '');
	if (defined $start) {
		$link = $self->{htext}->get_link_at_iter($start);
		$link = $$link[1] if $link;
		$text = $self->{buffer}->get_text($start, $end, 0);
		$text = undef if $text =~ /\n/;
	}

	my $title = defined($start) ? 'Edit Link' : 'Insert Link';
	
	($text, $link) = $self->{app}->prompt_link_dialog($text, $link, $title);

	return unless defined $text and ($text =~ /\S/ or $link =~ /\S/);
	
	# both entries default to the other
	$link = $text unless $link =~ /\S/;
	$text = $link unless $text =~ /\S/;

	# use delete + insert instead of apply because the text can be different
	my $buffer = $self->{buffer};
	if (defined $start) {
		$buffer->delete($start, $end);
	}
	else {
		$start = $buffer->get_iter_at_mark( $buffer->get_insert());
	}
	my $bit = $link eq $text;
	$self->{htext}->insert_link_at_iter($start, $text, [$bit, $link]);
	$self->{buffer}->set_modified(1);
}


1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

=cut

