package Zim;

use strict;
use vars qw/$AUTOLOAD %Config/;
use Gtk2;
use Gtk2::Gdk::Keysyms;
use POSIX qw(strftime);
use Zim::History;
use Zim::Components::TreeView;
use Zim::Components::PathBar;
use Zim::Components::PageView;

our $VERSION = 0.06;
our $LONG_VERSION = << "EOT";
zim $VERSION

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

Please report bugs to pardus\@cpan.org
EOT

=head1 NAME

Zim - The application object for zim

=head1 SYNOPSIS

	use Zim;
	use Zim::Repository;
	
	my $zim = Zim->new(\%SETTINGS);
	
	my $repository = Zim::Repository->new($DIR);
	$zim->set_repository($repository);
	
	$zim->main_loop;
	
	exit;

=head1 DESCRIPTION

This is developer documentation, for the user manual try
executing C<zim --doc>. For commandline options see L<zim>(1).

This module provides the application object for the Gtk2 application B<zim>.
The application has been split into several components for which the 
modules can be found in the ZIM::Components:: namespace. This object brings
together these components and manages the settings and data objects.

=head1 EXPORT

None by default.

=head1 METHODS

Undefined methods are AUTOLOADED either to an component or to the main
L<Gtk2::Window> object.

=over 4

=cut

our $LINK_ICON = Gtk2->CHECK_VERSION(2, 6, 0) ? 'gtk-connect' : 'gtk-convert';
	# gtk-connect stock item was introduced in 2.6.0

our %CONFIG = (  # Default config values
	pane_pos	=> 120,
	pane_vis	=> 0,
	pathbar_type	=> 'trace',
	icon_size	=> 'large-toolbar',
	width		=> 500,
	height		=> 350,
	default_root	=> undef,
	default_home	=> 'Home',
	browser		=> undef,
	date_string	=> '%A %d/%m/%Y',
	date_page	=> ':Date:%Y_%m_%d',
	hist_max	=> 20,
	undo_max	=> 50,
	follow_new_link => 1,
	use_xdg_cache	=> 0,
);

=item C<new(SETTINGS)>

Simple constructor.

=cut

sub new {
	my ($class, $settings) = @_;
	
	%$settings = (%CONFIG, %$settings); # set defaults
	my $self = bless {settings => $settings}, $class;
	$self->load_config;

	$self->{_message_timeout} = -1;
	$self->{_save_timeout} = -1;
	
	return $self;
}

sub DESTROY {
	my $self = shift;
	for (qw/_save_timeout _message_timeout/) {
		my $timeout = $self->{$_};
		Glib::Source->remove($timeout) if $timeout >= 0;
	}
}

sub AUTOLOAD {
	my $self = shift;
	$AUTOLOAD =~ s/^.*:://;
	return if $AUTOLOAD eq 'DESTROY';
	#warn join ' ', "Zim::AUTOLOAD called for $AUTOLOAD by: ", caller, "\n";
	return $self->{objects}{$AUTOLOAD} if exists $self->{objects}{$AUTOLOAD};
	return $self->{window}->$AUTOLOAD(@_);
}

=item C<set_repository(REPOSITORY)>

Set the repository object.

=cut

sub set_repository { $_[0]->{repository} = $_[1] }

=item C<gui_init()>

This method initializes all GUI objects that make up the application.

=cut

sub gui_init {
	my $self = shift;

	my $window = Gtk2::Window->new('toplevel');
	$window->set_default_size(@{$self->{settings}}{'width', 'height'});
	$window->signal_connect(delete_event => \&on_delete_event, $self);
	$window->signal_connect(destroy => sub { Gtk2->main_quit });
	$window->set_icon(
		Gtk2::Gdk::Pixbuf->new_from_file($self->{settings}{icon_file}) );
	$window->set_title('Zim');
	$self->{window} = $window;

	my $accels = Gtk2::AccelGroup->new;
	$window->add_accel_group($accels);
	$self->{accels} = $accels;

	my $vbox = Gtk2::VBox->new(0, 3);
	$window->add($vbox);
	$self->{vbox} = $vbox;

	my $toolbar = Gtk2::Toolbar->new();
	$toolbar->set_style('icons');
	$vbox->pack_start($toolbar, 0,1,0);
	$self->{toolbar} = $toolbar;

	my $hpaned = Gtk2::HPaned->new();
	$hpaned->set_position($self->{settings}{pane_pos});
	$vbox->add($hpaned);
	$self->{hpaned} = $hpaned;

	my $r_vbox = Gtk2::VBox->new(0, 3);
	$hpaned->add2($r_vbox);
	$self->{r_vbox} = $r_vbox;

	my $statusbar = Gtk2::Statusbar->new;
	$vbox->pack_start($statusbar, 0,1,0);
	$self->{statusbar} = $statusbar;

	$self->add_button(
		'Toggle Index', 'Alt I', 'gtk-index',
		sub {$self->toggle_pane($self->{settings}{pane_vis} ? 0 : 1)} );
	
	my $tree_view = Zim::Components::TreeView->new(app => $self);
	$tree_view->signal_connect(row_activated =>
		sub { $self->toggle_pane(0) if $self->{_pane_visible} == -1 } );
	$hpaned->add1($tree_view->widget);
	$self->{objects}{TreeView} = $tree_view;

	$toolbar->append_space;
	
	$self->add_button('Home', 'Alt Home', 'gtk-home',
		sub {$self->go(':'.$self->{settings}{home})} );
	my $back_button = $self->add_button(
		'Go Back', 'Alt Left', 'gtk-go-back', sub {$self->go_back(1)} );
	my $forw_button = $self->add_button(
		'Go Forward', 'Alt Right', 'gtk-go-forward', sub {$self->go_forw(1)} );
	$back_button->set_sensitive(0);
	$forw_button->set_sensitive(0);

	$toolbar->append_space;

	my $path_bar = Zim::Components::PathBar->new(app => $self);
	$r_vbox->pack_start($path_bar->widget, 0,1,0);
	$self->{objects}{PathBar} = $path_bar;

	my $page_view = Zim::Components::PageView->new(app => $self);
	$r_vbox->add($page_view->widget);
	$self->{objects}{PageView} = $page_view;

	if ($self->{settings}{use_xdg_cache}) {
		my $root = File::Spec->rel2abs($self->{settings}{root_dir});
		my ($vol, $dirs) = File::Spec->splitpath($root, 'NO_FILE');
		my @dirs = File::Spec->splitdir($dirs);
		$self->{settings}{hist_file} ||= File::Spec->catfile( 
			File::BaseDir->xdg_cache_home, 'zim', join ':', $vol, @dirs );
	}
	else {
		my $root = File::Spec->rel2abs($self->{settings}{root_dir});
		$self->{settings}{hist_file} ||= File::Spec->catfile($root, '.zim.history'); 
	}
	my $history = Zim::History->new(
		$self->{settings}{hist_file}, $page_view, $self->{settings}{hist_max});
	$self->{objects}{History} = $history;

	$toolbar->append_space;

	#my $actions_button = add_button(
	#	'Actions', '', 'gtk-execute', sub {});
	$self->add_button('About', 'F1', 'gtk-dialog-info',
		sub {$self->about_dialog} );


	$self->add_key('Ctrl S', sub {$self->save_page('FORCE')})
		unless $self->{settings}{read_only};

	$self->add_key('Ctrl G', sub {$self->goto_page_dialog});
	$self->add_key('Ctrl R', sub {$self->reload});
	$self->add_key('Ctrl W', sub {$self->quit});
	$self->add_key('Alt Up', sub { 
		my $namespace = $self->{page}->namespace;
		$namespace = $self->{settings}{home} if $namespace eq ':';
		$namespace =~ s/^:?/:/;
		$self->go($namespace);
	} );
	$self->add_key('Alt D', sub {
		my $page = strftime($self->{settings}{date_page}, localtime);
		$self->go($page);
	} );
	$self->add_key('Ctrl  ', sub { # Ctrl-Space
		if ($self->{settings}{pane_vis}) {
			if ($tree_view->has_focus) { $page_view->grab_focus }
			else                       { $tree_view->grab_focus }
		}
		else { $self->toggle_pane($self->{_pane_visible} ? 0 : -1) }
	} );
	$self->add_key('Alt J', sub {$self->toggle_path_bar} );

	$self->signal_connect('history_changed', sub {
		my ($back, $forw) = $history->state;
		$back_button->set_sensitive($back?1:0);
		$forw_button->set_sensitive($forw?1:0);
	} );

	# set save interval
	$self->{_save_timeout} = 
		Glib::Timeout->add(5000, sub {$self->save_page; 1})
		unless $self->{settings}{read_only};
	
	$window->show_all;
	$page_view->hide_search;
}

=item C<widget()>

Returns the root window widget.
Use this widget for things like show_all() and hide_all().

=cut

sub widget { return $_[0]->{window} }

=item C<signal_connect(NAME, CODE, DATA)>

=cut

sub signal_connect {
	my ($self, $signal, $code, @data) = @_;
	$self->{_signals}{$signal} ||= [];
	push @{$self->{_signals}{$signal}}, [$code, @data];
}

=item C<signal_emit(NAME, ARGS)>

=cut

sub signal_emit {
	my ($self, $signal, @args) = @_;
	return unless exists $self->{_signals}{$signal};
	for (grep defined($_), @{$self->{_signals}{$signal}}) {
		my ($code, @data) = @$_;
		eval { $code->($self, @args, @data) };
		warn if $@;
	}
}

=item C<load_config>

Read config file.

=cut

sub load_config {
	my $self = shift;
	my $name = $self->{settings}{read_only} ? 'conf.ro' : 'conf.rw' ;
	my ($file) =
		grep { -f $_ && -r $_ }
		map  { File::Spec->catdir($_, 'zim', $name) }
		( File::BaseDir->xdg_config_home, File::BaseDir->xdg_config_dirs );

	return unless defined $file;
	$self->{settings}{conf_file} = $file;
	
	open CONF, $file or $self->exit_error("Could not read\n$file");
	while (<CONF>) {
		/^(.+?)=(.*)$/ or next;
		$self->{settings}{$1} = $2 if length $2;
	}
	close CONF;
}

=item C<save_config>

Save config file.

=cut

sub save_config {
	my $self = shift;
	$self->{settings}{pane_pos} = $self->{hpaned}->get_position;
	
	my $dir  = File::Spec->catdir(
		File::BaseDir->xdg_config_home, 'zim');
	_mkdir($dir) unless -e $dir;
	my $name = $self->{settings}{read_only} ? 'conf.ro' : 'conf.rw' ;
	my $file = File::Spec->catfile($dir, $name);
	
	open CONF, ">$file" or $self->exit_error("Could not write config\n$file");
	print CONF "# zim version:$Zim::VERSION\n";
	print CONF "$_=$$self{settings}{$_}\n"
		for sort keys %CONFIG; # we do not save all settings
	close CONF;
	#print "saved config to $file\n";

}

sub _mkdir {
	my ($vol, $dirs) = (@_ == 1)
		 ? (File::Spec->splitpath(shift(@_), 'NO_FILE')) : (@_) ;
	my @dirs = File::Spec->splitdir($dirs);
	my $path = File::Spec->catpath($vol, shift @dirs);
	mkdir $path or die "Could not create dir $path\n"
		if length $path and ! -d $path;
	while (@dirs) {
		$path = File::Spec->catdir($path, shift @dirs);
		mkdir $path or die "Could not create dir $path\n"
			unless -d $path;
	}
}


=item C<main_loop()>

This method runs the GUI application.

=cut

sub main_loop {
	my $self = shift;
	
	$self->{message_lock} = 1;
	$self->toggle_pane($self->{settings}{pane_vis});
	$self->toggle_path_bar($self->{settings}{pathbar_type});
	$self->{message_lock} = 0;

	$self->{objects}{PageView}->grab_focus;

	Gtk2->main;
}

=item C<quit()>

Exit the main loop.

=cut

sub quit {
	my $self = shift;
	&on_delete_event($self->{window}, $self);
	Gtk2->main_quit;
}

sub on_delete_event { # Called when the window is deleted
	my $self = pop;
	my $window = shift;
	@{$self->{settings}}{'width', 'height'} = $window->get_size;
	
	$self->save_page;
	$self->save_config;

	unless ($self->{settings}{read_only}) {
		my ($vol, $dirs) = File::Spec->splitpath($self->{settings}{hist_file});
		_mkdir($vol, $dirs);
		$self->{objects}{History}->write;
	}

	return 0; # let window continu with destruction
}

=item C<go(NAME)>

Loads a new page adding it to the history or opens an external
uri in a browser. NAME is considered to be either an url or a page name.
Page names are resolved as relative links first.

=cut

sub go {
	my ($self, $link) = @_;
	return warn "Warning: You tried to folow an empty link.\n"
		unless length $link;

	if ($link =~ /^(\w+:\/\/|mailto:)/) { # link is an url
		my $browser = $self->{settings}{browser} || $ENV{BROWSER};
		return error_dialog(
			"You have no browser configured.".
			"Try setting the \$BROWSER environment variable".
			"or edit the zim config file."
		) unless $browser;
		$browser =~ s/\%s/$link/ or $browser .= ' '.$link;
		unless (fork) { # child process
			exec $browser;
			exit 1; # just to be sure
		}
		return;
	}

	$link = $self->{page}->resolve_link($link);
	
	return $self->error_dialog("Page does not exist:\n$link")
		if $self->{settings}{read_only}
		and ! $self->{repository}->page_exists($link);

	my $hist = $self->{objects}{History};
	$hist->current; # make sure the hist record is created
	$self->load_page($link);
	$hist->push($self->{page});
	$self->signal_emit('history_changed');
}

=item C<go_back(INT)>

Go back one or more steps in the history stack.

=cut

sub go_back {
	my $self = shift;
	my $i = shift || 1;
	my $hist = $self->{objects}{History};
	$hist->back($i) or return;
	$self->load_page($hist->current);
	$self->signal_emit('history_changed');
}

=item C<go_forw(INT)>

Go forward one or more steps in the history stack.

=cut

sub go_forw {
	my $self = shift;
	my $i = shift || 1;
	my $hist = $self->{objects}{History};
	$hist->forw($i) or return;
	$self->load_page($hist->current);
	$self->signal_emit('history_changed');
}

sub reload {
	$_[0]->load_page($_[0]->{objects}{History}->current);
}

=item C<load_page(PAGE)>

Method used by C<go()> and friends to actually load a new page.
Wrapper for C<Zim::Repository->load_page()>.

=cut

sub load_page {
	my ($self, $page) = @_;
	
	$self->save_page;

	$page = $page->{name} if ref($page) eq 'HASH'; # hist record
	if (ref $page) { $self->{page} = $page } # page object
	else {
		$page =~ s/^:+|:+$//g; # FIXME ??
		$self->{page} = $self->{repository}->load_page($page);
	}

	$self->{objects}{PageView}->load_page($self->{page});
	$self->signal_emit('page_loaded', $self->{page}->realname);
	
	$self->{window}->set_title($self->{page}->name.' - Zim');
	$self->update_status();
}

=item C<save_page(FORCE)>

Check if the current page needs to be saved. If FORCE is true the page is saved
whether it was modified or not. Wrapper for C<Zim::Repository->save_page()>.

=cut

sub save_page {
	my ($self, $force) = @_;
	my $modified = $self->{objects}{PageView}->modified;
	return 0 if $self->{app}{settings}{read_only};
	return 1 unless $force or $modified and ! $self->{save_lock};
	
	$self->{save_lock} = 1; # Used to block autosave while "could not save" dialog
	
	my $page = $self->{page};
	my $content = $self->{objects}{PageView}->save_page($page);
	
	eval {
		my $old_status = $page->status;
		if ($content) {
			$self->{repository}->save_page($page);
			$self->signal_emit('page_created', $page->realname)
				if $old_status eq 'new' or $old_status eq 'deleted';
		}
		else {
			$self->{repository}->delete_page($page);
			$self->signal_emit('page_deleted', $page->realname);
		}
	};
	unless ($@) {
		$self->{objects}{PageView}->modified(0);
		$self->signal_emit('page_saved', $page->realname);
		$self->update_status();
	}
	else { # FIXME this dialog could be a lot better
		my $page = $self->prompt_page_dialog($page->name,
			'Save As', 'Save Page As:', $@, 'gtk-dialog-error' );
		if ($page =~ /\S/) {
			# FIXME check page exists
			$page =~ s/^:+|:+$//g;
			$self->{page} = $self->{repository}->load_page($page);
			$self->save_page('FORCE'); # recurs
		}
	}

	$self->{save_lock} = 0;
}

=item C<message(STRING)>

Flash a message in the statusbar for a short time.

=cut

sub message {
	my ($self, $str) = @_;
	return if $self->{message_lock};
	my $timeout = $self->{_message_timeout};
	$self->push_status($str, 'message');
	Glib::Source->remove($timeout) if $timeout >= 0;
	$timeout = Glib::Timeout->add(2500, sub {
		$self->pop_status('message');
		$self->{_message_timeout} = -1;
		return 0; # removes timeout
	} );
	$self->{_message_timeout} = $timeout;
}

=item C<update_status()>

Sets the statusbar to display the current page name and some other
information.

=cut

sub update_status {
	my $self = shift;
	my $stat = ' '.$self->{page}->realname;
	if ($_ = $self->{page}->status()) { $stat .= ' ('.uc($_).')' }
	$stat .= $self->{objects}{PageView}->get_status;
	$self->push_status($stat, 'page');
}

=item C<push_status(STRING, CONTEXT)>

Put STRING in the status bar.

=cut

sub push_status {
	my ($self, $str, $id) = @_;
	my $statusbar = $self->{statusbar};
	$id = $statusbar->get_context_id($id);
	$statusbar->pop($id);
	$statusbar->push($id, $str);
}

=item pop_status(CONTEXT)

Removes a string from the status bar.

=cut

sub pop_status {
	my ($self, $id) = @_;
	my $statusbar = $self->{statusbar};
	$id = $statusbar->get_context_id($id);
	$statusbar->pop($id);
}

=item C<new_button(STOCK, TEXT)>

Creates a button with a stock image but different text.

=cut

sub new_button {
	my ($self, $stock, $text) = @_;
	my $hbox = Gtk2::HBox->new;
	$hbox->pack_start(
		Gtk2::Image->new_from_stock($stock, 'button'), 0,0,0);
	$hbox->pack_start(
		Gtk2::Label->new_with_mnemonic($text), 1,1,0);
	my $button = Gtk2::Button->new();
	$button->add(Gtk2::Alignment->new(0.5, 0.5, 0, 0));
	$button->child->add($hbox);
	return $button;
}

=item C<add_button(TEXT, KEY, STOCK, CODE)>

Add a button to the toolbar.

=cut

sub add_button {
	my ($self, $text, $key, $stock, $code) = @_;
	my $icon = Gtk2::Image->new_from_stock($stock, $self->{settings}{icon_size});
	my $button = $self->{toolbar}->append_item(
		#  text, tooltip_text, tooltip_private_text, icon, callback
		$text, $text.($key?" ($key)":''), 'Foo Bar', $icon, $code);
	if ($key) {
		my $mod = [];
		($mod, $key) = split ' ', $key, 2 if $key =~ / /;
		if    ($mod eq 'Alt')  { $mod = 'mod1-mask'    }
		elsif ($mod eq 'Ctrl') { $mod = 'control-mask' }
		$key = $Gtk2::Gdk::Keysyms{$key} || ord($key);
		$button->add_accelerator(
			'clicked', $self->{accels}, $key, $mod, 'visible');
	}
	return $button;
}

=item C<add_key(KEY, CODE)>

Add a key accelerator.

=cut

sub add_key {
	my ($self, $key, $code) = @_;
	
	my $mod = [];
	if    ($key =~ s/^alt //i)  { $mod = 'mod1-mask'    }
	elsif ($key =~ s/^ctrl //i) { $mod = 'control-mask' }
	$key = $Gtk2::Gdk::Keysyms{$key} || ord($key);

	$self->{accels}->connect($key, $mod, 'visible', $code);
}

=item toggle_pane(BOOLEAN)

Show or hide the side pane with the index tree.
If BOOLEAN is "-1" the pane will only be toggled on untill a page is selected.

=cut

sub toggle_pane {
	my ($self, $pane_visible) = @_;
	$self->{_pane_visible} = $pane_visible;
	my $tree_view = $self->{objects}{TreeView};
	my $hpaned = $self->{hpaned};
	if ($pane_visible) {
		$hpaned->set_position($self->{settings}{pane_pos});
		$tree_view->show;
		$tree_view->load_index();
		$tree_view->grab_focus;
	}
	else {
		$self->{settings}{pane_pos} = $hpaned->get_position();
		$tree_view->hide;
		$self->{objects}{PageView}->grab_focus;
	}
	$self->{settings}{pane_vis} = $pane_visible
		unless $pane_visible == -1;
}

=item C<toggle_path_bar(TYPE)>

Set or cycle the type of the pathbar.

=cut

sub toggle_path_bar {
	my $self = shift;
	my $type = $_[0] ? shift :
		($self->{settings}{pathbar_type} eq 'trace')     ? 'namespace' :
		($self->{settings}{pathbar_type} eq 'namespace') ? 'hidden'    : 'trace';
	$self->{settings}{pathbar_type} = $type;
	
	my $path_bar = $self->{objects}{PathBar};
	if ($type eq 'trace' or $type eq 'namespace') {
		$path_bar->widget->show_all;
		$path_bar->set_type($type);
		$path_bar->on_history_changed();
	}
	elsif ($type eq 'hidden') {
		$path_bar->widget->hide_all;
		$path_bar->clear_items;
	}
	else { warn "unknown pathbar_type: $type\n" }

	$self->message("Pathbar type: $type");
}



# TODO the various dialogs could be out-sourced using Autoloader

=item C<exit_error(ERROR)>

Like C<error_dialog> but exits afterwards.

=cut

sub exit_error {
	my $self = shift;
	$self->error_dialog(@_);
	exit 1;
}

=item C<error_dialog(ERROR)>

This method is used to display errors.

=cut

sub error_dialog {
	my ($self, $text1, $text2) = @_;
	$text2 ||= $@ || $text1;
	my $dialog = Gtk2::MessageDialog->new(
		$self->{window}, 'modal', 'error', 'ok', $text1 );
		# parent, flags, type, buttons, format, ...
	print STDERR "zim: $text2\n";
	$dialog->run;
	$dialog->destroy;
}

=item C<goto_page_dialog()>

Prompts the user for a page name. This page is then opened.

=cut

sub goto_page_dialog {
	my $self = shift;
	my $page = $self->{page}->namespace;
	$page = $self->prompt_page_dialog($page, 'Go to', '_Go to Page:');
	$self->go($page) if $page =~ /\S/;
}

=item C<promt_page_dialog(PAGE, TITLE, PROMPT, TEXT, ICON)>

Method used to prompt the user for a page name. Returns PAGE.

The I<text> and I<icon> arguments are optional.

=cut

sub prompt_page_dialog {
	my ($self, $page, $title, $prompt, $text, $icon) = @_;

	my $dialog = Gtk2::Dialog->new(
		$title, $self->{window},
	       	[qw/modal destroy-with-parent no-separator/],
		'gtk-cancel' => 'cancel',
		'gtk-open'   => 'ok',
	);
	$dialog->set_resizable(0);
	$dialog->set_border_width(5);
	$dialog->set_icon($self->{window}->get_icon);
	$dialog->set_default_response('ok');

	if (defined $text) {
		my $text_hbox = Gtk2::HBox->new(0, 5);
		$text_hbox->set_border_width(5);
		$dialog->vbox->add($text_hbox);
		
		if (defined $icon) {
			my $image = Gtk2::Image->new_from_stock($icon, 'dialog');
			$text_hbox->add($image);
		}
		
		my $text_label = Gtk2::Label->new($text);
		$text_hbox->add($text_label);
	}
	
	my $hbox  = Gtk2::HBox->new(0, 5);
	$hbox->set_border_width(5);
	$dialog->vbox->add($hbox);
	
	my $label = Gtk2::Label->new_with_mnemonic($prompt);
	$hbox->add($label);
	
	my $entry = Gtk2::Entry->new();
	$entry->signal_connect(activate => sub { $dialog->response('ok') });
	$entry->set_text($page);
	$hbox->add($entry);
	
	$dialog->show_all;
	if ($dialog->run eq 'ok') { $page = $entry->get_text }
	else                      { $page = undef            }
	$dialog->destroy;

	return $page;
}

=item C<prompt_link_dialog(TEXT, LINK, TITLE)>

Method used to prompt the user for a link. Returns TEXT and TITLE.

=cut

sub prompt_link_dialog {
	my ($self, $text, $link, $title) = @_;
	
	my $dialog = Gtk2::Dialog->new(
		$title, $self->{window},
	       	[qw/modal destroy-with-parent no-separator/],
		'gtk-cancel'  => 'cancel',
	);
	$dialog->set_resizable(0);
	$dialog->set_border_width(5);
	$dialog->set_icon($self->{window}->get_icon);
	
	my $link_button = $self->new_button($LINK_ICON, '_Link');
	$dialog->add_action_widget($link_button, 'ok');
	# $dialog->set_default_response('ok'); FIXME

	my $table = Gtk2::Table->new(2, 2);
	$table->set_border_width(5);
	$table->set_row_spacings(5);
	$table->set_col_spacings(5);
	$dialog->vbox->add($table);
	
	my $text_label = Gtk2::Label->new('Text:');
	my $link_label = Gtk2::Label->new('Links to:');
	my $text_entry = Gtk2::Entry->new();
	$text_entry->signal_connect(activate => sub { $dialog->response('ok') });
	my $link_entry = Gtk2::Entry->new();
	$link_entry->signal_connect(activate => sub { $dialog->response('ok') });
	$table->attach_defaults($text_label, 0,1, 0,1);
	$table->attach_defaults($text_entry, 1,2, 0,1);
	$table->attach_defaults($link_label, 0,1, 1,2);
	$table->attach_defaults($link_entry, 1,2, 1,2);
	
	$link_entry->set_text($link) if defined $link;
	if (defined $text) {
		$text_entry->set_text($text);
		$link_entry->grab_focus;
	}
	else { $text_entry->grab_focus }
	
	$dialog->show_all;
	if ($dialog->run eq 'ok') {
		$text = $text_entry->get_text;
		$link = $link_entry->get_text;
		$dialog->destroy;
		return $text, $link;
	}
	else {
		$dialog->destroy;
		return undef;
	}
}

=item C<about_dialog()>

This dialog tells you about the version of zim you are using,
the copyright and the current config file. Most importantly 
it has a button to open the manual.

=cut

sub about_dialog {
	my $self = shift;
	
	my $dialog = Gtk2::Dialog->new(
		'About Zim', $self->{window},
	       	[qw/modal destroy-with-parent/],
		'gtk-help' => 'help',
		'gtk-ok' => 'ok',
	);
	$dialog->set_resizable(0);
	$dialog->set_border_width(5);
	$dialog->set_icon($self->{window}->get_icon);
	$dialog->set_default_response('ok');
	
	$dialog->vbox->add(
		Gtk2::Image->new_from_file($self->{settings}{icon_file}) );

	my $text = $LONG_VERSION;
	$text =~ s/^(.*)$/<b>$1<\/b>/m; # set first line bold
	$text .= "\n\nThe current config file is: $self->{settings}{conf_file}\n";
	$text .= "\nThe repository config is: $self->{repository}{conffile}\n"
		if exists $self->{repository}{conffile};
	my $label = Gtk2::Label->new();
	$label->set_markup($text);
	$label->set_justify('center');
	$dialog->vbox->add($label);
	
	$dialog->show_all;
	if ($dialog->run eq 'help') {
		unless (fork) { # child process
			no warnings;
			@{$self->{settings}}{'width', 'height'} = $self->{window}->get_size;
			$self->save_config(); # maybe we are read_only too
			exec $0, '--doc';
			# backup procedure to find perl
			# when $0 is not executable for some reason
			eval 'use Config';
			exit 1 if $@;
			my $perl = $Config{perl5} || 'perl';
			exec $perl, $0, '--doc';
			exit 1; # just to be sure
		}
	}
	$dialog->destroy;
}

# "Search" dialog
#
# This dialog asks for a search query and then generates
# a page with search results.

#sub search_dialog {
#	my $dialog = Gtk2::Dialog->new(
#		# title, parent
#		'Search', $window,
#		# flags
#	       	[qw/modal destroy-with-parent no-separator/],
#		# buttons
#		'gtk-cancel' => 'cancel',
#		'gtk-find'   => 'ok',
#	);
#	$dialog->set_resizable(0);
#	$dialog->set_border_width(5);
#	$dialog->set_icon($window->get_icon);
#	#$dialog->set_default_response('ok');
#
#	my $table = Gtk2::Table->new(2, 2);
#	$table->set_border_width(5);
#	$table->set_row_spacings(5);
#	$table->set_col_spacings(5);
#	$dialog->vbox->add($table);
#	
#	my $label1 = Gtk2::Label->new('Find what:');
#	my $query_entry = Gtk2::Entry->new();
#	$query_entry->signal_connect(activate => sub { $dialog->response('ok') });
#	$table->attach_defaults($label1, 0,1, 0,1);
#	$table->attach_defaults($query_entry, 1,2, 0,1);
#	
#	$dialog->show_all;
#	if ($dialog->run eq 'ok') {
#		print "TODO: search algorithm that doesn't depend on grep(1)\n";
#		my $query = $query_entry->get_text;
#		return unless $query =~ /\S/;
#		my $page = Zim::Page->new($zim, 'Search Results');
#		$page->push_blocks(
#			['head1', 'Search Results'], "\n\n",
#			['B', '-- Search algorithm not yet stable --'],
#			"\n\nSearching for \"$query\"\n\n",
#		);
#		$query =~ s/\s/\\W+/g;
#		open GREP, "grep -ERi \"\\b$query\\b\" \"$ROOT\" |" or return;
#		while (<GREP>) {
#			/^(.+?):(.+)$/ or next;
#			my ($file, $text) = ($1, $2);
#			$file =~ s/^\Q$ROOT\E//;
#			next if $file =~ /^\./;
#			$page->push_blocks(
#				['link', $zim->pagename($file)],
#				"\n", ['pre', $text], "\n\n"
#			);
#		}
#		close GREP;
#		load_page($page);
#	}
#	$dialog->destroy;
#}

1;

__END__

=back

=head1 BUGS

Please mail the author if you find any bugs.

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

L<zim>(1),
L<Zim::Repository>

=cut

